/* An implementation of the Viridian hypercall interface */
#include <xen/sched.h>
#include <public/sched.h>
#include <public/hvm/hvm_op.h>
#include <xen/perfc.h>
#include <xen/hypercall.h>
#include <xen/domain_page.h>
#include <asm/paging.h>
#include <asm/p2m.h>
#include <asm/hvm/support.h>
#include <xen/keyhandler.h>


#define VIRIDIAN_MSR_BASE   0x40000000
#define VIRIDIAN_MSR_LIST \
    VIRIDIAN_MSR(GUEST_OS_ID,       0x00) \
    VIRIDIAN_MSR(HYPERCALL,         0x01) \
    VIRIDIAN_MSR(VP_INDEX,          0x02) \
    VIRIDIAN_MSR(SYSTEM_RESET,      0x03) \
    VIRIDIAN_MSR(VP_RUNTIME,        0x10) \
    VIRIDIAN_MSR(TIME_REF_COUNT,    0x20) \
    VIRIDIAN_MSR(EOI,               0x70) \
    VIRIDIAN_MSR(ICR,               0x71) \
    VIRIDIAN_MSR(TPR,               0x72) \
    VIRIDIAN_MSR(SCONTROL,          0x80) \
    VIRIDIAN_MSR(SVERSION,          0x81) \
    VIRIDIAN_MSR(SIEFP,             0x82) \
    VIRIDIAN_MSR(SIMP,              0x83) \
    VIRIDIAN_MSR(EOM,               0x84) \
    VIRIDIAN_MSR(SINT0,             0x90) \
    VIRIDIAN_MSR(SINT1,             0x91) \
    VIRIDIAN_MSR(SINT2,             0x92) \
    VIRIDIAN_MSR(SINT3,             0x93) \
    VIRIDIAN_MSR(SINT4,             0x94) \
    VIRIDIAN_MSR(SINT5,             0x95) \
    VIRIDIAN_MSR(SINT6,             0x96) \
    VIRIDIAN_MSR(SINT7,             0x97) \
    VIRIDIAN_MSR(SINT8,             0x98) \
    VIRIDIAN_MSR(SINT9,             0x99) \
    VIRIDIAN_MSR(SINT10,            0x9a) \
    VIRIDIAN_MSR(SINT11,            0x9b) \
    VIRIDIAN_MSR(SINT12,            0x9c) \
    VIRIDIAN_MSR(SINT13,            0x9d) \
    VIRIDIAN_MSR(SINT14,            0x9e) \
    VIRIDIAN_MSR(SINT15,            0x9f) \
    VIRIDIAN_MSR(STIMER0_CONFIG,    0xb0) \
    VIRIDIAN_MSR(STIMER0_COUNT,     0xb1) \
    VIRIDIAN_MSR(STIMER1_CONFIG,    0xb2) \
    VIRIDIAN_MSR(STIMER1_COUNT,     0xb3) \
    VIRIDIAN_MSR(STIMER2_CONFIG,    0xb4) \
    VIRIDIAN_MSR(STIMER2_COUNT,     0xb5) \
    VIRIDIAN_MSR(STIMER3_CONFIG,    0xb6) \
    VIRIDIAN_MSR(STIMER3_COUNT,     0xb7)

/*
 * Create an enumaration of the viridian MSR numbers.
 */

#define VIRIDIAN_MSR(name,offset) \
    VIRIDIAN_MSR_ ## name = (VIRIDIAN_MSR_BASE + offset),
    
typedef enum
{
    VIRIDIAN_MSR_LIST
    VIRIDIAN_MSR_LIST_MAX
} VIRIDIAN_MSR_INDEX;

#undef VIRIDIAN_MSR
#undef VIRIDIAN_MSR_LIST

/*
 * Viridian Hypercall Status Codes
 */

#define HV_STATUS_SUCCESS                       0x0000
#define HV_STATUS_INVALID_HYPERCALL_CODE        0x0002
#define HV_STATUS_INVALID_HYPERCALL_INPUT       0x0003
#define HV_STATUS_INVALID_ALIGNMENT             0x0004
#define HV_STATUS_INVALID_PARAMETER             0x0005
#define HV_STATUS_ACCESS_DENIED                 0x0006
#define HV_STATUS_INVALID_PARTITION_STATE       0x0007
#define HV_STATUS_OPERATION_DENIED              0x0008
#define HV_STATUS_UNKNOWN_PROPERTY              0x0009
#define HV_STATUS_PROPERTY_OUT_OF_RANGE         0x000a
#define HV_STATUS_INSUFFICIENT_MEMORY           0x000b
#define HV_STATUS_PARTITION_TOO_DEEP            0x000c
#define HV_STATUS_INVALID_PARTITION_ID          0x000d
#define HV_STATUS_INVALID_VP_INDEX              0x000e
#define HV_STATUS_INVALID_PORT_ID               0x0011
#define HV_STATUS_INVALID_CONNECTION_ID         0x0012
#define HV_STATUS_INSUFFICIENT_BUFFERS          0x0013
#define HV_STATUS_NOT_ACKNOWLEDGED              0x0014
#define HV_STATUS_ACKNOWLEDGED                  0x0016
#define HV_STATUS_INVALID_SAVE_RESTORE_STATE    0x0017
#define HV_STATUS_INVALID_SYNIC_STATE           0x0018
#define HV_STATUS_OBJECT_IN_USE                 0x0019
#define HV_STATUS_INVALID_PROXIMITY_DOMAIN_INFO 0x001a
#define HV_STATUS_NO_DATA                       0x001b
#define HV_STATUS_INACTIVE                      0x001c
#define HV_STATUS_NO_RESOURCES                  0x001d
#define HV_STATUS_FEATURE_UNAVAILABLE           0x001e

/*
 * Viridian Hypercall Codes and Parameters
 */

#define HvSwitchVirtualAddressSpace             1

typedef struct
{
    uint64_t    AddressSpace;
} HvSwitchVirtualAddressSpaceInput;

#define HvFlushVirtualAddressSpace              2

typedef struct
{
    uint64_t    AddressSpace;
    uint64_t    Flags;
    uint64_t    ProcessorMask;
} HvFlushVirtualAddressSpaceInput;

#define HvFlushVirtualAddressList               3

typedef union
{
    uint64_t    raw;
    struct
    {
        uint64_t    Range:12;
        uint64_t    GuestVirtualPage:48;
    }           fields;
} GvaRange;

typedef struct
{
    uint64_t    AddressSpace;
    uint64_t    Flags;
    uint64_t    ProcessorMask;
    GvaRange    GvaRangeList[1];
} HvFlushVirtualAddressListInput;

#define HvNotifyLongSpinWait                    8

typedef struct
{
    uint64_t    Iterations; // or something, not really sure yet.
} HvNotifyLongSpinWaitInput;

/*
 * Virtual Address Flush Flags
 *
 * Note: FLUSH_ALL_PROCESSORS set means all processors in the guest,
 *       clear means all processors in the current process (address
 *       space).
 */

#define HV_FLUSH_ALL_PROCESSORS                 0x00000001
#define HV_FLUSH_ALL_VIRTUAL_ADDRESS_SPACES     0x00000002
#define HV_FLUSH_NON_GLOBAL_MAPPINGS_ONLY       0x00000004

/*
 * Viridian CPUID 4000003, Viridian MSR availability.
 */

#define CPUID3A_MSR_APIC_ACCESS (1 << 4)
#define CPUID3A_MSR_HYPERCALL   (1 << 5)
#define CPUID3A_MSR_VP_INDEX    (1 << 6)

/*
 * Viridian CPUID 4000004, Implementation Recommendations.
 */

#define CPUID4A_VMCALL_CR3_SWITCH               0x00000001
#define CPUID4A_VMCALL_LOCAL_TLB_FLUSH          0x00000002
#define CPUID4A_VMCALL_REMOTE_TLB_FLUSH         0x00000004
#define CPUID4A_MSR_BASED_APIC                  0x00000008
#define CPUID4A_MSR_INITIATED_SYSTEM_RESET      0x00000010
#define CPUID4A_RELAX_TIMER_INT_HANDLING        0x00000020

#define VIRIDIAN_HYPERCALL_TRAMPOLINE                                  \
"\x0d\x00\x00\x00\x80" /* orl $0x80000000, %eax */                     \
"\x0f\x01\xc1" /* vmcall */                                            \
"\xc3" /* ret */

#if defined(VIRIDIAN_STATS_VERBOSE)
#define NOTICE_K(count, string) \
        {                                       \
            static int seen_count = 0;          \
            if ((seen_count & (count - 1)) == 0)\
            {                                   \
                if (seen_count != 0)            \
                {                               \
                    printk("(%d more) ", count);\
                    seen_count = 0;             \
                }                               \
                printk string;                  \
            }                                   \
            seen_count++;                       \
        }
#else
#define NOTICE_K(count, string)
#endif

#define VIRIDIAN_BITS_MASTER                0x10000000
#define VIRIDIAN_BITS_CR3_SW                CPUID4A_VMCALL_CR3_SWITCH          
#define VIRIDIAN_BITS_LOCAL_TLB             CPUID4A_VMCALL_LOCAL_TLB_FLUSH      
#define VIRIDIAN_BITS_REMOTE_TLB            CPUID4A_VMCALL_REMOTE_TLB_FLUSH  
#define VIRIDIAN_BITS_MSR_APIC              CPUID4A_MSR_BASED_APIC          
#define VIRIDIAN_BITS_MSR_RESET             CPUID4A_MSR_INITIATED_SYSTEM_RESET      
#define VIRIDIAN_BITS_RELAX_TIMER_INT_HAND  CPUID4A_RELAX_TIMER_INT_HANDLING

#define VIRIDIAN_BITS_VALID (VIRIDIAN_BITS_MASTER |\
                             VIRIDIAN_BITS_CR3_SW |\
                             VIRIDIAN_BITS_LOCAL_TLB |\
                             VIRIDIAN_BITS_REMOTE_TLB |\
                             VIRIDIAN_BITS_MSR_APIC |\
                             VIRIDIAN_BITS_RELAX_TIMER_INT_HAND |\
                             0)

int viridian_long_spin_count = 2047;

int viridian_recommendations = VIRIDIAN_BITS_MSR_APIC |
                               VIRIDIAN_BITS_CR3_SW |
                               VIRIDIAN_BITS_RELAX_TIMER_INT_HAND;
int viridian_debug_selector = 0;

int viridian_debug_viridian_enabled(void)
{
    return (viridian_recommendations & VIRIDIAN_BITS_MASTER) != 0;
}

char * viridian_debug_feature_text(int selector)
{
    switch (selector)
    {
    case VIRIDIAN_BITS_MASTER:
        return "master enable/disable";
    case VIRIDIAN_BITS_CR3_SW:
        return "hypercall move to cr3";
    case VIRIDIAN_BITS_LOCAL_TLB:
        return "local tlb flush";
    case VIRIDIAN_BITS_REMOTE_TLB:
        return "remote tlb flush";
    case VIRIDIAN_BITS_MSR_APIC:
        return "msr apic access";
    case VIRIDIAN_BITS_RELAX_TIMER_INT_HAND:
        return "relaxed external timer handling";
    }
    printk("viridian feature %d accessed\n", selector);
    return "THIS CAN'T POSSIBLY HAPPEN!!!!";
}

void viridian_debug_key_shift(unsigned char key)
{
    viridian_debug_selector <<= 1;
    if ((viridian_debug_selector & ((VIRIDIAN_BITS_MASTER * 2) - 1)) == 0)
        viridian_debug_selector = 1;

    while ((viridian_debug_selector != 0) &&
          ((viridian_debug_selector & VIRIDIAN_BITS_VALID) == 0))
        viridian_debug_selector <<= 1;

    printk("select Viridian %s, current state is %s\n",
           viridian_debug_feature_text(viridian_debug_selector),
           (viridian_recommendations & viridian_debug_selector) ?
           "ON" : "OFF");
}

void viridian_debug_key_commit(unsigned char key)
{
    viridian_recommendations ^= viridian_debug_selector;
    printk("flip Viridian %s, new state is %s\n",
            viridian_debug_feature_text(viridian_debug_selector),
            (viridian_recommendations & viridian_debug_selector) ?
           "ON" : "OFF");
}

void viridian_debug_spin_count(unsigned char key)
{
    /* Really only positive numbers are allowed but we allow it to roll thru 0
       as a means of disabling this function all together. */
    viridian_long_spin_count <<= 1;
    if (viridian_long_spin_count == 0)
        viridian_long_spin_count = 1;
    printk("viridian long spin count set to 0x%x\n", viridian_long_spin_count);
}

static __init int viridian_register_keyhandlers(void)
{
    register_keyhandler(
        '<', viridian_debug_key_shift, "viridian feature select shift");
    register_keyhandler(
        '~', viridian_debug_key_commit, "viridian toggle selected feature");
    register_keyhandler(
        '>', viridian_debug_spin_count, "viridian adjust spin count");
    return 0;
}

__initcall(viridian_register_keyhandlers);

#define vir_get_eip()   guest_cpu_user_regs()->eip

int cpuid_viridian_leaves(unsigned int leaf, unsigned int *eax,
                          unsigned int *ebx, unsigned int *ecx,
                          unsigned int *edx)
{
    struct domain *d = current->domain;

    if ( !is_viridian_domain(d) )
        return 0;

    leaf -= 0x40000000;
    if ( leaf > 5 )
        return 0;

    *eax = *ebx = *ecx = *edx = 0;
    switch ( leaf )
    {
    case 0:
        *eax = 0x40000005; /* Maximum leaf */
        *ebx = 0x7263694d; /* Magic numbers  */
        *ecx = 0x666F736F;
        *edx = 0x76482074;
        break;
    case 1:
        *eax = 0x31237648; /* Version number */
        break;
    case 2:
        /* Hypervisor information, but only if the guest has set its
           own version number. */
        if ( d->viridian.guest_os_id.raw != 0 ) {
            *eax = 1; /* Build number */
            *ebx = (3 << 16) | 2; /* major and minor */
            *ecx = 0; /* SP */
            *edx = 0; /* Service branch and number */
        }
        break;
    case 3:
        /* What hypervisor MSRs are available to the guest */
        *eax =
            CPUID3A_MSR_APIC_ACCESS             |
            CPUID3A_MSR_HYPERCALL               |
            CPUID3A_MSR_VP_INDEX                |
            0;
        break;
    case 4:
        /* Recommended hypercall usage. */
        if ((d->viridian.guest_os_id.raw != 0)
         && (d->viridian.guest_os_id.fields.os >= 4))
        {
            *eax = viridian_recommendations & ~VIRIDIAN_BITS_MASTER;
#if !defined(NO_VIRIDIAN_DEBUG)
            {
                int i;
                if (*eax)
                    gdprintk(XENLOG_INFO, "Viridian features enabled\n");
                for (i = 1; i <= *eax; i <<= 1)
                    if (*eax & i)
                        gdprintk(XENLOG_INFO, "  %s\n",
                                 viridian_debug_feature_text(i));
            }
#endif
            *ebx = viridian_long_spin_count;
        }
        else
            *eax = 0;
        break;
    case 5:
        /* Implementation limits.  0 for unspecified. */
        break;
    }

    return 1;
}

static void enable_hypercall_page(void)
{
    struct domain *d = current->domain;
    unsigned long gmfn = d->viridian.hypercall_gpa.fields.pfn;
    unsigned long mfn = gmfn_to_mfn(d, gmfn);
    void *map;

    if ( !mfn_valid(mfn) ||
         !get_page_and_type(mfn_to_page(mfn), d, PGT_writable_page) ) {
        gdprintk(XENLOG_WARNING, "Bad GMFN %lx (MFN %lx)\n", gmfn, mfn);
        return;
    }
    
    map = map_domain_page(mfn);
    /* The Xen hypercall page consists of an array of hypercall 
       sequences indexed by hypercall number.  The Viridian
       hypercall page consists of a single vmcall instruction
       (vmx) or vmmcall instruction (svm) followed by a ret.  In
       the vmx case it is preceeded by a 5 byte move immediate of
       the hypercall number into eax, in viridian we preceed it
       by a 5 byte or immediate to set the top bit of eax which
       is a reserved field in the viridian hypercall calling 
       convention. This bit is used to differentiate Xen and
       Viridian hypercalls.  To get the correct code sequence
       we use Xen's hypercall page initialization then modify
       the first instruction and fill everything past the ret
       with breakpoint instructions. */
    hvm_hypercall_page_initialise(d, map);
    memcpy(map, VIRIDIAN_HYPERCALL_TRAMPOLINE, 5);
    memset((char *)map + sizeof(VIRIDIAN_HYPERCALL_TRAMPOLINE),
           0xCC, PAGE_SIZE - sizeof(VIRIDIAN_HYPERCALL_TRAMPOLINE));
    
    unmap_domain_page(map);
    put_page_and_type(mfn_to_page(mfn));
    return;
}

int wrmsr_viridian_regs(uint32_t idx, uint32_t eax, uint32_t edx)
{
    struct domain *d = current->domain;
    uint64_t val = ((uint64_t)edx << 32) | eax;

    if ( !is_viridian_domain(d) )
        return 0;

    switch (idx) {
    case VIRIDIAN_MSR_GUEST_OS_ID:
        perfc_incr(mshv_wrmsr_osid);
        d->viridian.guest_os_id.raw = val;
        gdprintk(XENLOG_INFO, "Guest os:\n");
        gdprintk(XENLOG_INFO, "\tvendor: %x\n",
               d->viridian.guest_os_id.fields.vendor);
        gdprintk(XENLOG_INFO, "\tos: %x\n",
               d->viridian.guest_os_id.fields.os);
        gdprintk(XENLOG_INFO, "\tmajor: %x\n",
               d->viridian.guest_os_id.fields.major);
        gdprintk(XENLOG_INFO, "\tminor: %x\n",
               d->viridian.guest_os_id.fields.minor);
        gdprintk(XENLOG_INFO, "\tsp: %x\n",
               d->viridian.guest_os_id.fields.service_pack);
        gdprintk(XENLOG_INFO, "\tbuild: %x\n",
               d->viridian.guest_os_id.fields.build_number);
        return 1;

    case VIRIDIAN_MSR_HYPERCALL:
        perfc_incr(mshv_wrmsr_hc_page);
        gdprintk(XENLOG_INFO, "Set hypercall page %"PRIx64".\n", val);
        if ( d->viridian.guest_os_id.raw == 0 )
            return 1;
        d->viridian.hypercall_gpa.raw = val;
        if ( d->viridian.hypercall_gpa.fields.enabled )
            enable_hypercall_page();
        return 1;

    case VIRIDIAN_MSR_VP_INDEX:
        perfc_incr(mshv_wrmsr_vp_index);
        gdprintk(XENLOG_INFO, "Set VP index %"PRIu64".\n", val);
        return 0;

    case VIRIDIAN_MSR_EOI:
        perfc_incr(mshv_wrmsr_eoi);
        NOTICE_K(0x10000, ("Viridian set EOI seen\n"));
        vlapic_EOI_set(vcpu_vlapic(current));
        return 1;

    case VIRIDIAN_MSR_ICR:
        perfc_incr(mshv_wrmsr_icr);
        NOTICE_K(0x4000, ("Viridian set ICR (send IPI) seen\n"));
        {
            struct vlapic *vlapic = vcpu_vlapic(current);
            eax &= ~(1 << 12);
            edx &= 0xff000000;
            vlapic_set_reg(vlapic, APIC_ICR2, edx);
            if ( vlapic_ipi(vlapic, eax, edx) == X86EMUL_OKAY )
                vlapic_set_reg(vlapic, APIC_ICR, eax);
        }
        return 1;

    case VIRIDIAN_MSR_TPR:
        perfc_incr(mshv_wrmsr_tpr);
        NOTICE_K(0x10000, ("Viridian set TPR seen\n"));
        vlapic_set_reg(vcpu_vlapic(current), APIC_TASKPRI, eax & 0xff);
        return 1;

    default:
        return 0;
    }
}

int rdmsr_viridian_regs(uint32_t idx, uint32_t *eax, uint32_t *edx)
{
    uint64_t val;
    struct vcpu *v = current;
    
    if ( !is_viridian_domain(v->domain) )
        return 0;

    switch (idx) {
    case VIRIDIAN_MSR_GUEST_OS_ID:
        perfc_incr(mshv_rdmsr_osid);
        val = v->domain->viridian.guest_os_id.raw;
        break;

    case VIRIDIAN_MSR_HYPERCALL:
        perfc_incr(mshv_rdmsr_hc_page);
        val = v->domain->viridian.hypercall_gpa.raw;
        break;

    case VIRIDIAN_MSR_VP_INDEX:
        perfc_incr(mshv_rdmsr_vp_index);
        val = v->vcpu_id;
        break;

    case VIRIDIAN_MSR_ICR:
        perfc_incr(mshv_rdmsr_icr);
        gdprintk(XENLOG_WARNING,
                 "viridian get ICR seen (*** unexpected ***)\n");
        {
            struct vlapic *vlapic = vcpu_vlapic(v);
            *edx = vlapic_get_reg(vlapic, APIC_ICR2);
            *eax = vlapic_get_reg(vlapic, APIC_ICR);
        }
        return 1;

    case VIRIDIAN_MSR_TPR:
        perfc_incr(mshv_rdmsr_tpr);
        NOTICE_K(0x10000, ("Viridian get TPR seen\n"));
        *eax = vlapic_get_reg(vcpu_vlapic(v), APIC_TASKPRI);
        *edx = 0;
        return 1;

    default:
        return 0;
    }
    *eax = val;
    *edx = val >> 32;
    return 1;
}

typedef union {
    uint64_t raw;
    struct {
        uint16_t call_code;
        uint16_t flag_fast:1;
        uint16_t rsvd1:15;
        unsigned rep_count:12;
        unsigned rsvd2:4;
        unsigned rep_start:12;
        unsigned rsvd3:4;
    };
} hypercall_input_t;

typedef union {
    uint64_t raw;
    struct {
        uint16_t result;
        uint16_t rsvd1;
        unsigned rep_complete:12;
        unsigned rsvd2:20;
    };
} hypercall_output_t;

int viridian_hypercall(struct cpu_user_regs *regs)
{
    hypercall_input_t input;
    hypercall_output_t output = {0};
    int mode = hvm_guest_x86_mode(current);
    unsigned long input_params_gpa, output_params_gpa;
    uint16_t status = HV_STATUS_SUCCESS;
    long rc;
    XEN_GUEST_HANDLE(void) null_handle;

    switch (mode)
    {
#ifdef __x86_64__
    case 8:
        input.raw = regs->rcx;
        input_params_gpa = regs->rdx;
        output_params_gpa = regs->r8;
        break;
#endif
    case 4:
        input.raw = ((uint64_t)regs->edx << 32) | regs->eax;
        input_params_gpa = ((uint64_t)regs->ebx << 32) | regs->ecx;
        output_params_gpa = ((uint64_t)regs->edi << 32) | regs->esi;
        break;
    default:
        goto out;
    }


    switch (input.call_code) {
    case HvSwitchVirtualAddressSpace:
        perfc_incr(mshv_call_sw_addr_space);

        /*
         * If the fast flag is not set, the input parameter points to the
         * guest physical address where the input data can be found.  For
         * simple inputs which can be contained wholly in a register the
         * fast flag indicates that's how it is.  This was not the case for
         * Vista SP0.
         */
        if ( !input.flag_fast )
        {
            if ( input_params_gpa & ((sizeof input_params_gpa) - 1) )
            {
                status = HV_STATUS_INVALID_ALIGNMENT;
                break;
            }
            if ( hvm_copy_from_guest_phys(&input_params_gpa,
                                          input_params_gpa,
                                          sizeof input_params_gpa) )
            {
                status = HV_STATUS_INVALID_PARAMETER;
                break;
            }
        }
        NOTICE_K(0x10000, ("HvSwitchAddressSpace(%lx)\n", input_params_gpa));
        hvm_set_cr3(input_params_gpa);
        status = HV_STATUS_SUCCESS;
        break;
    case HvFlushVirtualAddressList:
        perfc_incr(mshv_call_flush_tlb_list);
        NOTICE_K(0x10000, ("HvFlushTb list seen\n"));
        output.rep_complete = input.rep_count;
        null_handle.p = NULL;
        rc = do_hvm_op(HVMOP_flush_tlbs, null_handle);
        if (rc == -EAGAIN) // PLJ test this
            return HVM_HCALL_preempted;
        status = HV_STATUS_SUCCESS;
        break;
    case HvFlushVirtualAddressSpace:
        perfc_incr(mshv_call_flush_tlb_all);
        NOTICE_K(0x10000, ("HvFlushTb entire seen\n"));
        null_handle.p = NULL;
        rc = do_hvm_op(HVMOP_flush_tlbs, null_handle);
        if (rc == -EAGAIN) // PLJ test this
            return HVM_HCALL_preempted;
        status = HV_STATUS_SUCCESS;
        break;
    case HvNotifyLongSpinWait:
        perfc_incr(mshv_call_long_wait);
        NOTICE_K(0x10000, ("HvNotifyLongSpinWait seen\n"));
        do_sched_op_compat(SCHEDOP_yield, 0);
        status = HV_STATUS_SUCCESS;
        break;
    default:
        printk("Viridian hypercall %x unsupported\n", input.call_code);
        status = HV_STATUS_INVALID_HYPERCALL_CODE;
        break;
    }
out:
    output.result = status;
    switch (mode) {
#ifdef __x86_64__
    case 8:
        regs->rax = output.raw;
        break;
#endif
    default:
        regs->edx = output.raw >> 32;
        regs->eax = output.raw;
        break;
    }
    return HVM_HCALL_completed;
}

#ifndef __KXEN__
static int viridian_save_cpu_ctxt(struct domain *d, hvm_domain_context_t *h)
{
    struct hvm_viridian_context ctxt;

    ctxt.hypercall_gpa = d->viridian.hypercall_gpa.raw;
    ctxt.guest_os_id   = d->viridian.guest_os_id.raw;
    return (hvm_save_entry(VIRIDIAN, 0, h, &ctxt) != 0);
}

static int viridian_load_cpu_ctxt(struct domain *d, hvm_domain_context_t *h)
{
    struct hvm_viridian_context ctxt;

    if (hvm_load_entry(VIRIDIAN, h, &ctxt) != 0)
        return -EINVAL;

    d->viridian.hypercall_gpa.raw = ctxt.hypercall_gpa;
    d->viridian.guest_os_id.raw   = ctxt.guest_os_id;
    return 0;
}

HVM_REGISTER_SAVE_RESTORE(VIRIDIAN, viridian_save_cpu_ctxt,
                          viridian_load_cpu_ctxt, 1, HVMSR_PER_DOM);
#endif

