/*
 * entry.S: VMX architecture-specific entry/exit handling.
 * Copyright (c) 2004, Intel Corporation.
 * Copyright (c) 2008, Citrix Systems, Inc.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU General Public License,
 * version 2, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston, MA 02111-1307 USA.
 */

#include <xen/config.h>
#include <xen/errno.h>
#include <xen/softirq.h>
#include <asm/types.h>
#include <asm/asm_defns.h>
#include <asm/apicdef.h>
#include <asm/page.h>
#include <public/xen.h>

#define VMRESUME     .byte 0x0f,0x01,0xc3
#define VMLAUNCH     .byte 0x0f,0x01,0xc2
#define VMREAD(off)  .byte 0x0f,0x78,0x47,((off)-UREGS_rip)
#define VMWRITE(off) .byte 0x0f,0x79,0x47,((off)-UREGS_rip)

/* VMCS field encodings */
#define GUEST_RSP    0x681c
#define GUEST_RIP    0x681e
#define GUEST_RFLAGS 0x6820
#define HOST_RSP     0x6c14

#ifndef __KXEN__
#define get_current(reg)                        \
        mov $STACK_SIZE-BYTES_PER_LONG, r(reg); \
        or  r(sp), r(reg);                      \
        and $~(BYTES_PER_LONG-1),r(reg);        \
        mov (r(reg)),r(reg);
#else
#define get_current(reg)                        \
        mov CURRENT_vcpu_stack_offset(r(sp)), r(reg)

#define get_user_regs(reg)                      \
        lea VCPU_user_regs(r(bx)),r(reg)
#endif

#if defined(__x86_64__)
#define r(reg) %r##reg
#define UREGS_r(_r) UREGS_r ## _r
#define SIZEOF_REG 8
#define CURRENT_vcpu_stack_offset (4*SIZEOF_REG)
#define PARAM_POP(n) lea (8*n)(r(sp)), r(sp)
#define addr_of(lbl) lbl(%rip)
#ifndef __KXEN__
#define call_with_regs(fn)                      \
        mov  %rsp,%rdi;                         \
        call fn;
#else
#define call_with_regs(fn)                      \
        get_user_regs(di);                      \
        call fn;
#endif
#else /* defined(__i386__) */
#define r(reg) %e##reg
#define UREGS_r(_r) UREGS_e ## _r
#define SIZEOF_REG 4
#ifdef __KXEN_MACOS__
#define SIZEOF_DT 8
#define EXTRA_stack_space (3*SIZEOF_REG+2*SIZEOF_REG+2*SIZEOF_DT)
#else
#define EXTRA_stack_space 0
#endif
#define CURRENT_vcpu_stack_offset (5*SIZEOF_REG+EXTRA_stack_space)
#define PARAM_POP(n) lea (4*n)(r(sp)), r(sp)
#define addr_of(lbl) lbl
#define UREGS_rip UREGS_eip
#define UREGS_rsp UREGS_esp
#ifndef __KXEN__
#define call_with_regs(fn)                      \
        mov  %esp,%eax;                         \
        push %eax;                              \
        call fn;                                \
        add  $4,%esp;
#else
#define call_with_regs(fn)                      \
        get_user_regs(ax);                      \
        push r(ax);                             \
        call fn;                                \
        PARAM_POP(1);
#endif
#endif

        ALIGN
.globl vmx_asm_vmexit_handler
vmx_asm_vmexit_handler:

#ifndef __KXEN__
#if defined(__x86_64__)
        push %rdi
        push %rsi
        push %rdx
        push %rcx
        push %rax
        push %r8
        push %r9
        push %r10
        push %r11
        push %rbx
        push %rbp
        push %r12
        push %r13
        push %r14
        push %r15
#else /* defined(__i386__) */
        push %eax
        push %ebp
        push %edi
        push %esi
        push %edx
        push %ecx
        push %ebx
#endif
#else
#define savereg(_x) mov r(_x), UREGS_r(_x)(r(di))
        push r(di)
        mov  CURRENT_vcpu_stack_offset+SIZEOF_REG(r(sp)), r(di)
        lea  VCPU_user_regs(r(di)),r(di)
        savereg(ax)
        savereg(bp)
        savereg(si)
        savereg(dx)
        savereg(cx)
        savereg(bx)
        pop  UREGS_r(di)(r(di))

#if defined(__x86_64__)
        savereg(8)
        savereg(9)
        savereg(10)
        savereg(11)
        savereg(12)
        savereg(13)
        savereg(14)
        savereg(15)
#endif
#undef savereg
#endif

        get_current(bx)

        movb $1,VCPU_vmx_launched(r(bx))

#ifndef __KXEN__
        lea  UREGS_rip(r(sp)),r(di)
#else
        get_user_regs(di)
        add  $UREGS_rip,r(di)
#endif
        mov  $GUEST_RIP,r(ax)
        /*VMREAD(UREGS_rip)*/
        .byte 0x0f,0x78,0x07  /* vmread r(ax),(r(di)) */
        mov  $GUEST_RSP,r(ax)
        VMREAD(UREGS_rsp)
        mov  $GUEST_RFLAGS,r(ax)
        VMREAD(UREGS_eflags)

        mov  %cr2,r(ax)
        mov  r(ax),VCPU_hvm_guest_cr2(r(bx))

#ifndef NDEBUG
        mov  $0xbeef,%ax
#ifndef __KXEN__
        mov  %ax,UREGS_error_code(r(sp))
        mov  %ax,UREGS_entry_vector(r(sp))
        mov  %ax,UREGS_saved_upcall_mask(r(sp))
        mov  %ax,UREGS_cs(r(sp))
        mov  %ax,UREGS_ds(r(sp))
        mov  %ax,UREGS_es(r(sp))
        mov  %ax,UREGS_fs(r(sp))
        mov  %ax,UREGS_gs(r(sp))
        mov  %ax,UREGS_ss(r(sp))
#else
        get_user_regs(di)
        mov  %ax,UREGS_error_code(r(di))
        mov  %ax,UREGS_entry_vector(r(di))
        mov  %ax,UREGS_saved_upcall_mask(r(di))
        mov  %ax,UREGS_cs(r(di))
        mov  %ax,UREGS_ds(r(di))
        mov  %ax,UREGS_es(r(di))
        mov  %ax,UREGS_fs(r(di))
        mov  %ax,UREGS_gs(r(di))
        mov  %ax,UREGS_ss(r(di))
#endif
#endif

#ifndef __KXEN__
        call_with_regs(vmx_vmexit_handler)
#else
#ifndef __KXEN__
        push 0(r(sp))
        get_user_regs(ax)
        mov  r(ax),4(r(sp))
#endif
#ifdef __KXEN_MACOS__
        lidt (%esp)
        add  $8,%esp
        lgdt (%esp)
        /* add $8,%esp */       /* leave gdt on stack */
        mov  8(%esp),%eax
        andb $0xf8,%al          /* descriptor offset */
        add  2(%esp),%eax       /* add gdt base address */
        andl $~0x200,4(%eax)    /* clear busy flag in tr descriptor */
        mov  8(%esp),%eax
        ltr  %ax
        add  $12,%esp
        pop  %eax
        lldt %ax
        pop  %ds
        pop  %fs
        pop  %es
#endif

#if defined(__x86_64__)
        pop  r(15)
        pop  r(14)
        pop  r(13)
        pop  r(12)
#endif

        pop  r(di)
        pop  r(si)
        pop  r(bx)
        pop  r(bp)

        jmp vmx_vmexit_handler

vmx_asm_do_vmentry_no_push:
#if defined(__x86_64__)
        pop  r(15)
        pop  r(14)
        pop  r(13)
        pop  r(12)
#endif

#ifdef __KXEN_MACOS__
        add  $EXTRA_stack_space,%esp
#endif

        pop  r(di)
        pop  r(si)
        pop  r(bx)
        pop  r(bp)
        ret
#endif

.globl vmx_asm_do_vmentry
vmx_asm_do_vmentry:
        push r(bp)
        push r(bx)
        push r(si)
        push r(di)

#if defined(__x86_64__)
        push r(12)
        push r(13)
        push r(14)
        push r(15)
#endif

#ifdef __KXEN_MACOS__
        push %es
        push %fs
        push %ds
        xor  %eax, %eax
        sldt %eax
        push %eax
        str  %eax
        push %eax
        sub  $8,%esp
        sgdt (%esp)
        sub  $8,%esp
        sidt (%esp)
#endif

        call vmx_intr_assist

        get_current(bx)
        cli

        testb $0xff,VCPU_fpu_dirtied(r(bx))
        jz    1f
        movd  %xmm0, %eax
1:
#ifndef __KXEN__
        mov  VCPU_processor(r(bx)),%eax
        shl  $IRQSTAT_shift,r(ax)
        lea  addr_of(irq_stat),r(dx)
        cmpl $0,(r(dx),r(ax),1)
        jnz  .Lvmx_process_softirqs
#endif

        cmpl $0,VCPU_softirq_pending(r(bx))
        jnz  .Lvmx_process_softirqs_vcpu

        testb $0xff,VCPU_vmx_emulate(r(bx))
        jnz .Lvmx_goto_emulator
        testb $0xff,VCPU_vmx_realmode(r(bx))
        jz .Lvmx_not_realmode
        cmpw $0,VCPU_vm86_seg_mask(r(bx))
        jnz .Lvmx_goto_emulator
        call_with_regs(vmx_enter_realmode) 

.Lvmx_not_realmode:
        mov  VCPU_hvm_guest_cr2(r(bx)),r(ax)
        mov  r(ax),%cr2
        call vmx_trace_vmentry

#ifdef __KXEN__
        mov  $HOST_RSP, %eax
        .byte 0x0f,0x79,0xc4  /* vmwrite r(sp),r(ax) */
        get_user_regs(di)
        add  $UREGS_rip,r(di)
#else
        lea  UREGS_rip(r(sp)),r(di)
#endif
        mov  $GUEST_RIP,%eax
        /*VMWRITE(UREGS_rip)*/
        .byte 0x0f,0x79,0x07  /* vmwrite (r(di)),r(ax) */
        mov  $GUEST_RSP,%eax
        VMWRITE(UREGS_rsp)
        mov  $GUEST_RFLAGS,%eax
        VMWRITE(UREGS_eflags)

#ifndef __KXEN__
        cmpb $0,VCPU_vmx_launched(r(bx))
#if defined(__x86_64__)
        pop  %r15
        pop  %r14
        pop  %r13
        pop  %r12
        pop  %rbp
        pop  %rbx
        pop  %r11
        pop  %r10
        pop  %r9
        pop  %r8
        pop  %rax
        pop  %rcx
        pop  %rdx
        pop  %rsi
        pop  %rdi
#else /* defined(__i386__) */
        pop  %ebx
        pop  %ecx
        pop  %edx
        pop  %esi
        pop  %edi
        pop  %ebp
        pop  %eax
#endif
#else
#define restreg(_x)  mov UREGS_r(_x)(r(di)), r(_x)
        get_user_regs(di)
        cmpb $0,VCPU_vmx_launched(r(bx))
        restreg(ax)
        restreg(bp)
        restreg(si)
        restreg(dx)
        restreg(cx)
        restreg(bx)
#if defined(__x86_64__)
        restreg(8)
        restreg(9)
        restreg(10)
        restreg(11)
        restreg(12)
        restreg(13)
        restreg(14)
        restreg(15)
#endif
        restreg(di)
#undef restreg
#endif
        je   .Lvmx_launch

/*.Lvmx_resume:*/
        VMRESUME
        sti
        call vm_resume_fail
        ud2

.Lvmx_launch:
        VMLAUNCH
        sti
        call vm_launch_fail
        ud2

.Lvmx_goto_emulator:
        sti
        call_with_regs(vmx_realmode)
        jmp  vmx_asm_do_vmentry_no_push

#ifndef __KXEN__
.Lvmx_process_softirqs:
        sti
        call do_softirq
        jmp  vmx_asm_do_vmentry_no_push
#endif

.Lvmx_process_softirqs_vcpu:
        sti
#if !defined(__x86_64__)
        push r(bx)
#else
        mov  r(bx), r(di)
#endif
        call do_softirq_vcpu
#if !defined(__x86_64__)
        PARAM_POP(1)
#endif
        jmp  vmx_asm_do_vmentry_no_push
