
#include <xen/config.h>
#include <xen/init.h>
#include <xen/lib.h>
#include <xen/sched.h>
#include <xen/domain.h>
#include <xen/event.h>
#include <xen/time.h>
#include <xen/perfc.h>
#include <xen/sched-if.h>
#include <xen/softirq.h>
#include <asm/atomic.h>
#include <xen/errno.h>
#include <asm/ipi.h>

struct vm_info;
struct hostsched_info {
    struct vm_info_shared *hi_vm_info;
    int hi_host_halted;
};

void hostsched_kick_vcpu(struct vcpu *, unsigned int);

static int
hostsched_vcpu_init(struct vcpu *v)
{
    struct hostsched_info *hi;

    hi = xmalloc(struct hostsched_info);
    if (hi == NULL)
        return -1;

    memset(hi, 0, sizeof(*hi));
    v->sched_priv = hi;

    return 0;
}

static void
hostsched_vcpu_destroy(struct vcpu *v)
{
    xfree(v->sched_priv);
}

static void
hostsched_vcpu_sleep(struct vcpu *v)
{

    vcpu_raise_softirq(v, SCHEDULE_SOFTIRQ);
}

static void
hostsched_vcpu_wake(struct vcpu *v)
{
    struct hostsched_info *hi = v->sched_priv;
    ASSERT(hi);
    if (test_bit(0, &hi->hi_host_halted)) {
        perfc_incr(hostsched_wake_vm);
        hxen_info.ki_wake_vm(hi->hi_vm_info);
    }
}

static void
hostsched_vcpu_yield(struct vcpu *v) 
{
}

static void
hostsched_init(void)
{
}

void
hostsched_setup_vm(struct vcpu *v, struct vm_info_shared *vm_info)
{
    struct hostsched_info *hi = v->sched_priv;
    ASSERT(hi);
    hi->hi_vm_info = vm_info;
}

void
hostsched_halt_vm(struct vcpu *v)
{
    struct hostsched_info *hi = v->sched_priv;
    ASSERT(hi);
    perfc_incr(hostsched_halt_vm);
    set_bit(0, &hi->hi_host_halted);
    local_irq_enable();
    if (!softirq_pending_vcpu(current) &&
        current->runstate.state >= RUNSTATE_blocked)
        hxen_info.ki_halt_vm(hi->hi_vm_info);
    clear_bit(0, &hi->hi_host_halted);
}

void
hostsched_vcpu_softirq(struct vcpu *v)
{

    hostsched_vcpu_wake(v);
    if (smp_processor_id() != v->processor)
        hostsched_kick_vcpu(v, EVENT_CHECK_VECTOR);
}

void
hostsched_set_timer_vcpu(struct vcpu *v, uint64_t expire)
{
    struct hostsched_info *hi = v->sched_priv;

    if (hi->hi_vm_info)
        hxen_info.ki_set_timer_vcpu(hi->hi_vm_info, expire);
    else
        printk("hostsched_set_timer_vcpu %d no vm_info\n", v->vcpu_id);
}

void
hostsched_kick_vcpu(struct vcpu *v, unsigned int vector)
{
    struct hostsched_info *hi = v->sched_priv;

    if (hi->hi_vm_info)
        hxen_ivi(hi->hi_vm_info, vector);
    else
        printk("hostsched_kick_vcpu %d no vm_info\n", v->vcpu_id);
}

void
hostsched_dump_vcpu(struct vcpu *v)
{
    struct hostsched_info *hi = v->sched_priv;

    if (hi == NULL || hi->hi_vm_info == NULL)
        return;

    printk("    Notifying vcpu (halted %x, runstate %x, pause %lx/%x/%x)\n",
           hi->hi_host_halted, v->runstate.state, v->pause_flags,
		   atomic_read(&v->pause_count), atomic_read(&v->domain->pause_count));
    hxen_info.ki_wake_vm(hi->hi_vm_info);
}

struct scheduler sched_host_def = {
    .name           = "Host Scheduler",
    .opt_name       = "host",
    .sched_id       = XEN_SCHEDULER_HOST,

    .init_vcpu      = hostsched_vcpu_init,
    .destroy_vcpu   = hostsched_vcpu_destroy,

    .sleep          = hostsched_vcpu_sleep,
    .wake           = hostsched_vcpu_wake,
    .yield          = hostsched_vcpu_yield,

    .init           = hostsched_init,
};

/*
 * Local variables:
 * mode: C
 * c-set-style: "BSD"
 * c-basic-offset: 4
 * tab-width: 4
 * indent-tabs-mode: nil
 * End:
 */
