
#include <xen/types.h>
#include <xen/init.h>
#include <xen/spinlock.h>
#include <xen/serial.h>
#include <xen/lib.h>
#include <xen/irq.h>
#include <xen/smp.h>

#ifndef __KXEN__
#include <asm/processor.h>
#else
/* REP NOP (PAUSE) is a good thing to insert into busy-wait loops. */
static always_inline void rep_nop(void)
{
    asm volatile ( "rep;nop" : : : "memory" );
}

#define cpu_relax() rep_nop()
#endif

static struct serial_port com[2] = {
    { .rx_lock = SPIN_LOCK_UNLOCKED, .tx_lock = SPIN_LOCK_UNLOCKED },
    { .rx_lock = SPIN_LOCK_UNLOCKED, .tx_lock = SPIN_LOCK_UNLOCKED }
};

static void __serial_putc(struct serial_port *port, char c)
{
    if ( (port->txbuf != NULL) && !port->sync )
    {
        /* Interrupt-driven (asynchronous) transmitter. */

#ifndef __KXEN__
        if ( port->tx_quench )
        {
            /* Buffer filled and we are dropping characters. */
            if ( (port->txbufp - port->txbufc) > (serial_txbufsz / 2) )
                return;
            port->tx_quench = 0;
        }

        if ( (port->txbufp - port->txbufc) == serial_txbufsz )
        {
            if ( port->tx_log_everything )
            {
                /* Buffer is full: we spin waiting for space to appear. */
                int i;
                while ( !port->driver->tx_empty(port) )
                    cpu_relax();
                for ( i = 0; i < port->tx_fifo_size; i++ )
                    port->driver->putc(
                        port,
                        port->txbuf[mask_serial_txbuf_idx(port->txbufc++)]);
                port->txbuf[mask_serial_txbuf_idx(port->txbufp++)] = c;
            }
            else
            {
                /* Buffer is full: drop chars until buffer is half empty. */
                port->tx_quench = 1;
            }
            return;
        }

        if ( ((port->txbufp - port->txbufc) == 0) &&
                  port->driver->tx_empty(port) )
        {
            /* Buffer and UART FIFO are both empty. */
            port->driver->putc(port, c);
        }
        else
        {
            /* Normal case: buffer the character. */
            port->txbuf[mask_serial_txbuf_idx(port->txbufp++)] = c;
        }
#endif
    }
    else if ( port->driver->tx_empty )
    {
        /* Synchronous finite-capacity transmitter. */
        while ( !port->driver->tx_empty(port) )
            cpu_relax();
        port->driver->putc(port, c);
    }
    else
    {
        /* Simple synchronous transmitter. */
        port->driver->putc(port, c);
    }
}

void serial_putc(int handle, char c)
{
    struct serial_port *port;
    unsigned long flags;

    if ( handle == -1 )
        return;

    port = &com[handle & SERHND_IDX];
    if ( !port->driver || !port->driver->putc )
        return;

    spin_lock_irqsave(&port->tx_lock, flags);

    if ( (c == '\n') && (handle & SERHND_COOKED) )
        __serial_putc(port, '\r' | ((handle & SERHND_HI) ? 0x80 : 0x00));

    if ( handle & SERHND_HI )
        c |= 0x80;
    else if ( handle & SERHND_LO )
        c &= 0x7f;

    __serial_putc(port, c);

    spin_unlock_irqrestore(&port->tx_lock, flags);
}

void serial_puts(int handle, const char *s)
{
    struct serial_port *port;
    unsigned long flags;
    char c;

    if ( handle == -1 )
        return;

    port = &com[handle & SERHND_IDX];
    if ( !port->driver || !port->driver->putc )
        return;

    spin_lock_irqsave(&port->tx_lock, flags);

    while ( (c = *s++) != '\0' )
    {
        if ( (c == '\n') && (handle & SERHND_COOKED) )
            __serial_putc(port, '\r' | ((handle & SERHND_HI) ? 0x80 : 0x00));

        if ( handle & SERHND_HI )
            c |= 0x80;
        else if ( handle & SERHND_LO )
            c &= 0x7f;

        __serial_putc(port, c);
    }

    spin_unlock_irqrestore(&port->tx_lock, flags);
}

void __devinit serial_init_preirq(void)
{
    int i;
    for ( i = 0; i < ARRAY_SIZE(com); i++ )
        if ( com[i].driver && com[i].driver->init_preirq )
            com[i].driver->init_preirq(&com[i]);
}

void serial_register_uart(int idx, struct uart_driver *driver, void *uart)
{
    /* Store UART-specific info. */
    com[idx].driver = driver;
    com[idx].uart   = uart;

    /* Default is no transmit FIFO. */
    com[idx].tx_fifo_size = 1;
}
