/******************************************************************************
 * current.h
 * 
 * Information structure that lives at the bottom of the per-cpu Xen stack.
 */

#ifndef __X86_CURRENT_H__
#define __X86_CURRENT_H__

#include <xen/config.h>
#include <xen/percpu.h>
#include <public/xen.h>
#include <asm/page.h>

#ifndef __KXEN__
struct vcpu;

struct cpu_info {
    struct cpu_user_regs guest_cpu_user_regs;
    unsigned int         processor_id;
    struct vcpu         *current_vcpu;
};

static inline struct cpu_info *get_cpu_info(void)
{
    struct cpu_info *cpu_info;
    __asm__ ( "and %%"__OP"sp,%0; or %2,%0"
              : "=r" (cpu_info)
              : "0" (~(STACK_SIZE-1)), "i" (STACK_SIZE-sizeof(struct cpu_info))
        );
    return cpu_info;
}
#else
static inline int host_processor_id(void)
{
#if defined(USE_EXCEPTION_RECORD_HOOK)
    unsigned long cpu_index;
    __asm__ __volatile__ ( "742:    ud2; ud2; ud2; ud2               ;"
			   "743:                                     ;"
			   ".section .os_code_addr.cpu.bx,\"\",@note ;"
			   "        .align 4                         ;"
			   "        .long 742b                       ;"
			   "        .long 743b                       ;"
			   ".previous                                 "
			   : "=b" (cpu_index) );
    return cpu_index;
#else
    return hxen_info.ki_host_processor_id();
#endif
}

static inline struct vcpu *host_current(void)
{
#if defined(USE_EXCEPTION_RECORD_HOOK)
    struct vcpu *_current;
    __asm__ __volatile__ ( "742:    ud2; ud2; ud2; ud2; ud2; ud2         ;"
			   "743:                                         ;"
			   ".section .os_code_addr.current.bx,\"\",@note ;"
			   "        .align 4                             ;"
			   "        .long 742b                           ;"
			   "        .long 743b                           ;"
			   ".previous                                     "
			   : "=b" (_current) );
    return _current;
#else
    return (struct vcpu *)hxen_info.ki_get_current();
#endif
}

extern unsigned int x86_cpu_to_xenid[];
#define cpu_xen_id(hostcpu)	x86_cpu_to_xenid[hostcpu]
#endif

#ifndef __KXEN__
#define get_current()         (get_cpu_info()->current_vcpu)
#define set_current(vcpu)     (get_cpu_info()->current_vcpu = (vcpu))
#else
#define get_current()         (host_current())
#define set_current(vcpu)     hxen_info.ki_set_current(vcpu)
#endif
#define current               (get_current())

#ifndef __KXEN__
#define get_processor_id()    (get_cpu_info()->processor_id)
#define set_processor_id(id)  (get_cpu_info()->processor_id = (id))
#else
#define get_processor_id()    (cpu_xen_id(host_processor_id()))
#endif

#ifndef __KXEN__
#define guest_cpu_user_regs() (&get_cpu_info()->guest_cpu_user_regs)
#else
#define guest_cpu_user_regs() (&current->arch.guest_context.user_regs)
#endif

#ifndef __KXEN__
/*
 * Get the bottom-of-stack, as stored in the per-CPU TSS. This actually points
 * into the middle of cpu_info.guest_cpu_user_regs, at the section that
 * precisely corresponds to a CPU trap frame.
 */
#define get_stack_bottom()                      \
    ((unsigned long)&get_cpu_info()->guest_cpu_user_regs.es)

#define reset_stack_and_jump(__fn)              \
    __asm__ __volatile__ (                      \
        "mov %0,%%"__OP"sp; jmp "STR(__fn)      \
        : : "r" (guest_cpu_user_regs()) : "memory" )
#endif

#define schedule_tail(vcpu) (((vcpu)->arch.schedule_tail)(vcpu))

/*
 * Which VCPU's state is currently running on each CPU?
 * This is not necesasrily the same as 'current' as a CPU may be
 * executing a lazy state switch.
 */
DECLARE_PER_CPU(struct vcpu *, curr_vcpu);

#endif /* __X86_CURRENT_H__ */
