/*
 *  hxen.c
 *  hxen
 *
 *  Copyright 2009 Citrix Systems, Inc. All rights reserved.
 * 
 */

#include "hxen.h"
#include "xnu_hacks.h"

#include <sys/systm.h>
#include <mach/mach_types.h>

#include <libkern/libkern.h>

OSMallocTag hxen_malloc_tag = NULL;

lck_attr_t *hxen_lck_attr = NULL;
lck_grp_t *hxen_lck_grp = NULL;
lck_grp_attr_t *hxen_lck_grp_attr = NULL;

uint8_t *hxen_hv = NULL;
size_t hxen_size = 0;

boolean_t hxen_shutting_down = FALSE;

static void
free_all(void)
{
    if (hxen_call_mutex) {
        lck_mtx_free(hxen_call_mutex, hxen_lck_grp);
        hxen_call_mutex = NULL;
    }

    if (hxen_device_mutex) {
        lck_mtx_free(hxen_device_mutex, hxen_lck_grp);
        hxen_device_mutex = NULL;
    }

    if (hxen_lck_grp) {
        lck_grp_free(hxen_lck_grp);
        hxen_lck_grp = NULL;
    }

    if (hxen_malloc_tag) {
        OSMalloc_Tagfree(hxen_malloc_tag);
        hxen_malloc_tag = NULL;
    }

    if (hxen_lck_attr) {
        lck_attr_free(hxen_lck_attr);
        hxen_lck_attr = NULL;
    }

    if (hxen_lck_grp_attr) {
        lck_grp_attr_free(hxen_lck_grp_attr);
        hxen_lck_grp_attr = NULL;
    }
}

kern_return_t
hxen_start (kmod_info_t * ki, void * d)
{
    kern_return_t ret = KERN_FAILURE;

    dprintk("hxen.kext loading\n");

    hxen_malloc_tag = OSMalloc_Tagalloc(KXEN_BUNDLE_IDENTIFIER,
                                        OSMT_DEFAULT);
    if (hxen_malloc_tag == NULL)
        goto error;

    hxen_lck_attr = lck_attr_alloc_init();
    hxen_lck_grp_attr = lck_grp_attr_alloc_init();
    lck_attr_setdebug(hxen_lck_attr);

    hxen_lck_grp = lck_grp_alloc_init(KXEN_BUNDLE_IDENTIFIER,
				      hxen_lck_grp_attr);
    if (hxen_lck_grp == NULL)
	goto error;

    hxen_device_mutex = lck_mtx_alloc_init(hxen_lck_grp, hxen_lck_attr);
    if (hxen_device_mutex == NULL) {
	ret = ENOMEM;
	goto error;
    }

    hxen_call_mutex = lck_mtx_alloc_init(hxen_lck_grp, hxen_lck_attr);
    if (hxen_call_mutex == NULL) {
	ret = ENOMEM;
	goto error;
    }

    ret = hxen_device_start();
    if (ret != KERN_SUCCESS)
	goto error;

    ret = hxen_cpu_init();
    if (ret)
	goto error;

    hxen_printk("hxen.kext loaded\n");
    return KERN_SUCCESS;

error:
    free_all();
    dprintk("hxen.kext load failed: %d\n", ret);
    return ret;
}


kern_return_t
hxen_stop (kmod_info_t * ki, void * d)
{
    kern_return_t ret;

    dprintk("hxen.kext unloading\n");

    hxen_shutting_down = TRUE;

    ret = hxen_device_stop();
    if (ret != KERN_SUCCESS)
	return ret;

    free_all();

    hxen_printk("hxen.kext unloaded\n");
    return KERN_SUCCESS;
}

lck_mtx_t *
hxen_lck_mtx_alloc_init(void)
{

    return lck_mtx_alloc_init(hxen_lck_grp, hxen_lck_attr);
}

void
hxen_lck_mtx_free(lck_mtx_t *mtx)
{
    lck_mtx_free(mtx, hxen_lck_grp);
}
