/*
 *  hxen.h
 *  hxen
 *
 *  Copyright 2009 Citrix Systems, Inc. All rights reserved.
 *
 */

#ifndef _KXEN_H_
#define _KXEN_H_

#include <stdarg.h>
#include <libkern/libkern.h>
#include <sys/conf.h>
#include <mach/mach_types.h>
#include <kern/assert.h>
#include <xen/types.h>

#include <hxen_info.h>

#define KXEN_DPRINTK
#define KXEN_XNU_HACKS

#define	KXEN_DRIVER_VERSION_MAJOR	0x00000000
#define	KXEN_DRIVER_VERSION_MINOR	0x00000000
#define	KXEN_DRIVER_VERSION_TAG		"preview"

#include <IOKit/IOLocks.h>
#include <libkern/OSMalloc.h>

#include "hxen_def.h"
#include <hxen_os.h>
#include <hxen_ioctl.h>

#define	KXEN_NDEVICES	4

#define MAX_MACHINE_FRAME 0x100000;

// Apple system clock increments in 1ns units, ie 1,000,000,000Hz.
#define KXEN_HOST_TIME_FREQUENCY (1000000000 >> 7)

#define MAXIMUM_PROCESSORS /* MAX_CPUS */ 32

struct selinfo_space {
    unsigned char data[64];
};
/*
 * Requestable events.  If poll(2) finds any of these set, they are
 * copied to revents on return.
 */
#define	POLLIN		0x0001		/* any readable data available */
#define	POLLPRI		0x0002		/* OOB/Urgent readable data */
#define	POLLOUT		0x0004		/* file descriptor is writeable */
#define	POLLRDNORM	0x0040		/* non-OOB/URG data available */
#define	POLLWRNORM	POLLOUT		/* no write type differentiation */
#define	POLLRDBAND	0x0080		/* OOB/Urgent readable data */
#define	POLLWRBAND	0x0100		/* OOB/Urgent data can be written */

struct vm_info {
    struct vm_info_shared vi_shared;
    // KTIMER vi_timer;
    void *vi_timer;
    uint64_t vi_timer_deadline;
    // KDPC vi_timer_dpc;
    unsigned long vi_host_cpu;
    // KDPC vi_ipi_dpc;
    uint32_t vi_raised_vectors;
    thread_t vi_ipi_thread;
    semaphore_t vi_ipi_sem;
    uint32_t vi_ioreq_pending;
    struct selinfo_space vi_selinfo;
    lck_mtx_t *vi_execute_mtx;
    semaphore_t vi_halt_sem;
};

struct hxen_vmmem {
    void *vmmem;
    void *umap;
    void *kmap;
};

struct hxen_device {
    dev_t kd_dev;
    void *kd_cdev;
    // KEVENT de_runnable;
    int de_nrpages;
    struct vm_info de_vm_info;
    int de_nrmaps;
    struct hxen_vmmem *de_maps;
    /* KEVENT * */void *de_ioemu_request_event;
    // KEVENT *de_ioemu_completed_event;
};

/* hxen.c */
extern lck_mtx_t *hxen_device_mutex;
extern OSMallocTag hxen_malloc_tag;

extern uint8_t *hxen_hv;
extern size_t hxen_size;
extern struct hxen_device hxen_dev, *hxen_device;
extern struct hxen_device *hxen_device_table[];

extern boolean_t hxen_shutting_down;

extern lck_mtx_t *hxen_lck_mtx_alloc_init(void);
extern void hxen_lck_mtx_free(lck_mtx_t *);

/* hxen_call.c */
#define EEXCEPT 199
extern int hxen_try_call(long (__cdecl *)(void *), void *, void *);
extern void *__cdecl hxen_get_current(void);
extern void __cdecl hxen_set_current(void *);
extern lck_mtx_t *hxen_call_mutex;

/* hxen_cpu.c */
extern int hxen_cpu_init(void);
extern unsigned long hxen_cpu_vm;
extern unsigned long hxen_first_cpu(void);
extern void hxen_cpu_pin(unsigned long);
extern void hxen_cpu_pin_current(void);
extern void hxen_cpu_pin_first(void);
#define hxen_cpu_pin_dom0() hxen_cpu_pin_first()
extern void hxen_cpu_pin_vcpu(struct vm_info *);
extern void hxen_cpu_unpin(void);
extern void hxen_set_cpu_active_mask(void *, int);
extern void __cdecl hxen_on_each_cpu(void (*)(void *), void *);

/* hxen_device.c */
extern kern_return_t hxen_device_start(void);
extern kern_return_t hxen_device_stop(void);

/* hxen_load.c */
struct hxen_load_desc;
extern int hxen_load(struct hxen_load_desc *);
extern int hxen_unload(void);

/* hxen_vmmem.c */
extern void *hxen_vmmem_alloc(size_t, uint32_t);
extern void hxen_vmmem_free(void *);
extern void *hxen_vmmem_map(void *, task_t, uint8_t **);
extern void hxen_vmmem_unmap(void *);
extern uintptr_t hxen_vmmem_get_mfn(void *, uint32_t);
extern void *hxen_vmmem_mmap_pages(task_t, unsigned int, xen_pfn_t *, int);
extern int hxen_vmmem_munmap(uint8_t *);

/* hxen_mem.c */
extern void *kernel_malloc(size_t);
extern void kernel_free(void *, size_t);
uint32_t *kernel_get_mfn_list(uintptr_t, uint32_t);
int hxen_mmapbatch(struct hxen_mmapbatch_desc *);
int hxen_munmap(struct hxen_munmap_desc *);
int hxen_create_vmappings(struct hxen_vmappings_desc *, struct hxen_device *);
void hxen_free_vmappings(struct hxen_device *);
void * __cdecl hxen_map_pages(unsigned int, void *);
int __cdecl hxen_unmap_pages(unsigned int, void *);

/* hxen_ops.c */
extern void hxen_init_free_allocs(void);
extern int hxen_init(void);
extern int hxen_shutdown(void);
int hxen_version(struct hxen_version_desc *);
int hxen_execute(void);
int hxen_keyhandler(char *);
int hxen_set_ioemu_events(struct hxen_ioemu_events_desc *,
			  struct hxen_device *);
extern uint64_t hxen_cpu_timer_deadline[];

/* hxen_debug.c */
extern int __cdecl hxen_dprintk(const char *, ...);
#ifdef KXEN_DPRINTK
#define dprintk(fmt, args...) hxen_dprintk(fmt, ##args)
#else
#define dprintk(fmt, ...) do {} while (0)
#endif
#define fail_msg(fmt, args...) hxen_printk("hxen: " fmt "\n", ##args)
extern int hxen_printk(const char *, ...);

#ifdef KXEN_XNU_HACKS
/* xnu_hacks.c */
#include "xnu_hacks.h"
#endif

#endif	/* _KXEN_H_ */
