/*
 *  hxen_device.c
 *  hxen
 *
 *  Copyright 2009 Citrix Systems, Inc. All rights reserved.
 *
 */

#include <libkern/libkern.h>
#include <sys/conf.h>
#include <sys/errno.h>
#include <sys/proc.h>
#include <mach/mach_types.h>
#include <miscfs/devfs/devfs.h>

#include "hxen.h"
#include <asm/system.h>

static int hxen_device_major = -1;

struct hxen_device hxen_dev;
struct hxen_device *hxen_device;
struct hxen_device *hxen_device_table[KXEN_NDEVICES];

d_open_t hxen_device_open;
d_close_t hxen_device_close;
d_ioctl_t hxen_device_ioctl;
d_select_t hxen_device_select;

static struct cdevsw hxen_device_cdevsw = {
    hxen_device_open,		/* open */
    hxen_device_close,		/* close */
    eno_rdwrt,			/* read */
    eno_rdwrt,			/* write */
    hxen_device_ioctl,		/* ioctl */
    eno_stop,			/* stop */
    eno_reset,			/* reset */
    0,				/* ttys */
    hxen_device_select,		/* select */
    eno_mmap,			/* mmap */
    eno_strat,			/* strategy */
    eno_getc,			/* getc */
    eno_putc,			/* putc */
    0,				/* flags */
};

lck_mtx_t *hxen_device_mutex = NULL;

#define DEVICE_LOCK() lck_mtx_lock(hxen_device_mutex)
#define DEVICE_UNLOCK() lck_mtx_unlock(hxen_device_mutex)

int
hxen_device_open(dev_t dev, __unused int flags, __unused int devtype,
		 struct proc *p)
{
    int devid;

    dprintk("hxen_device_open\n");

    devid = minor(dev);
    if ((devid >= KXEN_NDEVICES) || (devid < 0))
	return ENOENT;

    return KERN_SUCCESS;
}

int
hxen_device_close(dev_t dev, __unused int flags, __unused int devtype,
		  struct proc *p)
{
    int devid;

    dprintk("hxen_device_close\n");

    devid = minor(dev);
    if ((devid >= KXEN_NDEVICES) || (devid < 0))
	return ENOENT;

    return KERN_SUCCESS;
}

kern_return_t
hxen_device_start(void)
{

    DEVICE_LOCK();

    bzero((void *)&hxen_dev, sizeof(hxen_dev));
    bzero((void *)hxen_device_table, sizeof(hxen_device_table));

    hxen_device_major = cdevsw_add(-1, &hxen_device_cdevsw);
    if (hxen_device_major == -1)
	goto error;

    hxen_dev.kd_dev = makedev(hxen_device_major, 0);
    hxen_dev.kd_cdev = devfs_make_node(hxen_dev.kd_dev, DEVFS_CHAR,
				       UID_ROOT, GID_OPERATOR,
				       0660, KXEN_DEVICE_PREFIX, 0);
    if (hxen_dev.kd_cdev == NULL)
	goto error;

    hxen_device_table[0] = &hxen_dev;
    hxen_device = &hxen_dev;

    DEVICE_UNLOCK();

    return KERN_SUCCESS;

  error:
    hxen_dev.kd_dev = 0;

    if (hxen_device_major != -1)
	(void)cdevsw_remove(hxen_device_major, &hxen_device_cdevsw);
    hxen_device_major = -1;

    DEVICE_UNLOCK();

    return KERN_FAILURE;
}

kern_return_t
hxen_device_stop(void)
{
    int i;

    DEVICE_LOCK();

    if (hxen_device_major == -1) {
	DEVICE_UNLOCK();
	return KERN_SUCCESS;
    }

    /* XXX check any open */

    hxen_unload();

    for (i = 0; i < KXEN_NDEVICES; i++) {
	if (hxen_device_table[i] == NULL)
	    continue;
	devfs_remove(hxen_device_table[i]->kd_cdev);
	hxen_device_table[i]->kd_cdev = NULL;
	hxen_device_table[i]->kd_dev = 0;
	if (i != 0) {
	    /* free */
	}
	hxen_device_table[i] = NULL;
    }

    (void)cdevsw_remove(hxen_device_major, &hxen_device_cdevsw);
    hxen_device_major = -1;

    DEVICE_UNLOCK();

    return KERN_SUCCESS;
}

int
hxen_device_select(dev_t dev, int events, void *wql, struct proc *p)
{
    struct vm_info *vi;
    int revents = 0;

    vi = &hxen_device->de_vm_info;

    if (events & (POLLIN | POLLRDNORM)) {
	if (vi->vi_ioreq_pending)
	    revents |= (events & (POLLIN | POLLRDNORM));
	else
	    selrecord((proc_t)p, (struct selinfo *)&vi->vi_selinfo,
		      wql);
    }

    return revents;
}
