/* 
 *  hxen_ioctl.c
 *  hxen
 *
 *  Copyright 2009 Citrix Systems, Inc. All rights reserved.
 * 
 */

#include "hxen.h"
#include "hxen_call.h"

#include <libkern/libkern.h>
#include <sys/conf.h>
#include <mach/mach_types.h>
// #include <miscfs/devfs/devfs.h>
#include <sys/proc.h>

#include <sys/errno.h>
#include <xen/types.h>

#define KXEN_DEFINE_SYMBOLS_PROTO
#include <hxen_link.h>
KXEN_PROTOTYPES(extern)

#if 0
#define IOCTL_TRACE(fmt, args...) dprintk(fmt, ##args)
#else
#define IOCTL_TRACE(fmt, ...)
#endif

static int hxen_mode;
#define	KXEN_MODE_IDLE		0
#define	KXEN_MODE_LOADED	1
#define	KXEN_MODE_INITIALIZED	2

#define IOCTL_FAILURE(r, fmt, args...) do {	\
	ret = r;				\
	fail_msg(fmt, ##args);			\
    } while (0)
    
#define SET_KXEN_MODE(m)			\
    hxen_mode = (m);
#define CHECK_KXEN_MODE(m, id) do {					\
	if (hxen_mode < (m)) {						\
	    IOCTL_FAILURE(EINVAL,					\
			  "hxen_ioctl(" id ") invalid sequence");	\
	    goto out;							\
	}								\
    } while (0)
#define CHECK_KXEN_MODE_NOT(m, id) do {					\
	if (hxen_mode >= (m)) {						\
	    IOCTL_FAILURE(EINVAL,					\
			  "hxen_ioctl(" id ") invalid sequence");	\
	    goto out;							\
	}								\
    } while (0)

#define KXEN_OP(name, fn, arg, fnargs...) do {				\
	IOCTL_TRACE("hxen_ioctl(" name ", %p)\n", udata);		\
	CHECK_KXEN_MODE(KXEN_MODE_INITIALIZED, name);			\
	if (udata == NULL) {						\
	    IOCTL_FAILURE(EINVAL,					\
			  "hxen_ioctl(" name ") input arguments");	\
	    goto out;							\
	}								\
	hxen_cpu_pin_dom0();						\
	ret = fn((arg *)udata, ##fnargs);				\
	hxen_cpu_unpin();						\
    } while (0)

#define KXEN_DOM0_CALL(name, fn, arg) do {				\
	DECLARE_EXCEPTION_REGISTRATION_RECORD(hxen_rec);		\
	IOCTL_TRACE("hxen_ioctl(" name ", %p)\n", udata);		\
	CHECK_KXEN_MODE(KXEN_MODE_INITIALIZED, name);			\
	if (udata == NULL) {						\
	    IOCTL_FAILURE(EINVAL,					\
			  "hxen_ioctl(" name ") input arguments");	\
	    goto out;							\
	}								\
	hxen_cpu_pin_dom0();						\
	HOOK_EXCEPTION_REGISTRATION_RECORD(hxen_rec,			\
					   hxen_info->ki_dom0_current); \
	ret = fn((arg *)udata);						\
	UNHOOK_EXCEPTION_REGISTRATION_RECORD(hxen_rec);			\
	hxen_cpu_unpin();						\
    } while (0)

int
hxen_device_ioctl(dev_t dev, u_long cmd, caddr_t udata,
		  __unused int flags, __unused struct proc *proc)
{
    int ret = EINVAL;
    int devid;
    struct hxen_device *devext;

    devid = minor(dev);
    if (devid != 0)
	return ENXIO;

    devext = hxen_device_table[devid];
    if (devext == NULL)
	return ENXIO;

    switch (cmd) {
    case KXENVERSION:
	IOCTL_TRACE("hxen_ioctl(KXENVERSION, %p)\n", udata);
	if (udata == NULL) {
	    IOCTL_FAILURE(EINVAL,
			  "hxen_ioctl(KXENVERSION) output arguments");
	    break;
	}
	ret = hxen_version((struct hxen_version_desc *)udata);
	break;
    case KXENLOAD:
	IOCTL_TRACE("hxen_ioctl(KXENLOAD, %p)\n", udata);
	CHECK_KXEN_MODE_NOT(KXEN_MODE_LOADED, "KXENLOAD");
	ret = hxen_load((struct hxen_load_desc *)udata);
	if (ret == 0)
	    SET_KXEN_MODE(KXEN_MODE_LOADED);
	break;
    case KXENUNLOAD:
	IOCTL_TRACE("hxen_ioctl(KXENUNLOAD)\n");
	CHECK_KXEN_MODE(KXEN_MODE_LOADED, "KXENUNLOAD");
	ret = hxen_unload();
	if (ret == 0)
	    SET_KXEN_MODE(KXEN_MODE_IDLE);
	break;
    case KXENINIT:
	IOCTL_TRACE("hxen_ioctl(KXENINIT)\n");
	CHECK_KXEN_MODE(KXEN_MODE_LOADED, "KXENINIT");
	CHECK_KXEN_MODE_NOT(KXEN_MODE_INITIALIZED, "KXENINIT");
	ret = hxen_init();
	if (ret == 0)
	    SET_KXEN_MODE(KXEN_MODE_INITIALIZED);
	/* XXX per VM/vcpu */
	// KeInitializeEvent(&devext->de_runnable, NotificationEvent, FALSE);
	break;
    case KXENSHUTDOWN:
	IOCTL_TRACE("hxen_ioctl(KXENSHUTDOWN)\n");
	CHECK_KXEN_MODE(KXEN_MODE_INITIALIZED, "KXENSHUTDOWN");
	ret = hxen_shutdown();
	if (ret == 0)
	    hxen_mode = KXEN_MODE_LOADED;
	break;
    case KXENREBOOT:
	IOCTL_TRACE("hxen_ioctl(KXENREBOOT)\n");
	// kdbgreboot();
	break;
    case KXENKEYHANDLER:
	KXEN_OP("KXENKEYHANDLER", hxen_keyhandler, char);
	break;
    case KXENHYPERCALL:
	KXEN_DOM0_CALL("KXENHYPERCALL", hxen_do_hypercall,
		       struct hxen_hypercall_desc);
	break;
    case KXENMEMOP:
	KXEN_DOM0_CALL("KXENMEMOP", hxen_do_memop, struct hxen_memop_desc);
	break;
    case KXENHVMOP:
	KXEN_DOM0_CALL("KXENHVMOP", hxen_do_hvmop, struct hxen_hvmop_desc);
	break;
    case KXENDOMCTL:
	KXEN_DOM0_CALL("KXENDOMCTL", hxen_do_domctl, struct hxen_domctl_desc);
	break;
    case KXENMMAPBATCH:
	KXEN_OP("KXENMMAPBATCH", hxen_mmapbatch, struct hxen_mmapbatch_desc);
	break;
    case KXENMUNMAP:
	KXEN_OP("KXENMUNMAP", hxen_munmap, struct hxen_munmap_desc);
	break;
    case KXENVMAPPINGS:
	KXEN_OP("KXENVMAPPINGS", hxen_create_vmappings,
		struct hxen_vmappings_desc, devext);
	break;
    case KXENEXECUTE:
	IOCTL_TRACE("hxen_ioctl(KXENEXECUTE)\n");
	CHECK_KXEN_MODE(KXEN_MODE_INITIALIZED, "KXENEXECUTE");
	ret = hxen_execute();
	break;
    case KXENSETIOEMUEVENTS:
	IOCTL_TRACE("hxen_ioctl(KXENSETIOEMUEVENTS)\n");
	KXEN_OP("KXENSETIOEMUEVENTS", hxen_set_ioemu_events,
		struct hxen_ioemu_events_desc, devext);
	break;
    case KXENIOREQCOMPLETE:
	IOCTL_TRACE("hxen_ioctl(KXENIOREQCOMPLETE)\n");
	CHECK_KXEN_MODE(KXEN_MODE_INITIALIZED, "KXENIOREQCOMPLETE");
	lck_mtx_lock(devext->de_vm_info.vi_execute_mtx);
	devext->de_vm_info.vi_ioreq_pending = 0;
	wakeup(&devext->de_vm_info.vi_ioreq_pending);
	lck_mtx_unlock(devext->de_vm_info.vi_execute_mtx);
	ret = 0;
	break;
    default:
	IOCTL_TRACE("hxen_ioctl(%lx)\n", cmd);
	IOCTL_FAILURE(EINVAL,
		      "hxen_ioctl(%lx) not implemented", cmd);
	break;
    }

  out:
    return ret;
}
