/*
 *  hxen_load.c
 *  hxen
 *
 *  Copyright 2009 Citrix Systems, Inc. All rights reserved.
 *
 */

#include "hxen.h"

#include <libkern/libkern.h>
#include <sys/conf.h>
#include <mach/mach_types.h>
#include <kern/assert.h>
#include <xen/errno.h>
#include <xen/types.h>

#include <public/libelf.h>

#define KXEN_DEFINE_SYMBOLS_CODE
#include <hxen_link.h>
#define _ignore_
KXEN_PROTOTYPES( _ignore_ )
#undef _ignore_
KXEN_GET_SYMS(hxen_get_symbols)
KXEN_CLEAR_SYMS(hxen_clear_symbols)

#include <hxen_load.h>

int
hxen_load(struct hxen_load_desc *kld)
{
    int ret;
    uint8_t *image, *image_u;
    struct elf_binary elf;
    const char *missing_symbol;

    if (hxen_hv) {
	fail_msg("load: already loaded");
	return EINVAL;
    }

    image = kernel_malloc(kld->kld_size);
    if (image == NULL) {
	fail_msg("load: malloc %d bytes failed", kld->kld_size);
	return ENOMEM;
    }

    get_xen_guest_handle(image_u, kld->kld_uvaddr);
    ret = copyin(CAST_USER_ADDR_T(image_u), image, kld->kld_size);
    if (ret != 0) {
	fail_msg("load: copyin failed");
	goto error;
    }

    ret = elf_init(&elf, (const char *)image, kld->kld_size);
    if (ret != 0) {
	fail_msg("load: elf_init failed");
	ret = EINVAL;
	goto error;
    }

    elf_set_verbose(&elf);
    elf_parse_binary(&elf);

    hxen_size = (size_t)(elf.pend - elf.pstart);
    hxen_hv = kernel_malloc(hxen_size);
    if (hxen_hv == NULL) {
	fail_msg("load: malloc %d bytes failed", hxen_size);
	ret = ENOMEM;
	goto error;
    }

    elf.dest = (char *)hxen_hv;
    elf_load_binary(&elf);

    ret = elf_reloc(&elf);
    if (ret != 0) {
	fail_msg("load: elf_reloc failed");
	ret = EINVAL;
	goto error;
    }

    ret = hxen_get_symbols(&elf, hxen_hv, &missing_symbol);
    if (ret != 0) {
	fail_msg("load: hxen get symbol %s failed", missing_symbol);
	ret = EINVAL;
	goto error;
    }

    ret = hxen_fixup_os_code(&elf, hxen_hv, "macos");
    if (ret != 0) {
	fail_msg("load: fixup os code failed");
	ret = EINVAL;
	goto error;
    }

    dprintk("hxen_load done\n");
    ret = 0;
error:
    if (ret && hxen_hv) {
	kernel_free(hxen_hv, hxen_size);
	hxen_hv = NULL;
    }
    if (image)
	kernel_free(image, kld->kld_size);
    return ret;
}

void
hxen_unload_free_allocs(void)
{

    hxen_init_free_allocs();

    if (hxen_hv) {
	kernel_free(hxen_hv, hxen_size);
	hxen_clear_symbols();
	hxen_hv = NULL;
    }
}

int
hxen_unload(void)
{

    if (hxen_hv == NULL)
	return 0;

    hxen_shutdown();

    hxen_unload_free_allocs();

    return 0;
}
