/*
 *  hxen_mem.c
 *  hxen
 *
 *  Copyright 2009 Citrix Systems, Inc. All rights reserved.
 *
 */

#include "hxen.h"

#include <xen/errno.h>
#include <public/xen.h>

#define KXEN_DEFINE_SYMBOLS_PROTO
#include <hxen_link.h>
KXEN_PROTOTYPES(extern)

void *
kernel_malloc(size_t size)
{
    return OSMalloc(size, hxen_malloc_tag);
}

void
kernel_free(void *addr, size_t size)
{
    OSFree(addr, size, hxen_malloc_tag);
}

struct pmap;
typedef struct pmap *pmap_t;
extern pmap_t kernel_pmap;
extern ppnum_t pmap_find_phys(pmap_t pmap, addr64_t va);

uint32_t *
kernel_get_mfn_list(uintptr_t va, uint32_t npages)
{
    uint32_t *pages;
    uint32_t i;

    pages = kernel_malloc(npages * sizeof(uint32_t));
    if (pages == NULL)
	return NULL;

    for (i = 0; i < npages; i++)
	pages[i] = pmap_find_phys(kernel_pmap,
				  (addr64_t)va + (i << PAGE_SHIFT));

    return pages;
}

int
hxen_mmapbatch(struct hxen_mmapbatch_desc *kmmapbd)
{
    void *addr;
    xen_pfn_t *mfns;

    get_xen_guest_handle(mfns, kmmapbd->kmd_arr);
    addr = hxen_vmmem_mmap_pages(current_task(), kmmapbd->kmd_num,
				 mfns, 1);
    if (addr == NULL) {
	fail_msg("hxen_mmapbatch hxen_vmmem_mmap_pages");
	return EINVAL;
    }

    kmmapbd->kmd_addr = (uint64_t)(uintptr_t)addr;
    return 0;
}

int
hxen_munmap(struct hxen_munmap_desc *kmd)
{

    dprintk("hxen_munmap\n");
    return hxen_vmmem_munmap((uint8_t *)(uintptr_t)kmd->kmd_addr);
}

int
hxen_create_vmappings(struct hxen_vmappings_desc *kvd,
		      struct hxen_device *devext)
{
    struct hxen_vmapping *vmappings;
    struct hxen_vmap *vmaptable;
    int ret = 0;
    int i, j, n;
    uint8_t *uvaddr, *kvaddr;
    uintptr_t mfn;
    XEN_GUEST_HANDLE(void) uvaddr_handle;

    if (devext->de_nrpages) {
	fail_msg("hxen_create_vmappings: cannot change nr pages");
	return EINVAL;
    }

    devext->de_nrpages = kvd->kvd_nrpages;

    devext->de_vm_info.vi_shared.vi_p2m = kernel_malloc(devext->de_nrpages *
							sizeof(uint32_t));
    if (devext->de_vm_info.vi_shared.vi_p2m == NULL) {
	fail_msg("hxen_create_vmappings: alloc p2m failed");
	ret = ENOMEM;
	goto out;
    }
    memset(devext->de_vm_info.vi_shared.vi_p2m, 0,
	   devext->de_nrpages * sizeof(uint32_t));

#define VMAPPING_MAP_PAGES (1 << KXEN_VMAPPING_SHIFT)

    devext->de_nrmaps = (devext->de_nrpages + VMAPPING_MAP_PAGES - 1) >>
	KXEN_VMAPPING_SHIFT;

    devext->de_maps = (struct hxen_vmmem*)kernel_malloc(
	devext->de_nrmaps * sizeof(struct hxen_vmmem));
    if (devext->de_maps == NULL) {
	fail_msg("hxen_create_vmappings: alloc maps array failed");
	ret = ENOMEM;
	goto out;
    }
    memset(devext->de_maps, 0, devext->de_nrmaps * sizeof(struct hxen_vmmem));

    get_xen_guest_handle(vmappings, kvd->kvd_vmappings);

    vmaptable = (struct hxen_vmap *)hxen_info->ki_vmaptable;

    for (i = 0; i < devext->de_nrmaps; i++) {
	n = devext->de_nrpages - (i << KXEN_VMAPPING_SHIFT);
	if (n > VMAPPING_MAP_PAGES)
	    n = VMAPPING_MAP_PAGES;
	devext->de_maps[i].vmmem = hxen_vmmem_alloc(n, hxen_info->ki_max_page);
	if (devext->de_maps[i].vmmem == NULL) {
	    fail_msg("hxen_create_vmappings: alloc failed");
	    ret = ENOMEM;
	    goto out;
	}
	devext->de_maps[i].umap = hxen_vmmem_map(devext->de_maps[i].vmmem,
						 current_task(), &uvaddr);
	if (devext->de_maps[i].umap == NULL) {
	    fail_msg("hxen_create_vmappings: user map failed");
	    ret = ENOMEM;
	    goto out;
	}
	devext->de_maps[i].kmap = hxen_vmmem_map(devext->de_maps[i].vmmem,
						 kernel_task, &kvaddr);
	if (devext->de_maps[i].kmap == NULL) {
	    fail_msg("hxen_create_vmappings: kernel map failed");
	    ret = ENOMEM;
	    goto out;
	}

	set_xen_guest_handle(uvaddr_handle, uvaddr);
	ret = copyout(&uvaddr_handle,
		      (user_addr_t)(uintptr_t)&vmappings[i].vaddr,
		      sizeof(vmappings[i].vaddr));
	if (ret != 0) {
	    fail_msg("hxen_create_vmappings: copyout failed");
	    ret = EINVAL;
	    goto out;
	}

	for (j = 0; j < n; j++) {
	    mfn = hxen_vmmem_get_mfn(devext->de_maps[i].kmap, j);
	    ASSERT(mfn < hxen_info->ki_max_page);
	    devext->de_vm_info.vi_shared.vi_p2m
		[(i << KXEN_VMAPPING_SHIFT) + j] = mfn;
	    vmaptable[mfn].vaddr = (kvaddr + (j << PAGE_SHIFT));
	}
    }

    devext->de_vm_info.vi_shared.vi_nrpages = devext->de_nrpages;

  out:
    if (ret)
	hxen_free_vmappings(devext);
    return ret;
}

void
hxen_free_vmappings(struct hxen_device *devext)
{
    int i;

    if (devext->de_maps) {
	for (i = 0; i < devext->de_nrmaps; i++) {
	    if (devext->de_maps[i].vmmem == NULL)
		break;
	    if (devext->de_maps[i].kmap)
		hxen_vmmem_unmap(devext->de_maps[i].kmap);
	    if (devext->de_maps[i].umap)
		hxen_vmmem_unmap(devext->de_maps[i].umap);
	    hxen_vmmem_free(devext->de_maps[i].vmmem);
	}
	kernel_free(devext->de_maps, devext->de_nrmaps *
		    sizeof(struct hxen_vmmem));
	devext->de_maps = NULL;
    }

    if (devext->de_vm_info.vi_shared.vi_p2m)
	kernel_free(devext->de_vm_info.vi_shared.vi_p2m,
		    devext->de_nrpages * sizeof(uint32_t));
    devext->de_vm_info.vi_shared.vi_p2m = NULL;

    devext->de_nrpages = 0;
}

void * __cdecl
hxen_map_pages(unsigned int num, /* xen_pfn_t * */ void *mfns)
{

    return hxen_vmmem_mmap_pages(kernel_task, num, mfns, 0);
}

int __cdecl
hxen_unmap_pages(unsigned int num, void *va)
{

    return hxen_vmmem_munmap(va);
}
