/*
 *  hxen_vmmem.cpp
 *  hxen
 *
 *  Copyright 2009 Citrix Systems, Inc. All rights reserved.
 *
 */

extern "C" {
#include "hxen.h"
#include "tree.h"
#include <sys/errno.h>
}

#include <IOKit/IOBufferMemoryDescriptor.h>

extern "C" void *
hxen_vmmem_alloc(size_t npages, uint32_t max_page)
{
    IOBufferMemoryDescriptor *m;
    IOReturn ioret;
    uint64_t mask;

    if ((max_page - 1) & max_page) {
	fail_msg("hxen_vmmem_alloc max_page");
	return NULL;
    }
    mask = ((uint64_t)max_page << PAGE_SHIFT) - 1;

    m = IOBufferMemoryDescriptor::inTaskWithPhysicalMask(
	kernel_task, kIOMemoryKernelUserShared | kIODirectionInOut,
	npages << PAGE_SHIFT, mask);
    if (m == NULL) {
	fail_msg("hxen_vmmem_alloc inTaskWithPhysicalMask failed");
	return NULL;
    }

    dprintk("alloc %lx => %lx %lx\n", npages << PAGE_SHIFT, m, m->getLength());
    ioret = m->prepare(kIODirectionInOut);
    if (ioret != kIOReturnSuccess) {
	fail_msg("hxen_vmmem_alloc prepare failed");
	goto error;
    }

    return m;

error:
    m->release();
    return NULL;
}

extern "C" void
hxen_vmmem_free(void *_m)
{
    IOBufferMemoryDescriptor *m = (IOBufferMemoryDescriptor *)_m;

    m->complete();
    m->release();
}

extern "C" void *
hxen_vmmem_map(void *_m, task_t task, uint8_t **vaddr)
{
    IOMemoryDescriptor *m = (IOMemoryDescriptor *)_m;
    IOMemoryMap *map;

    map = m->createMappingInTask(task, 0, kIOMapAnywhere | kIOMapDefaultCache,
				 0, 0);
    if (map == NULL) {
	fail_msg("hxen_vmmem_map createMappingInTask failed");
	return NULL;
    }

    *vaddr = (uint8_t *)map->getVirtualAddress();
    dprintk("hxen_vmmem_map %p at %p\n", m, *vaddr);

    return map;
}

extern "C" void
hxen_vmmem_unmap(void *_map)
{
    IOMemoryMap *map = (IOMemoryMap *)_map;

    map->release();
}

extern "C" uintptr_t
hxen_vmmem_get_mfn(void *_map, uint32_t pfn)
{
    IOMemoryMap *map = (IOMemoryMap *)_map;

    return map->getPhysicalSegment(pfn << PAGE_SHIFT, NULL) >> PAGE_SHIFT;
}

struct hxen_vmmem_mmap {
    uint8_t *vaddr;
    IOMemoryDescriptor *desc;
    IOMemoryMap *map;
    IOPhysicalRange *ranges;
    uint32_t num_ranges;
    SPLAY_ENTRY(hxen_vmmem_mmap) tree;
};

static inline int
hxen_vmmem_mmap_compare(struct hxen_vmmem_mmap *a, struct hxen_vmmem_mmap *b)
{
    return b->vaddr - a->vaddr;
}

SPLAY_HEAD(hxen_vmmem_mmap_tree, hxen_vmmem_mmap) hxen_vmmem_mmap_root =
    SPLAY_INITIALIZER(ignored);
SPLAY_PROTOTYPE(hxen_vmmem_mmap_tree, hxen_vmmem_mmap, tree,
		hxen_vmmem_mmap_compare);
SPLAY_GENERATE(hxen_vmmem_mmap_tree, hxen_vmmem_mmap, tree,
	       hxen_vmmem_mmap_compare);

extern "C" void *
hxen_vmmem_mmap_pages(task_t task, unsigned int num, xen_pfn_t *arr,
		      int from_user)
{
    IOReturn ioret;
    struct hxen_vmmem_mmap *vmmem;
    xen_pfn_t pfn;
    int i, ret;

    vmmem = (struct hxen_vmmem_mmap *)
	kernel_malloc(sizeof(struct hxen_vmmem_mmap));
    if (vmmem == NULL) {
	fail_msg("hxen_vmmem_mmap_pages kernel_malloc vmmem");
	return NULL;
    }
    memset(vmmem, 0, sizeof(struct hxen_vmmem_mmap));

    vmmem->num_ranges = num;
    vmmem->ranges = (IOPhysicalRange *)
	kernel_malloc(num * sizeof(struct IOPhysicalRange));
    if (vmmem->ranges == NULL) {
	fail_msg("hxen_vmmem_mmap_pages kernel_malloc ranges");
	goto error;
    }
    for (i = 0; i < num; i++) {
	if (from_user) {
	    ret = copyin((user_addr_t)(uintptr_t)&arr[i], &pfn, sizeof(pfn));
	    if (ret != 0)
		break;
	} else
	    pfn = arr[i];
	vmmem->ranges[i].address = pfn << PAGE_SHIFT;
	vmmem->ranges[i].length = 1 << PAGE_SHIFT;
    }

    vmmem->desc = IOMemoryDescriptor::withPhysicalRanges
	(vmmem->ranges, num, kIODirectionInOut, true);
    if (vmmem->desc == NULL) {
	fail_msg("hxen_vmmem_mmap_pages withPhysicalRanges failed\n");
	goto error;
    }

    ioret = vmmem->desc->prepare(kIODirectionInOut);
    if (ioret != kIOReturnSuccess) {
	fail_msg("hxen_vmmem_mmap_pages prepare failed");
	goto error2;
    }

    vmmem->map = (IOMemoryMap *)hxen_vmmem_map(vmmem->desc, task,
					       &vmmem->vaddr);
    if (vmmem->map == NULL) {
	fail_msg("hxen_vmmem_mmap_pages hxen_vmmem_map");
	goto error3;
    }

    SPLAY_INSERT(hxen_vmmem_mmap_tree, &hxen_vmmem_mmap_root, vmmem);

    return vmmem->vaddr;

  error3:
    vmmem->desc->complete();
  error2:
    vmmem->desc->release();
  error:
    if (vmmem->ranges)
	kernel_free(vmmem->ranges, num * sizeof(struct IOPhysicalRange));
    kernel_free(vmmem, sizeof(struct hxen_vmmem));
    return NULL;
}

extern "C" int
hxen_vmmem_munmap(uint8_t *vaddr)
{
    struct hxen_vmmem_mmap *vmmem, vmmem_key;

    vmmem_key.vaddr = vaddr;
    vmmem = SPLAY_FIND(hxen_vmmem_mmap_tree, &hxen_vmmem_mmap_root,
		       &vmmem_key);
    if (vmmem == NULL) {
	fail_msg("hxen_vmmem_munmap mapping %p not found\n", vaddr);
	return ENOENT;
    }

    SPLAY_REMOVE(hxen_vmmem_mmap_tree, &hxen_vmmem_mmap_root, vmmem);

    vmmem->map->release();
    vmmem->desc->complete();
    vmmem->desc->release();
    kernel_free(vmmem->ranges, vmmem->num_ranges *
		sizeof(struct IOPhysicalRange));
    kernel_free(vmmem, sizeof(struct hxen_vmmem));
    return 0;
}
