/* 
 *  block-vhd.c
 *  hxen
 *
 *  Copyright 2009 Citrix Systems, Inc. All rights reserved.
 * 
 */

/*
 * Todo:
 * - cancel: interrupt in progress requests
 * - td_prep_{read,write}: on error, add bh to run cb with failure code
 * - aio read/write win32: handle case where there's no wait objects
 *   + qemu_add_wait_object failure needs to be propagated
 *   + td_prep_{read,write} should put tiocb on queue
 *   + try to re-issue from cb or add bh for re-issue
 * - aio flush: implement
 *   + queue incoming requests
 *   + put callback into a bh
 *   + schedule bh when in_progress goes to 0
 */
#include "qemu-common.h"
#include "block_int.h"
#include "console.h"
#include "sysemu.h"

#include <libvhd.h>
#include <mm_malloc.h>

#include <tapdisk.h>
#include <tapdisk-driver.h>
#include <tapdisk-log.h>

#define dprintf(fmt, args...)
// #define dprintf(fmt, args...) term_printf(fmt, ##args)
#define aprintf(fmt, args...) term_printf(fmt, ##args)
#define eprintf(fmt, args...) do {		\
	term_printf(fmt, ##args);		\
	fprintf(stderr, fmt, ##args);		\
    } while (0)

// #define TRACK_TRUE_DATA
// #define VHD_TLOG
// #define VHD_NO_AIO

#define SECTOR_SHIFT 9

typedef struct BDRVVhdState {
#ifdef VHD_NO_AIO
    vhd_context_t ctx;
#else
    td_driver_t td_driver;
    BlockDriverState parent_bs;
    int has_parent;
#endif
    int in_progress;
} BDRVVhdState;

struct BlockDriverAIOCB_vhd {
    BlockDriverAIOCB common;
    QEMUBH *bh;
    int ret;
    int nsecs_remaining;
    int cancelled;
};

typedef struct BlockDriverAIOCB_vhd BDRVVhdAIOCB;

#ifdef TRACK_TRUE_DATA
static char **true_data;
#endif

struct libvhd_stub_handle {
    BlockDriverState *hd;
    off64_t pos;
    off64_t size;
};

static struct libvhd_stub_handle *h = NULL;
static int nrh = 0;
static int libvhd_stub_setup = 0;

static int
bad_handle(int fd)
{
    return fd >= nrh || h[fd].hd == NULL;
}

#define O_DIRECT 0

static int
libvhd_open_stub(const char *pathname, int flags, ...)
{
    int fd, ret;
    int bdrv_flags = BDRV_O_EXTENDABLE;

    if (flags & O_CREAT) {
	fprintf(stderr, "libvhd_open_stub O_CREAT not supported");
	errno = EINVAL;
	return -1;
    }

    if (flags & O_RDONLY)
	bdrv_flags |= BDRV_O_RDONLY;
    else
	bdrv_flags |= BDRV_O_RDWR;

    if (flags & O_DIRECT)
	bdrv_flags |= BDRV_O_DIRECT;

    for (fd = 0; fd < nrh; fd++)
	if (h[fd].hd == NULL)
	    break;

    if (fd == nrh) {
	struct libvhd_stub_handle *nh = h;
	int nnrh = 1 + nrh * 2;
	nh = realloc(h, nnrh * sizeof(struct libvhd_stub_handle));
	if (nh == NULL) {
	    fprintf(stderr, "Could not realloc BlockDriverState array");
	    return -1;
	}
	h = nh;
	while (nrh < nnrh) {
	    h[nrh].hd = NULL;
	    nrh++;
	}
    }

    ret = bdrv_file_open(&h[fd].hd, pathname, bdrv_flags);
    if (ret < 0) {
	fprintf(stderr, "bdrv_file_open %s failed\n", pathname);
	h[fd].hd = NULL;
	errno = -ret;
	return -1;
    }
    h[fd].pos = 0;
    h[fd].size = (off64_t)h[fd].hd->total_sectors << SECTOR_SHIFT;

    return fd;
}

static int
libvhd_close_stub(int fd)
{
    if (bad_handle(fd)) {
	errno = EBADF;
	return -1;
    }
    bdrv_delete(h[fd].hd);
    h[fd].hd = NULL;
    return 0;
}

static read_return_t
libvhd_read_stub(int fd, void *buf, size_t count)
{
    int ret;

    if (bad_handle(fd)) {
	errno = EBADF;
	return -1;
    }
    if (count & ((1 << SECTOR_SHIFT) - 1)) {
	eprintf("libvhd_read_stub count %"FMT_SIZE"x not multiple of "
		"sector size\n", count);
	errno = EINVAL;
	return -1;
    }
    if ((h[fd].pos + count) > h[fd].size) {
	eprintf("libvhd_read_stub read past end\n");
	errno = EINVAL;
	return -1;
    }
    ret = bdrv_read(h[fd].hd, h[fd].pos >> SECTOR_SHIFT, buf,
		    count >> SECTOR_SHIFT);
    if (ret) {
	eprintf("libvhd_read_stub failed %"FMT_SIZE"x@%"PRIx64"\n", count,
		h[fd].pos);
	errno = EIO;
	return -1;
    }

    h[fd].pos += count;
    return count;
}

static write_return_t
libvhd_write_stub(int fd, const void *buf, size_t count)
{
    int ret;

    if (bad_handle(fd)) {
	errno = EBADF;
	return -1;
    }
    if (count & ((1 << SECTOR_SHIFT) - 1)) {
	eprintf("libvhd_write_stub count %"FMT_SIZE"x not multiple of "
		"sector size\n", count);
	errno = EINVAL;
	return -1;
    }
    if (h[fd].pos > h[fd].size) {
	static char *zbuf = NULL;
	static int zbufsize = 4 * 4096;
	int zcount;
	if (zbuf == NULL) {
	    zbuf = _mm_malloc(zbufsize, 4096);
	    if (zbuf == NULL) {
		errno = EINVAL;
		return -1;
	    }
	    memset(zbuf, 0, zbufsize);
	}
	while (h[fd].pos > h[fd].size) {
	    zcount = h[fd].pos - h[fd].size;
	    if (zcount > zbufsize)
		zcount = zbufsize;
	    ret = bdrv_write(h[fd].hd, h[fd].size >> SECTOR_SHIFT,
			     (void *)zbuf, zcount >> SECTOR_SHIFT);
	    if (ret) {
		eprintf("libvhd_write_stub extend failed %x@%"PRIx64" -> %d\n",
			zcount, h[fd].pos, ret);
		errno = EIO;
		return -1;
	    }
	    h[fd].size += zcount;
	}
    }
    ret = bdrv_write(h[fd].hd, h[fd].pos >> SECTOR_SHIFT, buf,
		     count >> SECTOR_SHIFT);
    if (ret) {
	eprintf("libvhd_write_stub failed %"FMT_SIZE"x@%"PRIx64" -> %d\n",
		count, h[fd].pos, ret);
	errno = EIO;
	return -1;
    }

    h[fd].pos += count;
    if (h[fd].pos > h[fd].size)
	h[fd].size = h[fd].pos;
    return count;
}

static off64_t
libvhd_lseek64_stub(int fd, off64_t offset, int whence)
{

    if (bad_handle(fd)) {
	errno = EBADF;
	return -1;
    }

    switch (whence) {
    case SEEK_SET:
	h[fd].pos = offset;
	break;
    case SEEK_CUR:
	h[fd].pos += offset;
	break;
    case SEEK_END:
	h[fd].pos = h[fd].size + offset;
	break;
    default:
	errno = EINVAL;
	return -1;
	break;
    }
    return h[fd].pos;
}

static int
libvhd_fsync_stub(int fd)
{
    int ret;

    if (bad_handle(fd)) {
	errno = EBADF;
	return -1;
    }

    ret = bdrv_flush(h[fd].hd);
    if (ret) {
	errno = -ret;
	ret = -1;
    }
    return ret;
}

#ifndef VHD_NO_AIO
typedef void (*libvhd_aio_stub_cb)(void *, int);

static BlockDriverAIOCB *
libvhd_aio_read_stub(int fd, uint8_t *buf, off64_t offset, size_t bytes,
		      libvhd_aio_stub_cb cb, void *arg)
{
    BlockDriverAIOCB *aiocb;

    if (bad_handle(fd)) {
	errno = EBADF;
	return NULL;
    }
    aiocb = bdrv_aio_read(h[fd].hd, offset >> SECTOR_SHIFT, buf,
			   bytes >> SECTOR_SHIFT, cb, arg);
    dprintf("issued aio read for %x@%"PRIx64" => %p\n",
	    bytes >> SECTOR_SHIFT, offset >> SECTOR_SHIFT, aiocb);
    return aiocb;
}

static BlockDriverAIOCB *
libvhd_aio_write_stub(int fd, const uint8_t *buf, off64_t offset, size_t bytes,
		      libvhd_aio_stub_cb cb, void *arg)
{
    BlockDriverAIOCB *aiocb;

    if (bad_handle(fd)) {
	errno = EBADF;
	return NULL;
    }
    aiocb = bdrv_aio_write(h[fd].hd, offset >> SECTOR_SHIFT, buf,
			   bytes >> SECTOR_SHIFT, cb, arg);
    dprintf("issued aio write for %x@%"PRIx64" => %p\n",
	    bytes >> SECTOR_SHIFT, offset >> SECTOR_SHIFT, aiocb);
    return aiocb;
}
#endif

static int bdrv_vhd_probe(const uint8_t *buf, int buf_size,
			  const char *filename)
{
    eprintf("bdrv_vhd_probe %s\n", filename);

    return -100;
}

#ifdef VHD_NO_AIO
static int bdrv_vhd_open(BlockDriverState *bs, const char *filename, int flags)
{
    BDRVVhdState *s = bs->opaque;
    int ret;

    dprintf("bdrv_vhd_open %s\n", filename);

    if (!libvhd_stub_setup) {
	vhd_set_fops(libvhd_open_stub, libvhd_close_stub,
		     libvhd_read_stub, libvhd_write_stub, libvhd_lseek64_stub);
	libvhd_stub_setup = 1;
#ifdef TRACK_TRUE_DATA
	true_data = calloc(1, 0x1000000 * sizeof(char *));
	if (true_data == NULL)
	    return -1;
#endif
    }

    if (!strncmp(filename, "vhd:", 4))
	filename = &filename[4];

    ret = vhd_open(&s->ctx, filename, VHD_OPEN_RDWR);
    if (ret < 0)
	return ret;

    bs->total_sectors = s->ctx.footer.curr_size >> SECTOR_SHIFT;

    return 0;
}

static int bdrv_vhd_read(BlockDriverState *bs, int64_t sector_num,
                    uint8_t *buf, int nb_sectors)
{
    BDRVVhdState *s = bs->opaque;
    int ret;

    ret = vhd_io_read(&s->ctx, buf, sector_num, nb_sectors);
#ifdef TRACK_TRUE_DATA
    if (ret == 0) {
	int i;
	for (i = sector_num; i < sector_num + nb_sectors; i++) {
	    int failed;
	    if (true_data[i] == NULL) {
		int j;
		failed = 0;
		for (j = 0; j < (1 << SECTOR_SHIFT); j++)
		    if (buf[((i - sector_num) << SECTOR_SHIFT) + j]) {
			failed = 1;
			break;
		    }
	    } else {
		failed = memcmp(true_data[i],
				&buf[(i - sector_num) << SECTOR_SHIFT],
				1 << SECTOR_SHIFT);
	    }
	    if (failed) {
		eprintf("vhd read verify failed for sector %"PRIx64" td %p\n",
			sector_num, true_data[i]);
		asm ("int3");
	    }
	}
    }
#endif
    dprintf("bdrv_vhd_read %x@%"PRIx64" => %d\n", nb_sectors, sector_num, ret);
    return ret;
}

static int bdrv_vhd_write(BlockDriverState *bs, int64_t sector_num,
                     const uint8_t *buf, int nb_sectors)
{
    BDRVVhdState *s = bs->opaque;
    int ret;

#ifdef TRACK_TRUE_DATA
    {
	int i;
	for (i = sector_num; i < sector_num + nb_sectors; i++) {
	    if (true_data[i] == NULL) {
		true_data[i] = malloc(1 << SECTOR_SHIFT);
		if (true_data[i] == NULL)
		    return -errno;
	    }
	    memcpy(true_data[i], &buf[(i - sector_num) << SECTOR_SHIFT],
		   1 << SECTOR_SHIFT);
	}
    }
#endif
    ret = vhd_io_write(&s->ctx, buf, sector_num, nb_sectors);
    dprintf("bdrv_vhd_write %x@%"PRIx64" => %d\n", nb_sectors, sector_num, ret);
    return ret;
}
#endif

static int bdrv_vhd_create(const char *filename, int64_t total_size,
		      const char *backing_file, int flags)
{
    dprintf("bdrv_vhd_create %s\n", filename);

    return -1;
}

#ifdef VHD_NO_AIO
static void bdrv_vhd_close(BlockDriverState *bs)
{
    BDRVVhdState *s = bs->opaque;

    dprintf("bdrv_vhd_close\n");

    vhd_close(&s->ctx);
}
#endif

static int bdrv_vhd_is_allocated(BlockDriverState *bs, int64_t sector_num,
			    int nb_sectors, int *pnum)
{
    dprintf("bdrv_vhd_is_allocated\n");

    return 1;
}

#ifdef VHD_NO_AIO
static int bdrv_vhd_flush(BlockDriverState *bs)
{
    BDRVVhdState *s = bs->opaque;

    dprintf("bdrv_vhd_flush\n");

    return libvhd_fsync_stub(s->ctx.fd);
}
#endif

#ifndef VHD_NO_AIO
static int bdrv_vhd_aio_open(BlockDriverState *bs, const char *filename,
			     int flags)
{
    BDRVVhdState *s = bs->opaque;
    td_disk_id_t parent_id;
    int ret;

    dprintf("bdrv_vhd_aio_open %s\n", filename);

    if (!libvhd_stub_setup) {
	vhd_set_fops(libvhd_open_stub, libvhd_close_stub,
		     libvhd_read_stub, libvhd_write_stub, libvhd_lseek64_stub);
	libvhd_stub_setup = 1;
#ifdef TRACK_TRUE_DATA
	true_data = calloc(1, 0x1000000 * sizeof(char *));
	if (true_data == NULL)
	    return -1;
#endif
#ifdef VHD_TLOG
	open_tlog("tapdisk.%d.log", (64 << 10), TLOG_DBG, 1);
	atexit(tlog_flush);
#endif
    }

    if (!strncmp(filename, "vhd:", 4))
	filename = &filename[4];

    ret = _vhd_open(&s->td_driver, filename, 0/* flags */);
    if (ret < 0)
	return ret;

    bs->total_sectors = s->td_driver.info.size;

    s->in_progress = 0;

    ret = vhd_get_parent_id(&s->td_driver, &parent_id);
    if (ret == 0) {
	char *parent;

	if (parent_id.drivertype == DISK_TYPE_VHD) {
	    ret = asprintf(&parent, "vhd:%s", parent_id.name);
	    if (ret == -1) {
		_vhd_close(&s->td_driver);
		return -ENOMEM;
	    }
	} else
	    parent = parent_id.name;

	flags &= ~BDRV_O_RDWR;
	ret = bdrv_open(&s->parent_bs, parent, flags | BDRV_O_RDONLY);
	if (ret < 0)
	    _vhd_close(&s->td_driver);

	s->has_parent = 1;

	if (parent_id.drivertype == DISK_TYPE_VHD)
	    free(parent);
    }

    return ret;
}

static void bdrv_vhd_aio_close(BlockDriverState *bs)
{
    BDRVVhdState *s = bs->opaque;

    dprintf("bdrv_vhd_aio_close\n");

    _vhd_close(&s->td_driver);
    if (s->has_parent)
	bdrv_delete(&s->parent_bs);
#ifdef VHD_TLOG
    close_tlog();
#endif
}

static int bdrv_vhd_flush(BlockDriverState *bs)
{
    BDRVVhdState *s = bs->opaque;

    dprintf("bdrv_vhd_flush\n");

#ifdef VHD_TLOG
    tlog_flush();
#endif

    qemu_aio_wait_start();
    qemu_aio_poll();
    while (s->in_progress)
	qemu_aio_wait();
    qemu_aio_wait_end();

    return libvhd_fsync_stub(vhd_fd(&s->td_driver));
}

/* XXX per fd */
struct queued_treq {
    struct queued_treq *next;
    td_request_t treq;
};
struct queued_treq *unused_queued_treqs = NULL;
struct queued_treq *queued_treqs = NULL;
struct queued_treq **queued_treqs_tail = &queued_treqs;

static int bdrv_vhd_aio_queue_blocked(td_request_t *treq)
{
    struct queued_treq *qt;
    BDRVVhdAIOCB *acb = (BDRVVhdAIOCB *)treq->cb_data;

    aprintf("bdrv_vhd_aio_queue_blocked %p: %x of %x\n", acb, treq->secs,
	    acb->nsecs_remaining);

    qt = unused_queued_treqs;
    if (qt)
	unused_queued_treqs = qt->next;
    else {
	qt = malloc(sizeof(struct queued_treq));
	if (qt == NULL)
	    return -EIO;
	memset(qt, 0, sizeof(struct queued_treq));
    }

    qt->treq = *treq;

    *queued_treqs_tail = qt;
    queued_treqs_tail = &qt->next;

    return -EBUSY;
}

static void bdrv_vhd_aio_queue_reissue(void)
{
    struct queued_treq *qt;

    if (queued_treqs == NULL)
	return;

    while (queued_treqs) {
	td_driver_t *driver;
	BDRVVhdAIOCB *acb;

	qt = queued_treqs;
	queued_treqs = queued_treqs->next;

	driver = qt->treq.private;

	acb = (BDRVVhdAIOCB *)qt->treq.cb_data;
	aprintf("bdrv_vhd_aio_queue_reissue %p: %x of %x\n", acb,
		qt->treq.secs, acb->nsecs_remaining);

	switch (qt->treq.op) {
	case TD_OP_READ:
	    vhd_queue_read(driver, qt->treq);
	    break;
	case TD_OP_WRITE:
	    vhd_queue_write(driver, qt->treq);
	    break;
	}

	qt->next = unused_queued_treqs;
	unused_queued_treqs = qt;
    }
    queued_treqs_tail = &queued_treqs;
}

static void bdrv_vhd_aio_cb(td_request_t treq, int res)
{
    BDRVVhdAIOCB *acb = (BDRVVhdAIOCB *)treq.cb_data;

    if (res && res != -EBUSY)
	eprintf("bdrv_vhd_aio_cb %p failed: %x of %x: res %d\n", acb,
		treq.secs, acb->nsecs_remaining, -res);

    dprintf("bdrv_vhd_aio_cb %p: %x of %x: res %d\n", acb, treq.secs,
	    acb->nsecs_remaining, -res);
    if (acb->cancelled)
	res = 0;
    if (res == 0)
	acb->nsecs_remaining -= treq.secs;
    if (res == -EBUSY)
	res = bdrv_vhd_aio_queue_blocked(&treq);

    if (acb->nsecs_remaining == 0 || (res && res != -EBUSY)) {
	if (acb->cancelled == 0)
	    acb->common.cb(acb->common.opaque, -res);
	qemu_aio_release(acb);
	((BDRVVhdState *)acb->common.bs->opaque)->in_progress--;
    }

    if (queued_treqs != NULL && res != -EBUSY)
	bdrv_vhd_aio_queue_reissue();
}

static BDRVVhdAIOCB *
bdrv_vhd_aio_setup(BlockDriverState *bs, td_driver_t *driver,
		   td_request_t *treq, int64_t sector_num, uint8_t *buf,
		   int nb_sectors, BlockDriverCompletionFunc *cb, void *opaque)
{
    BDRVVhdAIOCB *acb;
    static int id = 0;

    acb = qemu_aio_get(bs, cb, opaque);
    if (acb == NULL)
	return NULL;
    memset(treq, 0, sizeof(*treq));
    treq->id = id++;
    treq->sidx = 0;
    treq->blocked = 0;
    treq->buf = (char *)buf;
    treq->sec = sector_num;
    treq->secs = nb_sectors;
    treq->image = NULL/* image */;
    treq->cb = bdrv_vhd_aio_cb;
    treq->cb_data = acb;
    treq->private = driver;

    acb->nsecs_remaining = nb_sectors;

    return acb;
}

static BlockDriverAIOCB *
bdrv_vhd_aio_read(BlockDriverState *bs,
		  int64_t sector_num, uint8_t *buf, int nb_sectors,
		  BlockDriverCompletionFunc *cb, void *opaque)
{
    BDRVVhdState *s = bs->opaque;
    BDRVVhdAIOCB *acb;
    td_request_t treq;

    acb = bdrv_vhd_aio_setup(bs, &s->td_driver, &treq, sector_num, buf,
			     nb_sectors, cb, opaque);
    if (acb == NULL)
	goto out;
    treq.op = TD_OP_READ;
    vhd_queue_read(&s->td_driver, treq);
    s->in_progress++;
  out:
    dprintf("bdrv_vhd_aio_read %x@%"PRIx64" => %p\n", nb_sectors, sector_num,
	    acb);
    return (BlockDriverAIOCB *)acb;
}

static BlockDriverAIOCB *bdrv_vhd_aio_write(BlockDriverState *bs,
        int64_t sector_num, const uint8_t *buf, int nb_sectors,
        BlockDriverCompletionFunc *cb, void *opaque)
{
    BDRVVhdState *s = bs->opaque;
    BDRVVhdAIOCB *acb;
    td_request_t treq;

    acb = bdrv_vhd_aio_setup(bs, &s->td_driver, &treq, sector_num,
			     (uint8_t *)buf, nb_sectors, cb, opaque);
    if (acb == NULL)
	goto out;
    treq.op = TD_OP_WRITE;
    vhd_queue_write(&s->td_driver, treq);
    s->in_progress++;
  out:
    dprintf("bdrv_vhd_aio_write %x@%"PRIx64" => %p\n", nb_sectors, sector_num,
	    acb);
    return (BlockDriverAIOCB *)acb;
}

static void bdrv_vhd_aio_cancel(BlockDriverAIOCB *_acb)
{
    BDRVVhdAIOCB *acb = (BDRVVhdAIOCB *)_acb;

    eprintf("bdrv_vhd_aio_cancel %p\n", acb);
    acb->cancelled = 1;
    bdrv_vhd_flush(acb->common.bs);
}

#include "tapdisk-interface.h"

static void
td_forward_request_cb(void *opaque, int ret)
{
    td_request_t *treq = (td_request_t *)opaque;

    td_complete_request(*treq, ret);
    free(treq);
}

void
td_forward_request(td_request_t treq)
{
    BDRVVhdAIOCB *acb = (BDRVVhdAIOCB *)treq.cb_data;
    BDRVVhdState *s = (BDRVVhdState *)acb->common.bs->opaque;
    td_request_t *ptreq;

    if (s->has_parent == 0) {
	memset(treq.buf, 0, treq.secs << SECTOR_SHIFT);
	td_complete_request(treq, 0);
    } else {
	BlockDriverAIOCB *aiocb;

	if (treq.op != TD_OP_READ) {
	    eprintf("td_forward_request: non-read request\n");
	    td_complete_request(treq, -EINVAL);
	    return;
	}
	ptreq = malloc(sizeof(td_request_t));
	if (ptreq == NULL) {
	    eprintf("td_forward_request: out of memory\n");
	    td_complete_request(treq, -ENOMEM);
	    return;
	}
	*ptreq = treq;
	aiocb = bdrv_aio_read(&s->parent_bs, treq.sec,
			      (unsigned char *)treq.buf, treq.secs,
			      td_forward_request_cb, ptreq);
	if (aiocb == NULL) {
	    eprintf("td_forward_request: bdrv_aio_read failed\n");
	    td_complete_request(treq, -ENOMEM);
	    return;
	}
    }
}

void
td_complete_request(td_request_t treq, int res)
{
    // asm("int3");
    ((td_callback_t)treq.cb)(treq, res);
}

void
td_queue_tiocb(td_driver_t *driver, struct tiocb *tiocb)
{
}

static void
td_tiocb_complete_cb(void *opaque, int ret)
{
    struct tiocb *tiocb = opaque;

    tiocb->cb(tiocb->arg, tiocb, ret);
}

void
td_prep_read(struct tiocb *tiocb, int fd, char *buf, size_t bytes,
	     long long offset, td_queue_callback_t cb, void *arg)
{
 
    // asm("int3");
    tiocb->cb = cb;
    tiocb->arg = arg;

    tiocb->opaque = libvhd_aio_read_stub(fd, (unsigned char *)buf, offset,
					 bytes, td_tiocb_complete_cb, tiocb);
    if (tiocb->opaque == NULL) {
	/* Add bh to run cb with failure code */
	eprintf("td_prep_read: libvhd_aio_read_stub failed\n");
    }
}

void
td_prep_write(struct tiocb *tiocb, int fd, char *buf, size_t bytes,
	      long long offset, td_queue_callback_t cb, void *arg)
{

    // asm("int3");
    tiocb->cb = cb;
    tiocb->arg = arg;

    tiocb->opaque = libvhd_aio_write_stub(fd, (unsigned char *)buf, offset,
					  bytes, td_tiocb_complete_cb, tiocb);
    if (tiocb->opaque == NULL) {
	/* Add bh to run cb with failure code */
	eprintf("td_prep_write: libvhd_aio_write_stub failed\n");
    }
}
#endif

#ifdef VHD_NO_AIO
BlockDriver bdrv_vhd = {
    "vhd",
    sizeof(BDRVVhdState),
    bdrv_vhd_probe,
    bdrv_vhd_open,
    bdrv_vhd_read,
    bdrv_vhd_write,
    bdrv_vhd_close,
    bdrv_vhd_create,
    bdrv_vhd_flush,
    bdrv_vhd_is_allocated,
    .protocol_name = "vhd",
};
#else
BlockDriver bdrv_vhd = {
    "vhd",
    sizeof(BDRVVhdState),
    bdrv_vhd_probe,
    bdrv_vhd_aio_open,
    NULL,
    NULL,
    bdrv_vhd_aio_close,
    bdrv_vhd_create,
    bdrv_vhd_flush,
    bdrv_vhd_is_allocated,

    .bdrv_aio_read = bdrv_vhd_aio_read,
    .bdrv_aio_write = bdrv_vhd_aio_write,
    .bdrv_aio_cancel = bdrv_vhd_aio_cancel,
    .aiocb_size = sizeof(BDRVVhdAIOCB),

    .protocol_name = "vhd",
};
#endif
