/* Stupid driver to make it easy to get kernel crashdumps out of
   Windows guests */
#include <sys/types.h>
#include <stdint.h>
#include <dirent.h>
#include <limits.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>
#include "cpu.h"
#include "exec-all.h"

#include "hw.h"
#include "pci.h"
#include "qemu-xen.h"

static long long avail_quota;
static int crashdump_fd = -1;
static char *crashdump_path;
static int have_crashdump;

static void write_to_control_port(void *ign, uint32_t ign2, uint32_t data)
{
    have_crashdump = 1;
}

static void write_to_data_port(void *ign, uint32_t ign2, uint32_t data)
{
    char buf[TARGET_PAGE_SIZE];
    target_phys_addr_t addr = data;
    int cntr, r;

    if (avail_quota < 0) {
	fprintf(logfile, "out of quota writing crash dump.\n");
	return;
    }
    avail_quota -= TARGET_PAGE_SIZE;
    addr *= TARGET_PAGE_SIZE;
    cpu_physical_memory_rw(addr, (uint8_t *)buf, TARGET_PAGE_SIZE, 0);
    for (cntr = 0; cntr < TARGET_PAGE_SIZE; cntr += r) {
	r = write(crashdump_fd, buf, TARGET_PAGE_SIZE);
	if (r < 0 || r == 0) {
	    fprintf(logfile, "failed to write to dump file (%d, %s)\n",
		    r, strerror(errno));
	    break;
	}
    }
}

static void tidy_up(void)
{
    fprintf(logfile, "Running cleanup.\n");
    if (!have_crashdump)
	unlink(crashdump_path);
}

void setup_crashdump(const char *crashdump_dir, const char *crashdump_quota)
{
    DIR *d;
    int e;
    struct dirent *de;
    int nr_files;
    char *path;
    int fd = 0;

    if (crashdump_quota)
	avail_quota = strtoll(crashdump_quota, NULL, 0) * 1024 * 1024;
    else
	avail_quota = LONG_LONG_MAX;
    d = opendir(crashdump_dir);
    if (!d) {
	fprintf(logfile, "cannot open %s: %s\n", crashdump_dir,
		strerror(errno));
	exit(1);
    }

    nr_files = 0;
    while ((de = readdir(d))) {
	struct stat sb;

	if (!strcmp(de->d_name, ".") || !strcmp(de->d_name, ".."))
	    continue;

	if (asprintf(&path, "%s/%s", crashdump_dir, de->d_name) == -1) {
	    fprintf(logfile, "cannot allocate memory\n");
	    exit(1);
	}
	if (stat(path, &sb) < 0) {
	    fprintf(logfile, "cannot stat %s: %s\n", path, strerror(errno));
	    exit(1);
	}
	if (sb.st_size == 0) {
	    /* XXX While we're here, unlink any empty dump files.  The
	       atexit hook doesn't always run, since xend sometimes
	       SIGKILLs qemu rather than letting it exit normally, and
	       so this is pretty much the best we can do. */
	    unlink(path);
	} else {
	    avail_quota -= sb.st_size;
	    nr_files++;
	}
	free(path);
    }
    /* If we run below 64k, give up, since we won't even be able to
       get a minidump out. */
    if (avail_quota < 65536) {
	fprintf(logfile, "out of quota in %s\n", crashdump_dir);
	exit(1);
    }
    closedir(d);

    /* Try to create the dump file now, so as we know that the
       directory is writable etc. */
    for (e = nr_files; e < nr_files * 2 + 1; e++) {
	if (asprintf(&path, "%s/%d", crashdump_dir, e) < 0) {
	    fprintf(logfile, "allocating memory: %s\n",
		    strerror(errno));
	    exit(1);
	}
	fd = open(path, O_WRONLY | O_CREAT | O_EXCL, 0600);
	if (fd >= 0)
	    break;
	if (errno != EEXIST) {
	    fprintf(logfile, "openning %s: %s\n", path, strerror(errno));
	    exit(1);
	}
	free(path);
    }
    if (e == nr_files * 2 + 1) {
	fprintf(logfile, "cannot find a place to dump to in %s\n",
		crashdump_dir);
	exit(1);
    }
    crashdump_path = path;
    crashdump_fd = fd;

    atexit(tidy_up);

    register_ioport_write(0xeb, 1, 1, write_to_control_port, NULL);
    register_ioport_write(0xec, 4, 4, write_to_data_port, NULL);
}
