/*
 *  hxen-macos.c
 *  hxen
 *
 *  Copyright 2009 Citrix Systems, Inc. All rights reserved.
 *
 */

#include <err.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/errno.h>

#include <hxenctllib.h>
#include <hxen_desc.h>
#include <hxen_info.h>

static HANDLE hxen_handle = INVALID_HANDLE_VALUE;

static void
close_and_unload(void)
{
    if (hxen_handle != INVALID_HANDLE_VALUE)
	hxen_close(hxen_handle);
    hxen_handle = INVALID_HANDLE_VALUE;

    (void)hxen_manage_driver(FALSE, FALSE);
}

int
hxen_setup(void)
{
    int ret;

    atexit(close_and_unload);

    hxen_handle = hxen_open(0, TRUE);
    if (hxen_handle == INVALID_HANDLE_VALUE)
	err(1, "hxen_open");

    ret = hxen_load(hxen_handle, "hxen.elf");
    if (ret && ret != EEXIST)
	err(1, "hxen_load");

    ret = hxen_init(hxen_handle);
    if (ret && ret != EEXIST)
	err(1, "hxen_init");

    return 0;
}

void
hxen_exit(void)
{
    (void)hxen_shutdown(hxen_handle);
}

struct hxen_vmapping *
hxen_alloc_mem(size_t guest_size)
{
    int ret;
    struct hxen_vmappings_desc kvd;
    int nr_pages;
    struct hxen_vmapping *vmappings;

    nr_pages = (guest_size + KXEN_PAGE_SIZE - 1) >> KXEN_PAGE_SHIFT;
    vmappings = malloc(sizeof(struct hxen_vmapping) *
		       ((nr_pages + ((1 << KXEN_VMAPPING_SHIFT) - 1)) >>
			KXEN_VMAPPING_SHIFT));
    if (vmappings == NULL)
	return NULL;
    kvd.kvd_nrpages = nr_pages;
    set_xen_guest_handle(kvd.kvd_vmappings, vmappings);
    ret = hxen_vmappings(hxen_handle, &kvd);
    if (ret)
	return NULL;
    return vmappings;
}

int
hxen_run(void)
{
    int ret;

    ret = hxen_execute(hxen_handle);
    return ret;
}

int
hxen_fd(void)
{

    return hxen_handle;
}

int
hxen_xc_hvmop(unsigned cmd, union hxen_hvmop_arg *kha)
{
    int ret;
    struct hxen_hvmop_desc khd;

    khd.khd_cmd = cmd;
    set_xen_guest_handle(khd.khd_kha, kha);

    ret = hxen_hvmop(hxen_handle, &khd);
    return ret;
}

int
hxen_xc_memop(unsigned cmd, union hxen_memop_arg *kma)
{
    int ret;
    struct hxen_memop_desc kmd;

    kmd.kmd_cmd = cmd;
    set_xen_guest_handle(kmd.kmd_kma, kma);

    ret = hxen_memop(hxen_handle, &kmd);
    return ret;
}

int
hxen_xc_domctl(struct xen_domctl *xd)
{
    int ret;
    struct hxen_domctl_desc kdd;

    xd->interface_version = XEN_DOMCTL_INTERFACE_VERSION;
    set_xen_guest_handle(kdd.kdd_xd, xd);

    ret = hxen_domctl(hxen_handle, &kdd);
    return ret;
}

int
hxen_xc_version(int cmd, void *arg)
{
    int ret;
    struct hxen_hypercall_desc khd;

    khd.khd_op = __HYPERVISOR_xen_version;
    khd.khd_arg[0] = cmd;
    khd.khd_arg[1] = (uint64_t)(uintptr_t)arg;

    ret = hxen_hypercall(hxen_handle, &khd);
    return ret;
}

void *
hxen_xc_mmapbatch(xen_pfn_t *pfn, int num, int prot, domid_t dom)
{
    int ret;
    struct hxen_mmapbatch_desc kmd;

    kmd.kmd_num = num;
    kmd.kmd_domid = dom;
    set_xen_guest_handle(kmd.kmd_arr, pfn);
    ret = hxen_mmapbatch(hxen_handle, &kmd);
    if (ret != 0)
	kmd.kmd_addr = 0;
    return (void *)(uintptr_t)kmd.kmd_addr;
}

void
hxen_xc_munmap(void *addr, int num)
{
    struct hxen_munmap_desc kmd;

    kmd.kmd_num = num;
    kmd.kmd_addr = (uint64_t)(uintptr_t)addr;
    (void)hxen_munmap(hxen_handle, &kmd);
}

int
hxen_xc_keyhandler(const char *keys)
{
    int ret;

    ret = hxen_trigger_keyhandler(hxen_handle, keys);
    return ret;
}

#if defined(_WIN32)
int
hxen_ioemu_events(HANDLE requestEvent, HANDLE completedEvent)
{
    int ret;
    struct hxen_ioemu_events_desc kied;

    memset(&kied, 0, sizeof(kied));
    kied.kied_requestEvent = requestEvent;
    kied.kied_completedEvent = completedEvent;
    ret = hxen_set_ioemu_events(hxen_handle, &kied);
    return ret;
}
#endif

void
hxen_ioreq_complete(void)
{

    hxen_set_ioreq_complete(hxen_handle);
}
