
#define _WIN32_WINNT 0x0501	/* XP */

#include <windows.h>
#include <winbase.h>
#include <winnt.h>
#include <winioctl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "tenkei2.h"
#include <tk2_ioctl.h>
#include <xen/errno.h>

#include <xen/types.h>
#include <hxen_info.h>

void destroy_hvm_domain(void);

#define HV_SIZE_MIN 512
#define HV_SIZE_MAX (512*1024*1024)

//
// Globals
//

static HANDLE tk2_handle = INVALID_HANDLE_VALUE;
BOOLEAN ExitFlag = FALSE;

BOOLEAN
LoadBlob(
    PMEMDESC Descriptor
    )
{
    HANDLE blob;
    PVOID buffer;
    LARGE_INTEGER blobSize;
    ULONG bytesRead;

    RtlZeroMemory(Descriptor, sizeof(*Descriptor));

    blob = CreateFile("hxen.elf",
                      GENERIC_READ,
                      FILE_SHARE_READ,
                      NULL,
                      OPEN_EXISTING,
                      0,
                      NULL);
    if (blob == INVALID_HANDLE_VALUE) {
        printf("Open of blob file failed, err %ld\n", GetLastError());
        return FALSE;
    }

    if (!GetFileSizeEx(blob, &blobSize)) {
        printf("Failed to get size of blob file, err %ld\n", GetLastError());
        return FALSE;
    }

    // Note: We expect HV_SIZE_MAX to be less than 4GB, otherwise we
    // can't interchangably use blobSize.u.LowPart below.

    if ((blobSize.QuadPart < HV_SIZE_MIN) ||
        (blobSize.QuadPart > HV_SIZE_MAX)) {
        printf("Blob file size out of expected range.\n");
        return FALSE;
    }

    buffer = malloc((SIZE_T)blobSize.u.LowPart);
    if (!buffer)
    {
        printf("Failed to allocate %ld bytes for blob.\n",
                blobSize.u.LowPart);
        return FALSE;
    }

    if (!ReadFile(blob,
                  buffer,
                  blobSize.u.LowPart,
                  &bytesRead,
                  NULL)) {
        printf("Blob read failed, err %ld\n", GetLastError());
        return FALSE;
    }

    if (bytesRead != blobSize.u.LowPart) {
        printf("Size mismatch, read %ld bytes, expected %ld\n",
               bytesRead,
               blobSize.u.LowPart);
        return FALSE;
    }
    Descriptor->Base = buffer;
    Descriptor->Length = bytesRead;
    CloseHandle(blob);

    printf("Loaded blob, %ld bytes.\n", bytesRead);

    return TRUE;
}

BOOLEAN
InstallBlob(
    HANDLE   hDevice,
    PMEMDESC Blob
    )
{
    BOOLEAN result;
    ULONG bytesCopied;

    result = Ioctl(hDevice,
                   TENKEI2CTL_loadhv,
                   Blob->Base,
                   Blob->Length,
                   NULL,
                   0,
                   &bytesCopied);
    free(Blob->Base);
    Blob->Base = NULL;
    Blob->Length = 0;
    printf("Copied %ld bytes to kernel space.\n", bytesCopied);
    return result;
}

static void
close_and_unload(void)
{
    if (tk2_handle != INVALID_HANDLE_VALUE)
	ExitDriver(tk2_handle);

    //
    // Unload the driver.  Ignore any errors.
    //
    ManageDriver(DRIVER_NAME,
                 NULL,
                 DRIVER_FUNC_REMOVE
	);
}

int
hxen_setup(void)
{
    BOOLEAN result;
    ULONG driverVersion;
    ULONG returnedLength;

    if ((tk2_handle = InitDriver()) == INVALID_HANDLE_VALUE) {
	 return -1;
    }

    atexit(close_and_unload);

    result = Ioctl(tk2_handle,
                   TENKEI2CTL_get_ver,
                   NULL,
                   0,
                   &driverVersion,
                   sizeof(driverVersion),
                   &returnedLength);

    if (result) {
        printf("IOCTL GET VERSION returned 0x%08lx\n", driverVersion);
    } else {
        printf("IOCTL GET VERSION FAILED, ERROR %ld\n", GetLastError());
    }

    return result ? 0 : -1;
}

void
hxen_exit(void)
{
    ULONG result;

    Ioctl(tk2_handle, TENKEI2CTL_shutdown, NULL, 0, NULL, 0, &result);
}

unsigned char *
hxen_alloc_mem(size_t guest_size)
{
    BOOLEAN result;
    ULONG returnedLength;
    int nr_pages;
    void *vmappings;

    // guest_size arg to kernel is a page count giving us a max guest size
    // of 16TB.  If more is needed we need to pass a proper input buffer
    // with a 64 bit value (or use the input buffer address as a size which
    // is kind of gross).
    nr_pages = (guest_size + KXEN_PAGE_SIZE - 1) >> KXEN_PAGE_SHIFT;
    vmappings = malloc(sizeof(struct hxen_vmap) *
		       ((nr_pages + ((1 << KXEN_VMAPPING_SHIFT) - 1)) >>
			KXEN_VMAPPING_SHIFT));
    if (vmappings == NULL)
	return NULL;
    result = Ioctl(tk2_handle, TENKEI2CTL_alloc_guest_mem,
                   vmappings, nr_pages, NULL, 0, &returnedLength);
    if (!result)
        return NULL;

    return (unsigned char *)vmappings;
}

int
hxen_run(void)
{
    BOOLEAN result;
    ULONG returnedLength;

    result = Ioctl(tk2_handle,
                   TENKEI2CTL_vcpu_run,
                   NULL, 0, NULL, 0, &returnedLength);
    return !result;
}

int
hxen_hvm_op(int Op, PVOID InBuffer, ULONG InLength)
{
    ULONG returnValue;

    if (Ioctl(tk2_handle, TENKEI2CTL_hvm_op, InBuffer, InLength, NULL,
                 Op,        // N.B. Coopted OutLength to be hvm op.
                 &returnValue) == 0)
	return -EINVAL;

    return (int)returnValue;
}

int
hxen_domctl(PVOID InBuffer, ULONG InLength)
{
    ULONG returnValue;

    if (Ioctl(tk2_handle, TENKEI2CTL_domctl, InBuffer, InLength, NULL,
                0, &returnValue) == 0)
	return -EINVAL;

    return (int)returnValue;
}

int
hxen_memory_op(int Op, PVOID InBuffer, ULONG InLength)
{
    ULONG returnValue;

    if (Ioctl(tk2_handle, TENKEI2CTL_memory_op, InBuffer, InLength, NULL,
                 Op,        // N.B. Coopted OutLength to be memory op.
                 &returnValue) == 0)
	return -EINVAL;

    return (int)returnValue;
}

int
hxen_xen_version(int Op, PVOID InBuffer, ULONG InLength)
{
    ULONG returnValue;

    if (Ioctl(tk2_handle, TENKEI2CTL_xen_version, InBuffer, InLength, NULL,
                 Op,        // N.B. Coopted OutLength to be version cmd.
                 &returnValue) == 0)
	return -EINVAL;

    return (int)returnValue;
}

void *
hxen_mmap_batch(PVOID arr, ULONG num, ULONG prot, ULONG domid)
{
    ULONG returnValue;

    if (Ioctl(tk2_handle, TENKEI2CTL_mmap_batch, arr, num,
	      (PVOID)prot, domid, &returnValue) == 0)
	return NULL;

    return (void *)returnValue;
}

void
hxen_munmap(PVOID addr, ULONG num)
{
    ULONG returnValue;

    Ioctl(tk2_handle, TENKEI2CTL_munmap, addr, num, NULL, 0, &returnValue);
}

int
hxen_trigger_keyhandler(unsigned char key)
{
    ULONG returnValue;

    if (Ioctl(tk2_handle, TENKEI2CTL_trigger_keyhandler, NULL, (ULONG)key,
	      NULL, 0, &returnValue) == 0)
	return -EINVAL;

    return (int)returnValue;
}
