/*
 * QEMU VNC display driver
 * 
 * Copyright (C) 2006 Anthony Liguori <anthony@codemonkey.ws>
 * Copyright (C) 2006 Fabrice Bellard
 * Copyright (C) 2006 Christian Limpach <Christian.Limpach@xensource.com>
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

//PLJTMP
#define QEMU_KXEN 1
//PLJTMPend
#include "vnc.h"
#include "qemu_socket.h"
#include <assert.h>
#ifndef _WIN32
#include <netinet/in.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#else
#include <winsock2.h>
#define EADDRINUSE WSAEADDRINUSE
#endif

//#define DEBUG_VNC
#ifdef DEBUG_VNC
#define dprintf(s, ...) printf(s, ## __VA_ARGS__)
#else
#define dprintf(s, ...)
#endif

/* The refresh interval starts at BASE.  If we scan the buffer and
   find no change, we increase by INC, up to MAX.  If the mouse moves
   or we get a keypress, the interval is set back to BASE.  If we find
   an update, halve the interval.

   All times in milliseconds. */
#define VNC_REFRESH_INTERVAL_BASE 30
#define VNC_REFRESH_INTERVAL_INC  50
#define VNC_REFRESH_INTERVAL_MAX  2000

/* Wait at most one second between updates, so that we can detect a
   minimised vncviewer reasonably quickly. */
#define VNC_MAX_UPDATE_INTERVAL   5000

#include "vnc_keysym.h"
#include "keymaps.c"
#include "d3des.h"

void xenstore_set_guest_clipboard(const char *text, size_t len);

typedef struct Buffer
{
    size_t capacity;
    size_t offset;
    uint8_t *buffer;
} Buffer;

typedef struct VncState VncState;
struct VncClientState;

typedef int VncReadEvent(struct VncClientState *vcs, uint8_t *data,
			 size_t len);

typedef void VncWritePixels(struct VncClientState *vcs, void *data, int size);

typedef void VncSendHextileTile(struct VncClientState *vcs,
                                uint8_t *data, int stride,
                                int w, int h,
                                void *last_bg, 
                                void *last_fg,
                                int *has_bg, int *has_fg);

struct vnc_pm_server_cut_text {
    char *text;
};

struct vnc_pending_messages {
    uint8_t vpm_resize;
    uint8_t vpm_bell;
    uint8_t vpm_null_update;
    uint8_t vpm_server_cut_text;
    uint8_t vpm_cursor_update;
};

struct VncClientState
{
    struct VncState *vs;
    int csock;
    Buffer output;
    Buffer input;

    int is_vncviewer;
    int is_xencenter;
    
    int has_resize;
    int has_hextile;
    int has_pointer_type_change;
    int has_cursor_encoding;

    int absolute;
    int last_x;
    int last_y;

    int width;
    int height;

    uint64_t *update_row;	/* outstanding updates */
    int update_requested;
    int ux;
    int uy;

    /* current output mode information */
    VncWritePixels *write_pixels;
    VncSendHextileTile *send_hextile_tile;
    int pix_bpp, pix_big_endian;
    int red_shift, red_max;
    int green_shift, green_max;
    int blue_shift, blue_max;

    VncReadEvent *read_handler;
    size_t read_handler_expect;

    struct vnc_pending_messages vpm;
};

#define VCS_INUSE(vcs) ((vcs) && (vcs)->csock != -1)
#define VCS_ACTIVE(vcs) ((vcs) && (vcs)->pix_bpp)

#define MAX_CLIENTS 8

struct VncState
{
    char *title;

    void *timer;
    int timer_interval;
    int64_t last_update_time;

    int lsock;

    DisplayState *ds;
    struct VncClientState *vcs[MAX_CLIENTS];

    uint64_t *dirty_row;	/* screen regions which are possibly dirty */
    int dirty_pixel_shift;

    int has_update;		/* there's outstanding updates in the
				 * visible area */

    uint8_t *old_data;

    int depth; /* internal VNC frame buffer byte per pixel */
    int red_shift1, red_max1;
    int green_shift1, green_max1;
    int blue_shift1, blue_max1;

    int visible_x;
    int visible_y;
    int visible_w;
    int visible_h;

    const char *display;

    char *kbd_layout_name;
    kbd_layout_t *kbd_layout;

    /* input */
    uint8_t modifiers_state[256];


    char *server_cut_text;
};

#if 0
static VncState *vnc_state; /* needed for info vnc */
#endif

#define DIRTY_PIXEL_BITS 64
#define X2DP_DOWN(vs, x) ((x) >> (vs)->dirty_pixel_shift)
#define X2DP_UP(vs, x) \
  (((x) + (1ULL << (vs)->dirty_pixel_shift) - 1) >> (vs)->dirty_pixel_shift)
#define DP2X(vs, x) ((x) << (vs)->dirty_pixel_shift)

#if 0
void do_info_vnc(void)
{
    if (vnc_state == NULL)
	term_printf("VNC server disabled\n");
    else {
	term_printf("VNC server active on: ");
	term_print_filename(vnc_state->display);
	term_printf("\n");

	if (vnc_state->csock == -1)
	    term_printf("No client connected\n");
	else
	    term_printf("Client connected\n");
    }
}
#endif

/* TODO
   1) Get the queue working for IO.
   2) there is some weirdness when using the -S option (the screen is grey
      and not totally invalidated
*/

static inline void vnc_write_pending(struct VncClientState *vcs);
static inline void vnc_write_pending_all(struct VncState *vs);
static void vnc_write(struct VncClientState *vcs, const void *data,
		      size_t len);
static void vnc_write_u32(struct VncClientState *vcs, uint32_t value);
static void vnc_write_s32(struct VncClientState *vcs, int32_t value);
static void vnc_write_u16(struct VncClientState *vcs, uint16_t value);
static void vnc_write_u8(struct VncClientState *vcs, uint8_t value);
static void vnc_flush(struct VncClientState *vcs);
static void _vnc_update_client(void *opaque);
static void vnc_update_client(void *opaque);
static void vnc_client_read(void *opaque);
static void framebuffer_set_updated_all(VncState *vs, int x, int y, int w, int h);
static void vnc_colourdepth(DisplayState *ds, int depth);
static int make_challenge(unsigned char *random, int size);
static void set_seed(unsigned int *seedp);
static void get_random(int len, unsigned char *buf);

static int shared_buf;

#if 0
static inline void vnc_set_bit(uint32_t *d, int k)
{
    d[k >> 5] |= 1 << (k & 0x1f);
}

static inline void vnc_clear_bit(uint32_t *d, int k)
{
    d[k >> 5] &= ~(1 << (k & 0x1f));
}

static inline void vnc_set_bits(uint32_t *d, int n, int nb_words)
{
    int j;

    j = 0;
    while (n >= 32) {
        d[j++] = -1;
        n -= 32;
    }
    if (n > 0) 
        d[j++] = (1 << n) - 1;
    while (j < nb_words)
        d[j++] = 0;
}

static inline int vnc_get_bit(const uint32_t *d, int k)
{
    return (d[k >> 5] >> (k & 0x1f)) & 1;
}

static inline int vnc_and_bits(const uint32_t *d1, const uint32_t *d2, 
                               int nb_words)
{
    int i;
    for(i = 0; i < nb_words; i++) {
        if ((d1[i] & d2[i]) != 0)
            return 1;
    }
    return 0;
}
#endif

static void set_bits_in_row(VncState *vs, uint64_t *row,
			    int x, int y, int w, int h)
{
    int x1, x2;
    uint64_t mask;

    if (w == 0)
	return;
    if (x > vs->ds->width)
	return;
    if (x + w > vs->ds->width)
	w = vs->ds->width - x;

    x1 = X2DP_DOWN(vs, x);
    x2 = X2DP_UP(vs, x + w);

    if (X2DP_UP(vs, w) != DIRTY_PIXEL_BITS)
	mask = ((1ULL << (x2 - x1)) - 1) << x1;
    else
	mask = ~(0ULL);

    h += y;
    if (h > vs->ds->height)
        h = vs->ds->height;
    for (; y < h; y++)
	row[y] |= mask;
}

static void vnc_dpy_update(DisplayState *ds, int x, int y, int w, int h)
{
    VncState *vs = ds->opaque;
    
    if (ds->dpy_update_sdl)
        ds->dpy_update_sdl(ds, x, y, w, h);

    if (y >= vs->ds->height) 
        return;

    set_bits_in_row(vs, vs->dirty_row, x, y, w, h);

    vs->has_update = 1;
}

static void vnc_framebuffer_update(struct VncClientState *vcs, int x, int y,
				   int w, int h, int32_t encoding)
{
    vnc_write_u16(vcs, x);
    vnc_write_u16(vcs, y);
    vnc_write_u16(vcs, w);
    vnc_write_u16(vcs, h);

    vnc_write_s32(vcs, encoding);
}

static void vnc_send_bell(DisplayState *ds)
{
    VncState *vs = ds->opaque;
    struct VncClientState *vcs;
    int i;

    dprintf("send bell\n");
    for (i = 0; i < MAX_CLIENTS; i++) {
	if (!VCS_ACTIVE(vs->vcs[i]))
	    continue;
	vcs = vs->vcs[i];

	vcs->vpm.vpm_bell++;
	vnc_write_pending(vcs);
    }
}

static void vnc_send_resize(DisplayState *ds)
{
    VncState *vs = ds->opaque;
    struct VncClientState *vcs;
    int i;

    dprintf("send resize\n");
    for (i = 0; i < MAX_CLIENTS; i++) {
	vcs = vs->vcs[i];
	if (!VCS_ACTIVE(vcs) || !vcs->has_resize)
	    continue;

	vcs->vpm.vpm_resize = 1;
	vnc_write_pending(vcs);
    }
}

static void vnc_reset_pending_messages(struct vnc_pending_messages *vpm)
{
    vpm->vpm_resize = 0;
    vpm->vpm_bell = 0;
    vpm->vpm_server_cut_text = 0;
}

static void vnc_dpy_resize_shared(DisplayState *ds, int w, int h, int depth, int linesize, void *pixels)
{
    static int allocated;
    int size_changed;
    VncState *vs = ds->opaque;
    int o, i;

    vnc_colourdepth(ds, depth);
    if (!shared_buf) {
        ds->linesize = w * vs->depth;
        if (allocated)
            ds->data = realloc(ds->data,  h * ds->linesize);
        else
            ds->data = malloc(h * ds->linesize);
        allocated = 1;
    } else {
        ds->linesize = linesize;
        if (allocated) {
            free(ds->data);
            allocated = 0;
        }
    }
    for (i = 0; i < MAX_CLIENTS; i++) {
	if (!VCS_INUSE(vs->vcs[i]))
	    continue;
	vs->vcs[i]->update_row =
	    qemu_reallocz(vs->vcs[i]->update_row,
			  h * sizeof(vs->vcs[i]->update_row[0]));
	if (vs->vcs[i]->update_row == NULL) {
	    fprintf(stderr, "vnc: memory allocation failed\n");
	    exit(1);
	}
    }
    vs->old_data = qemu_reallocz(vs->old_data, w * h * vs->depth);
    vs->dirty_row = qemu_reallocz(vs->dirty_row, h * sizeof(vs->dirty_row[0]));

    if (ds->data == NULL || vs->old_data == NULL ||
	vs->dirty_row == NULL) {
	fprintf(stderr, "vnc: memory allocation failed\n");
	exit(1);
    }

    size_changed = ds->width != w || ds->height != h;
    ds->width = w;
    ds->height = h;
    if (size_changed)
        vnc_send_resize(ds);
    vs->dirty_pixel_shift = 0;
    for (o = DIRTY_PIXEL_BITS; o < ds->width; o *= 2)
	vs->dirty_pixel_shift++;
    framebuffer_set_updated_all(vs, 0, 0, ds->width, ds->height);
    if (shared_buf) ds->data = pixels;
    if (!ds->dpy_resize_sdl) {
        ds->shared_buf = shared_buf;
        ds->depth = vs->depth * 8;
    }
}

static void vnc_dpy_resize(DisplayState *ds, int w, int h)
{
    if (ds->dpy_resize_sdl) {
        ds->dpy_resize_sdl(ds, w, h);
        vnc_dpy_resize_shared(ds, w, h, ds->depth, ds->linesize, ds->data);
    } else {
        vnc_dpy_resize_shared(ds, w, h, 0, w * (ds->depth / 8), NULL);
    }
}

/* fastest code */
static void vnc_write_pixels_copy(struct VncClientState *vcs, void *pixels,
				  int size)
{
    vnc_write(vcs, pixels, size);
}

/* slowest but generic code. */
static void vnc_convert_pixel(struct VncClientState *vcs, uint8_t *buf,
                  uint32_t v)
{
    uint8_t r, g, b;
    VncState *vs = vcs->vs;
    r = ((v >> vs->red_shift1) & vs->red_max1) * (vcs->red_max + 1) / (vs->red_max1 + 1);
    g = ((v >> vs->green_shift1) & vs->green_max1) * (vcs->green_max + 1) / (vs->green_max1 + 1);
    b = ((v >> vs->blue_shift1) & vs->blue_max1) * (vcs->blue_max + 1) / (vs->blue_max1 + 1);
    switch(vcs->pix_bpp) {
    case 1:
        buf[0] = (r << vcs->red_shift) | (g << vcs->green_shift) | (b << vcs->blue_shift);
        break;
    case 2:
    {
        uint16_t *p = (uint16_t *) buf;
        *p = (r << vcs->red_shift) | (g << vcs->green_shift) | (b << vcs->blue_shift);
        if (vcs->pix_big_endian) {
            *p = htons(*p);
        }
    }
        break;
    default:
    case 4:
    {
        uint32_t *p = (uint32_t *) buf;
        *p = (r << vcs->red_shift) | (g << vcs->green_shift) | (b << vcs->blue_shift);
        if (vcs->pix_big_endian) {
            *p = htonl(*p);
        }
        break;
    }
    }
}

static void vnc_write_pixels_generic(struct VncClientState *vcs,
				     void *pixels1, int size)
{
    uint8_t buf[4];

    if (vcs->vs->depth == 4) {
        uint32_t *pixels = pixels1;
        int n, i;
        n = size >> 2;
        for(i = 0; i < n; i++) {
            vnc_convert_pixel(vcs, buf, pixels[i]);
            vnc_write(vcs, buf, vcs->pix_bpp);
        }
    } else if (vcs->vs->depth == 2) {
        uint16_t *pixels = pixels1;
        int n, i;
        n = size >> 1;
        for(i = 0; i < n; i++) {
            vnc_convert_pixel(vcs, buf, pixels[i]);
            vnc_write(vcs, buf, vcs->pix_bpp);
        }
    } else if (vcs->vs->depth == 1) {
        uint8_t *pixels = pixels1;
        int n, i;
        n = size;
        for(i = 0; i < n; i++) {
            vnc_convert_pixel(vcs, buf, pixels[i]);
            vnc_write(vcs, buf, vcs->pix_bpp);
        }
    } else {
        fprintf(stderr, "vnc_write_pixels_generic: VncState color depth not supported\n");
    }
}

unsigned char cursorbmsk[16] = {
	0xff, /* 11111111 */
	0x3c, /* 00111100 */
	0x18, /* 00011000 */
	0x18, /* 00011000 */
	0x18, /* 00011000 */
	0x18, /* 00011000 */
	0x18, /* 00011000 */
	0x18, /* 00011000 */
	0x18, /* 00011000 */
	0x18, /* 00011000 */
	0x18, /* 00011000 */
	0x18, /* 00011000 */
	0x18, /* 00011000 */
	0x18, /* 00011000 */
	0x3c, /* 00111100 */
	0xff, /* 11111111 */
};

static void writeGrey(struct VncClientState *vcs, unsigned char **cur)
{
    uint8_t r, g, b;
    r = (0xc0) * (vcs->red_max + 1) / (256);
    g = (0xc0) * (vcs->green_max + 1) / (256);
    b = (0xc0) * (vcs->blue_max + 1) / (256);
    switch(vcs->pix_bpp) {
    case 1:
        **cur = (r << vcs->red_shift) | (g << vcs->green_shift) | (b << vcs->blue_shift);
        *cur = *cur + 1;
        break;
    case 2:
    {
        uint16_t *p = (uint16_t *) *cur;
        *p = (r << vcs->red_shift) | (g << vcs->green_shift) | (b << vcs->blue_shift);
        if (vcs->pix_big_endian) {
            *p = htons(*p);
        }
        *cur = *cur + 2;
    }
        break;
    case 3:
        if (vcs->pix_big_endian) {
            (*cur)[0] = r;
            (*cur)[1] = g;
            (*cur)[2] = b;
        } else {
            (*cur)[0] = b;
            (*cur)[1] = g;
            (*cur)[2] = r;
        }
        *cur = *cur + 3;
        break;    
    default:
    case 4:
        if (vcs->pix_big_endian) {
            (*cur)[0] = 255;
            (*cur)[1] = r;
            (*cur)[2] = g;
            (*cur)[3] = b;
        } else {
            (*cur)[0] = b;
            (*cur)[1] = g;
            (*cur)[2] = r;
            (*cur)[3] = 255;
        }
        *cur = *cur + 4;
        break;
    }
}

static void writeZero(struct VncClientState *vcs, unsigned char **cur)
{
    memset(*cur, 0x00, vcs->pix_bpp);
    *cur = *cur + vcs->pix_bpp;
}

static void vnc_send_custom_cursor(struct VncClientState *vcs)
{
    unsigned char *cursorcur, *cur;
    unsigned int size, i, j;
     
    if (vcs->has_cursor_encoding != 1)
    return;

    dprintf("sending custom cursor %d with bpp %d\n", vcs->csock,
        vcs->pix_bpp);
    size = sizeof(cursorbmsk) * 8 * vcs->pix_bpp;
    cursorcur = malloc(size);
    if (cursorcur == NULL)
    return;

    cur = (unsigned char *) cursorcur;

    for (i = 0; i < sizeof(cursorbmsk); i++) {
        for (j = 0; j < 8; j++) {
            if (cursorbmsk[i] & (1 << (8 - j)))
                writeGrey(vcs, &cur);
            else
                writeZero(vcs, &cur);
        }
    }

    vnc_write_u16(vcs, 0);
    vnc_write_u16(vcs, 1); /* number of rects */

    /* width 8, height - number of bytes in mask, hotspot in the middle */
    vnc_framebuffer_update(vcs, 8 / 2, sizeof(cursorbmsk) / 2, 8,
               sizeof(cursorbmsk), -239);
    vnc_write_pixels_copy(vcs, cursorcur, size);
    vnc_write(vcs, cursorbmsk, sizeof(cursorbmsk));

    free(cursorcur);
}

static void hextile_enc_cord(uint8_t *ptr, int x, int y, int w, int h)
{
    ptr[0] = ((x & 0x0F) << 4) | (y & 0x0F);
    ptr[1] = (((w - 1) & 0x0F) << 4) | ((h - 1) & 0x0F);
}

#define BPP 8
#include "vnchextile.h"
#undef BPP

#define BPP 16
#include "vnchextile.h"
#undef BPP

#define BPP 32
#include "vnchextile.h"
#undef BPP

#define GENERIC
#define BPP 8
#include "vnchextile.h"
#undef BPP
#undef GENERIC

#define GENERIC
#define BPP 16
#include "vnchextile.h"
#undef BPP
#undef GENERIC

#define GENERIC
#define BPP 32
#include "vnchextile.h"
#undef BPP
#undef GENERIC

static inline int find_update_height(struct VncClientState *vcs, int y,
				     int maxy, uint64_t mask)
{
    int h = 0;

    while (vcs->update_row[y + h] & mask) {
	vcs->update_row[y + h] &= ~mask;
	h++;
	if (y + h == maxy)
	    break;
    }

    return h;
}

static void _vnc_update_client(void *opaque)
{
    VncState *vs = opaque;
    int64_t now;
    int y;
    uint8_t *row;
    uint8_t *old_row;
    uint64_t width_mask;
    int tile_bytes = vs->depth * DP2X(vs, 1);
    int i;

    now = vs->ds->get_clock();

    if (vs->ds->width != DP2X(vs, DIRTY_PIXEL_BITS))
	width_mask = (1ULL << X2DP_UP(vs, vs->ds->width)) - 1;
    else
	width_mask = ~(0ULL);

    /* Walk through the dirty map and eliminate tiles that really
       aren't dirty */
    row = vs->ds->data;
    old_row = vs->old_data;

    for (y = 0; y < vs->ds->height; y++) {
	if (vs->dirty_row[y] & width_mask) {
	    int x, i;
	    uint8_t *ptr, *old_ptr;

	    ptr = row;
	    old_ptr = old_row;

	    for (x = 0; x < X2DP_UP(vs, vs->ds->width); x++) {
		if (vs->dirty_row[y] & (1ULL << x)) {
		    if (memcmp(old_ptr, ptr, tile_bytes)) {
                        for (i = 0; i < MAX_CLIENTS; i++) {
                            if (!VCS_ACTIVE(vs->vcs[i]))
                                continue;

                            vs->vcs[i]->update_row[y] |= (1ULL << x);
                        }
			memcpy(old_ptr, ptr, tile_bytes);
		    }
		}

		ptr += tile_bytes;
		old_ptr += tile_bytes;
	    }
	    vs->dirty_row[y] &= ~width_mask;
	}
  
	row += vs->ds->linesize;
	old_row += vs->ds->linesize;
    }

    if (!vs->has_update || vs->visible_y >= vs->ds->height ||
	vs->visible_x >= vs->ds->width) {
        /* No update -> back off a bit */
        vs->timer_interval += VNC_REFRESH_INTERVAL_INC;
        if (vs->timer_interval > VNC_REFRESH_INTERVAL_MAX) {
            vs->timer_interval = VNC_REFRESH_INTERVAL_MAX;
            if (now - vs->last_update_time >= VNC_MAX_UPDATE_INTERVAL) {
                /* Send a null update.  If the client is no longer
                   interested (e.g. minimised) it'll ignore this, and we
                   can stop scanning the buffer until it sends another
                   update request. */
                /* It turns out that there's a bug in realvncviewer 4.1.2
                   which means that if you send a proper null update (with
                   no update rectangles), it gets a bit out of sync and
                   never sends any further requests, regardless of whether
                   it needs one or not.  Fix this by sending a single 1x1
                   update rectangle instead. */
                dprintf("send null update\n");
                framebuffer_set_updated_all(vs, 0, 0, 1, 1);
            }
        }
    } else {
        vs->timer_interval /= 2;
        if (vs->timer_interval < VNC_REFRESH_INTERVAL_BASE)
            vs->timer_interval = VNC_REFRESH_INTERVAL_BASE;
    }

    for (i = 0; i < MAX_CLIENTS; i++) {
	if (!VCS_ACTIVE(vs->vcs[i]))
	    continue;

	if (vs->vcs[i]->ux >= X2DP_UP(vs, vs->ds->width) &&
	    vs->vcs[i]->update_requested) {
	    vs->vcs[i]->ux = X2DP_DOWN(vs, vs->visible_x);
	    vnc_write_pending(vs->vcs[i]);
	}
    }

    vs->has_update = 0;
    vs->last_update_time = now;
    vs->ds->set_timer(vs->timer, now + vs->timer_interval);
    return;
}

static void vnc_set_server_text(DisplayState *ds, char *text)
{
    VncState *vs = ds->opaque;
    int i;

    if (vs->server_cut_text)
	free(vs->server_cut_text);
    vs->server_cut_text = text;
    for (i = 0; i < MAX_CLIENTS; i++) {
	if (!VCS_ACTIVE(vs->vcs[i]))
	    continue;
	vs->vcs[i]->vpm.vpm_server_cut_text = 1;
	vnc_write_pending(vs->vcs[i]);
    }
    dprintf("set server text %s\n", vs->server_cut_text);
}

static void buffer_reserve(Buffer *buffer, size_t len)
{
    if ((buffer->capacity - buffer->offset) < len) {
        buffer->capacity += (len + 1024);
        buffer->buffer = realloc(buffer->buffer, buffer->capacity);
        if (buffer->buffer == NULL) {
            fprintf(stderr, "vnc: out of memory\n");
            exit(1);
        }
    }
}

static int buffer_empty(Buffer *buffer)
{
    return buffer->offset == 0;
}

static uint8_t *buffer_end(Buffer *buffer)
{
    return buffer->buffer + buffer->offset;
}

static void buffer_reset(Buffer *buffer)
{
    buffer->offset = 0;
}

static void buffer_append(Buffer *buffer, const void *data, size_t len)
{
    memcpy(buffer->buffer + buffer->offset, data, len);
    buffer->offset += len;
}

static void vnc_colourdepth(DisplayState *ds, int depth)
{
    int i, host_big_endian_flag;
    struct VncClientState *vcs;
    VncState *vs;
    
    switch (depth) {
        case 24:
            shared_buf = 0;
            if (ds->depth == 32) return;
            depth = 32;
            break;
        case 8:
        case 0:
            shared_buf = 0;
            return;
        default:
            shared_buf = 1;
            break;
    }
 
#ifdef WORDS_BIGENDIAN
    host_big_endian_flag = 1;
#else
    host_big_endian_flag = 0;
#endif
    vs = ds->opaque;
    switch (depth) {
        case 8:
            vs->depth = depth / 8;
            vs->red_max1 = 7;
            vs->green_max1 = 7;
            vs->blue_max1 = 3;
            vs->red_shift1 = 5;
            vs->green_shift1 = 2;
            vs->blue_shift1 = 0;
            break;
        case 16:
            vs->depth = depth / 8;
            vs->red_max1 = 31;
            vs->green_max1 = 63;
            vs->blue_max1 = 31;
            vs->red_shift1 = 11;
            vs->green_shift1 = 5;
            vs->blue_shift1 = 0;
            break;
        case 32:
            vs->depth = 4;
            vs->red_max1 = 255;
            vs->green_max1 = 255;
            vs->blue_max1 = 255;
            vs->red_shift1 = 16;
            vs->green_shift1 = 8;
            vs->blue_shift1 = 0;
            break;
        default:
            return;
    }
    for (i = 0; i < MAX_CLIENTS; i++) {
        if (VCS_ACTIVE(vs->vcs[i])) {
            vcs = vs->vcs[i];
            /* If this client is not XenCenter or our own modified vncviewer version,
             * do the colour conversion.*/
            if (!vcs->is_vncviewer && !vcs->is_xencenter) {
                if (vcs->pix_bpp == 4 && vs->depth == 4 &&
                    host_big_endian_flag == vcs->pix_big_endian &&
                    vcs->red_max == 0xff && vcs->green_max == 0xff && vcs->blue_max == 0xff &&
                    vcs->red_shift == 16 && vcs->green_shift == 8 && vcs->blue_shift == 0) {
                    vcs->write_pixels = vnc_write_pixels_copy;
                    vcs->send_hextile_tile = send_hextile_tile_32;                   
                } else if (vcs->pix_bpp == 2 && vs->depth == 2 &&
                    host_big_endian_flag == vcs->pix_big_endian &&
                    vcs->red_max == 31 && vcs->green_max == 63 && vcs->blue_max == 31 &&
                    vcs->red_shift == 11 && vcs->green_shift == 5 && vcs->blue_shift == 0) {
                    vcs->write_pixels = vnc_write_pixels_copy;
                    vcs->send_hextile_tile = send_hextile_tile_16;
                } else if (vcs->pix_bpp == 1 && vs->depth == 1 &&
                    host_big_endian_flag == vcs->pix_big_endian &&
                    vcs->red_max == 7 && vcs->green_max == 7 && vcs->blue_max == 3 &&
                    vcs->red_shift == 5 && vcs->green_shift == 2 && vcs->blue_shift == 0) {
                    vcs->write_pixels = vnc_write_pixels_copy;
                    vcs->send_hextile_tile = send_hextile_tile_8;
                } else {
                    if (vs->depth == 4) {
                        vcs->send_hextile_tile = send_hextile_tile_generic_32;
                    } else if (vcs->vs->depth == 2) {
                        vcs->send_hextile_tile = send_hextile_tile_generic_16;
                    } else {
                        vcs->send_hextile_tile = send_hextile_tile_generic_8;
                    }
                    vcs->write_pixels = vnc_write_pixels_generic;
                }
            /* this client is XenCenter or our own vncviewer version:
             * just close the connection. */
            } else {
                ds->set_fd_handler(vcs->csock, NULL, NULL, NULL, NULL);
                closesocket(vcs->csock);
                vcs->csock = -1;
                buffer_reset(&vcs->input);
                buffer_reset(&vcs->output);
                vnc_reset_pending_messages(&vcs->vpm);
                vcs->pix_bpp = 0;
                free(vcs->update_row);
                vcs->update_row = NULL;
            }
            
        }
    }
}

static void vnc_update_client(void *opaque)
{
    VncState *vs = opaque;

    vs->ds->dpy_refresh(vs->ds);
    _vnc_update_client(vs);
}

static void vnc_timer_init(VncState *vs)
{
    if (vs->timer == NULL) {
	vs->timer = vs->ds->init_timer(vnc_update_client, vs);
	vs->timer_interval = VNC_REFRESH_INTERVAL_BASE;
    }
}

static void vnc_dpy_refresh(DisplayState *ds)
{
    if (ds->hw_update)
	ds->hw_update(ds->hw_opaque);
}

#if 0
static int vnc_listen_poll(void *opaque)
{
    return 1;
}
#endif

static int vnc_client_io_error(struct VncClientState *vcs, int ret,
			       int last_errno)
{
    struct VncState *vs = vcs->vs;
    int i;

    if (ret != 0 && ret != -1)
	return ret;

    if (ret == -1 && (last_errno == EINTR || last_errno == EAGAIN))
	return 0;

    dprintf("vnc_client_io_error on %d\n", vcs->csock);
    vs->ds->set_fd_handler(vcs->csock, NULL, NULL, NULL, NULL);
    closesocket(vcs->csock);
    vcs->csock = -1;
    buffer_reset(&vcs->input);
    buffer_reset(&vcs->output);
    vnc_reset_pending_messages(&vcs->vpm);
    vcs->pix_bpp = 0;
    free(vcs->update_row);
    vcs->update_row = NULL;
    for (i = 0; i < MAX_CLIENTS; i++)
	if (VCS_ACTIVE(vs->vcs[i]))
	    return 0;
    vs->ds->idle = 1;
    return 0;
}

static void vnc_client_error(struct VncClientState *vcs)
{
    vnc_client_io_error(vcs, -1, EINVAL);
}

static int vnc_process_messages(struct VncClientState *vcs)
{
    struct vnc_pending_messages *vpm;
    struct VncState *vs = vcs->vs;
    int nr_rects = 0, nr_rects_offset = 0;

    if (vcs->output.offset)
	return vcs->output.offset;

    vpm = &vcs->vpm;
    if (vpm == NULL)
	return 0;

    dprintf("processing messages\n");
    if (vpm->vpm_resize) {
	dprintf("++ resize\n");
	vnc_write_u8(vcs, 0);  /* msg id */
	vnc_write_u8(vcs, 0);
	vnc_write_u16(vcs, 1); /* number of rects */
	vnc_framebuffer_update(vcs, 0, 0, vs->ds->width, vs->ds->height, -223);
	vcs->width = vs->ds->width;
	vcs->height = vs->ds->height;
	vpm->vpm_resize = 0;
    }
    while (vpm->vpm_bell) {
	dprintf("++ bell\n");
	vnc_write_u8(vcs, 2);  /* msg id */
	vpm->vpm_bell--;
    }
    if (vpm->vpm_server_cut_text) {
	char pad[3] = { 0, 0, 0 };
	dprintf("++ cut text\n");
	vnc_write_u8(vcs, 3);	/* ServerCutText */
	vnc_write(vcs, pad, 3);	/* padding */
	vnc_write_u32(vcs, strlen(vs->server_cut_text));	/* length */
	vnc_write(vcs, vs->server_cut_text,
		  strlen(vs->server_cut_text)); /* text */
	vpm->vpm_server_cut_text = 0;
    }
    if (vpm->vpm_cursor_update) {
	dprintf("++ custom cursor\n");
	vnc_send_custom_cursor(vcs);
	vpm->vpm_cursor_update = 0;
    }
#define BATCHSIZE (4 * 1024 * 1024)
    while (VCS_ACTIVE(vcs) && vcs->ux < X2DP_UP(vs, vs->ds->width) &&
	vcs->output.offset <= BATCHSIZE) {
	uint64_t mask = 1ULL << vcs->ux;
	int y, h = 0;
	for (y = vcs->uy; y < vs->ds->height; y++) {
	    h = find_update_height(vcs, y, vs->ds->height, mask);
	    if (h != 0) {
		int x1 = DP2X(vs, vcs->ux);
		int w = MIN(DP2X(vs, vcs->ux + 1), vcs->width) - x1;
		int y1 = y;
		int i, j, stride;
		uint8_t *row;

		if (x1 >= vcs->width || y1 >= vcs->height) {
		    y += h;
		    h = 0;
		    continue;
		}

		if (nr_rects == 0) {
		    vnc_write_u8(vcs, 0);  /* msg id */
		    vnc_write_u8(vcs, 0);
		    nr_rects_offset = vcs->output.offset;
		    vnc_write_u16(vcs, 0);
		}

		nr_rects++;
		vnc_framebuffer_update(vcs, x1, y1, w, h,
				       vcs->has_hextile ? 5 : 0);
		row = vs->ds->data + y1 * vs->ds->linesize +
		    x1 * vs->depth;
		stride = vs->ds->linesize;
		if (vcs->has_hextile) {
		    int has_fg, has_bg;
            void *last_fg, *last_bg;
            last_fg = (void *) malloc(vcs->vs->depth);
            last_bg = (void *) malloc(vcs->vs->depth);
		    has_fg = has_bg = 0;
		    for (j = 0; j < h; j += 16) {
    			for (i = 0; i < w; i += 16) {
    			    vcs->send_hextile_tile(vcs, row + i * vs->depth,
    						   stride,
    						   MIN(16, w - i),
    						   MIN(16, h - j),
    						   last_bg, last_fg,
    						   &has_bg, &has_fg);
    			}
    			row += 16 * stride;
		    }
            free(last_fg);
            free(last_bg);
		} else {
		    for (i = 0; i < h; i++) {
			vcs->write_pixels(vcs, row, w * vs->depth);
			row += stride;
		    }
		}
		dprintf("-- sent %d %d %d %d\n", x1, y1, w, h);

		y += h;
		vcs->uy = y;
		if (vcs->output.offset > BATCHSIZE)
		    break;
		h = 0;
	    }
	}
	if (h != 0)
	    break;
	vcs->uy = vs->visible_y;
	vcs->ux++;
    }
    if (nr_rects) {
	uint8_t buf[2];

	buf[0] = (nr_rects >> 8) & 0xFF;
	buf[1] = nr_rects & 0xFF;
	
	memcpy(vcs->output.buffer + nr_rects_offset, buf, 2);
	vcs->update_requested--;
	if (vcs->update_requested < 0)
	    vcs->update_requested = 0;
    }
    return vcs->output.offset;
}

#if defined(QEMU_KXEN)
static void vnc_client_write(void *);

static void vnc_client_write_retry(void *opaque)
{
    struct VncClientState *vcs = opaque;

    dprintf("disable write retry\n");
    vcs->vs->ds->set_fd_handler(vcs->csock, NULL, vnc_client_read, NULL,
                                vcs);
    vnc_client_write(vcs);
}

static void vnc_client_write(void *opaque)
{
    long ret;
    struct VncClientState *vcs = opaque;

    while (1) {
	if (vnc_process_messages(vcs) == 0)
	    break;

	dprintf("write %d\n", vcs->output.offset);
	ret = send(vcs->csock, vcs->output.buffer, vcs->output.offset, 0);
	ret = vnc_client_io_error(vcs, ret, socket_error());
	if (!ret) {
	    dprintf("write error %d with %d\n", errno, vcs->output.offset);
	    return;
	}

        if (ret != vcs->output.offset) {
            memmove(vcs->output.buffer, vcs->output.buffer + ret,
                    vcs->output.offset - ret);
            vcs->output.offset -= ret;

            dprintf("enable write retry, %d bytes remaining\n", ret);
            vcs->vs->ds->set_fd_handler(vcs->csock, NULL, vnc_client_read,
                                        vnc_client_write_retry, vcs);
	    break;
        } else
            vcs->output.offset = 0;
    }
}
#else
static void vnc_client_write(void *opaque)
{
    long ret;
    struct VncClientState *vcs = opaque;
    struct VncState *vs = vcs->vs;

    while (1) {
	if (vnc_process_messages(vcs) == 0) {
	    dprintf("disable write\n");
	    vs->ds->set_fd_handler(vcs->csock, NULL, vnc_client_read, NULL,
				   vcs);
	    break;
	}

	dprintf("write %d\n", vcs->output.offset);
	ret = send(vcs->csock, vcs->output.buffer, vcs->output.offset, 0);
	ret = vnc_client_io_error(vcs, ret, socket_error());
	if (!ret) {
	    dprintf("write error %d with %d\n", errno, vcs->output.offset);
	    return;
	}

	memmove(vcs->output.buffer, vcs->output.buffer + ret,
		vcs->output.offset - ret);
	vcs->output.offset -= ret;

	if (vcs->output.offset)
	    break;
    }
}
#endif

static void vnc_read_when(struct VncClientState *vcs, VncReadEvent *func,
			  size_t expecting)
{
    vcs->read_handler = func;
    vcs->read_handler_expect = expecting;
}

static void vnc_client_read(void *opaque)
{
    struct VncClientState *vcs = opaque;
    long ret;

    buffer_reserve(&vcs->input, 4096);

    ret = recv(vcs->csock, buffer_end(&vcs->input), 4096, 0);
    ret = vnc_client_io_error(vcs, ret, socket_error());
    if (!ret)
	return;

    vcs->input.offset += ret;

    while (vcs->read_handler &&
	   vcs->input.offset >= vcs->read_handler_expect) {
	size_t len = vcs->read_handler_expect;
	int ret;

	ret = vcs->read_handler(vcs, vcs->input.buffer, len);
	if (vcs->csock == -1)
	    return;

	if (!ret) {
	    memmove(vcs->input.buffer, vcs->input.buffer + len,
		    vcs->input.offset - len);
	    vcs->input.offset -= len;
	} else {
	    assert(ret > vcs->read_handler_expect);
	    vcs->read_handler_expect = ret;
	}
    }
}

static inline void vnc_write_pending(struct VncClientState *vcs)
{
    if (buffer_empty(&vcs->output)) {
#if !defined(QEMU_KXEN)
	dprintf("enable write\n");
	vcs->vs->ds->set_fd_handler(vcs->csock, NULL, vnc_client_read,
			            vnc_client_write, vcs);
#else
        vnc_client_write(vcs);
#endif
    }
}

static inline void vnc_write_pending_all(struct VncState *vs)
{
    int i;

    for (i = 0; i < MAX_CLIENTS; i++)
	if (VCS_ACTIVE(vs->vcs[i]))
	    vnc_write_pending(vs->vcs[i]);
}

static void vnc_write(struct VncClientState *vcs, const void *data, size_t len)
{
    buffer_reserve(&vcs->output, len);

#if !defined(QEMU_KXEN)
    vnc_write_pending(vcs);
#endif

    buffer_append(&vcs->output, data, len);
#if defined(QEMU_KXEN)
    vnc_write_pending(vcs);
#endif
}

static void vnc_write_s32(struct VncClientState *vcs, int32_t value)
{
    vnc_write_u32(vcs, *(uint32_t *)&value);
}

static void vnc_write_u32(struct VncClientState *vcs, uint32_t value)
{
    uint8_t buf[4];

    buf[0] = (value >> 24) & 0xFF;
    buf[1] = (value >> 16) & 0xFF;
    buf[2] = (value >>  8) & 0xFF;
    buf[3] = value & 0xFF;

    vnc_write(vcs, buf, 4);
}

static void vnc_write_u16(struct VncClientState *vcs, uint16_t value)
{
    uint8_t buf[2];

    buf[0] = (value >> 8) & 0xFF;
    buf[1] = value & 0xFF;

    vnc_write(vcs, buf, 2);
}

static void vnc_write_u8(struct VncClientState *vcs, uint8_t value)
{
    vnc_write(vcs, &value, 1);
}

static void vnc_flush(struct VncClientState *vcs)
{
    if (vnc_process_messages(vcs))
	vnc_client_write(vcs);
}

static uint8_t read_u8(uint8_t *data, size_t offset)
{
    return data[offset];
}

static uint16_t read_u16(uint8_t *data, size_t offset)
{
    return ntohs(*(uint16_t *)(data + offset));
}

static uint32_t read_u32(uint8_t *data, size_t offset)
{
    return ntohl(*(uint32_t *)(data + offset));
}

static int32_t read_s32(uint8_t *data, size_t offset)
{
    return (int32_t)read_u32(data, offset);
}

static void client_cut_text(VncState *vs, size_t len, char *text)
{
    xenstore_set_guest_clipboard(text, len);
}

static void check_pointer_type_change(struct VncClientState *vcs, int absolute)
{
    struct VncState *vs = vcs->vs;
    if (vcs->has_pointer_type_change && vcs->absolute != absolute) {
	dprintf("pointer type change\n");
	vnc_write_u8(vcs, 0);
	vnc_write_u8(vcs, 0);
	vnc_write_u16(vcs, 1);
	vnc_framebuffer_update(vcs, absolute, 0,
			       vs->ds->width, vs->ds->height, -257);
	vnc_flush(vcs);
    }
    vcs->absolute = absolute;
}

static void pointer_event(struct VncClientState *vcs, int button_mask,
			  int x, int y)
{
    struct VncState *vs = vcs->vs;
    int buttons = 0;
    int dz = 0;

    if (button_mask & 0x01)
	buttons |= MOUSE_EVENT_LBUTTON;
    if (button_mask & 0x02)
	buttons |= MOUSE_EVENT_MBUTTON;
    if (button_mask & 0x04)
	buttons |= MOUSE_EVENT_RBUTTON;
    /* scrolling wheel events */
    if (button_mask & 0x08)
	dz = -1;
    if (button_mask & 0x10)
	dz = 1;

    if (vcs->absolute) {
	vs->ds->mouse_event(x * 0x7FFF / vs->ds->width,
			    y * 0x7FFF / vs->ds->height,
			    dz, buttons, vs->ds->mouse_opaque);
    } else if (vcs->has_pointer_type_change) {
	x -= 0x7FFF;
	y -= 0x7FFF;

	vs->ds->mouse_event(x, y, dz, buttons, vs->ds->mouse_opaque);
    } else {
	if (vcs->last_x != -1)
	    vs->ds->mouse_event(x - vcs->last_x,
				y - vcs->last_y,
				dz, buttons,
				vs->ds->mouse_opaque);
	vcs->last_x = x;
	vcs->last_y = y;
    }

    check_pointer_type_change(vcs,
			      vs->ds->mouse_is_absolute(vs->ds->mouse_opaque));
}

static void reset_keys(VncState *vs)
{
    int i;
    for(i = 0; i < 256; i++) {
        if (vs->modifiers_state[i]) {
            if (i & 0x80)
                vs->ds->kbd_put_keycode(0xe0);
            vs->ds->kbd_put_keycode(i | 0x80);
            vs->modifiers_state[i] = 0;
        }
    }
}

static void press_key(VncState *vs, int keysym)
{
    vs->ds->kbd_put_keycode(keysym2scancode(vs->kbd_layout, keysym) & 0x7f);
    vs->ds->kbd_put_keycode(keysym2scancode(vs->kbd_layout, keysym) | 0x80);
}

static void do_key_event(VncState *vs, int down, uint32_t sym)
{
    int keycode;

    dprintf("do_key_event down %d sym %x\n", down, sym);
    keycode = keysym2scancode(vs->kbd_layout, sym & 0xFFFF);
    
    /* QEMU console switch */
    switch(keycode) {
    case 0x2a:                          /* Left Shift */
    case 0x36:                          /* Right Shift */
    case 0x1d:                          /* Left CTRL */
    case 0x9d:                          /* Right CTRL */
    case 0x38:                          /* Left ALT */
    case 0xb8:                          /* Right ALT */
        if (down)
            vs->modifiers_state[keycode] = 1;
        else
            vs->modifiers_state[keycode] = 0;
        break;
    case 0x02 ... 0x0a: /* '1' to '9' keys */ 
        if (down && vs->modifiers_state[0x1d] && vs->modifiers_state[0x38]) {
            /* Reset the modifiers sent to the current console */
            reset_keys(vs);
            // console_select(keycode - 0x02);
            return;
        }
        break;
    case 0x45:			/* NumLock */
	if (!down)
	    vs->modifiers_state[keycode] ^= 1;
	break;
    }

    if (keycodeIsKeypad(vs->kbd_layout, keycode)) {
        /* If the numlock state needs to change then simulate an additional
           keypress before sending this one.  This will happen if the user
           toggles numlock away from the VNC window.
        */
        if (keysymIsNumlock(vs->kbd_layout, sym & 0xFFFF)) {
	    if (!vs->modifiers_state[0x45]) {
		vs->modifiers_state[0x45] = 1;
		press_key(vs, 0xff7f);
	    }
	} else {
	    if (vs->modifiers_state[0x45]) {
		vs->modifiers_state[0x45] = 0;
		press_key(vs, 0xff7f);
	    }
        }
    }

    if (vs->ds->graphic_mode) {
        if (keycode & 0x80)
            vs->ds->kbd_put_keycode(0xe0);
        if (down)
            vs->ds->kbd_put_keycode(keycode & 0x7f);
        else
            vs->ds->kbd_put_keycode(keycode | 0x80);
    } else {
        /* QEMU console emulation */
        if (down) {
	    int mod = 0;
	    if (vs->modifiers_state[0x1d] || vs->modifiers_state[0x9d])
		mod += QEMU_KEY_MOD_CTRL;
            switch (keycode) {
            case 0x2a:                          /* Left Shift */
            case 0x36:                          /* Right Shift */
            case 0x1d:                          /* Left CTRL */
            case 0x9d:                          /* Right CTRL */
            case 0x38:                          /* Left ALT */
            case 0xb8:                          /* Right ALT */
                break;
            case 0xc8:
                vs->ds->kbd_put_keysym(QEMU_KEY_UP + mod);
                break;
            case 0xd0:
                vs->ds->kbd_put_keysym(QEMU_KEY_DOWN + mod);
                break;
            case 0xcb:
                vs->ds->kbd_put_keysym(QEMU_KEY_LEFT + mod);
                break;
            case 0xcd:
                vs->ds->kbd_put_keysym(QEMU_KEY_RIGHT + mod);
                break;
            case 0xd3:
                vs->ds->kbd_put_keysym(QEMU_KEY_DELETE + mod);
                break;
            case 0xc7:
                vs->ds->kbd_put_keysym(QEMU_KEY_HOME + mod);
                break;
            case 0xcf:
                vs->ds->kbd_put_keysym(QEMU_KEY_END + mod);
                break;
            case 0xc9:
                vs->ds->kbd_put_keysym(QEMU_KEY_PAGEUP + mod);
                break;
            case 0xd1:
                vs->ds->kbd_put_keysym(QEMU_KEY_PAGEDOWN + mod);
                break;
            default:
		if (vs->modifiers_state[0x1d] || vs->modifiers_state[0x9d])
		    sym &= 0x1f;
                vs->ds->kbd_put_keysym(sym);
                break;
            }
        }
    }
}

static void key_event(VncState *vs, int down, uint32_t sym)
{
    if (sym >= 'A' && sym <= 'Z' && vs->ds->graphic_mode)
	sym = sym - 'A' + 'a';
    do_key_event(vs, down, sym);
}

static void scan_event(VncState *vs, int down, uint32_t code)
{

    /* Prefix with 0xe0 if high bit set, except for NumLock key. */
    if (code & 0x80 && code != 0xc5)
	vs->ds->kbd_put_keycode(0xe0);
    if (down)
	vs->ds->kbd_put_keycode(code & 0x7f);
    else
	vs->ds->kbd_put_keycode(code | 0x80);
}

static void framebuffer_set_updated_all(VncState *vs, int x, int y, int w, int h)
{

    int i;

    for (i = 0; i < MAX_CLIENTS; i++) {
	if (!VCS_ACTIVE(vs->vcs[i]))
	    continue;

	set_bits_in_row(vs, vs->vcs[i]->update_row, x, y, w, h);
    }

    vs->has_update = 1;
}

static void framebuffer_set_updated(struct VncClientState *vcs,
				    int x, int y, int w, int h)
{
    VncState *vs = vcs->vs;

    set_bits_in_row(vs, vcs->update_row, x, y, w, h);
    vs->has_update = 1;
}

static void framebuffer_update_request(struct VncClientState *vcs,
				       int incremental,
				       int x_position, int y_position,
				       int w, int h)
{
    struct VncState *vs = vcs->vs;
    if (!incremental)
	framebuffer_set_updated(vcs, x_position, y_position, w, h);
    /* XXX */
    vs->visible_x = 0;
    vs->visible_y = 0;
    vs->visible_w = vs->ds->width;
    vs->visible_h = vs->ds->height;
    vcs->update_requested++;
    dprintf("update requested on %d\n", vcs->csock);

    vs->ds->set_timer(vs->timer, vs->ds->get_clock());
}

static void set_encodings(struct VncClientState *vcs, int32_t *encodings,
			  size_t n_encodings)
{
    struct VncState *vs = vcs->vs;
    int i;

    vcs->has_hextile = 0;
    vcs->has_resize = 0;
    vcs->has_pointer_type_change = 0;
    vcs->has_cursor_encoding = 0;
    vcs->absolute = -1;

    for (i = n_encodings - 1; i >= 0; i--) {
	switch (encodings[i]) {
	case 0: /* Raw */
	    vcs->has_hextile = 0;
	    break;
	case 5: /* Hextile */
	    vcs->has_hextile = 1;
	    break;
	case -223: /* DesktopResize */
	    vcs->has_resize = 1;
	    break;
#if 0
	case -239: /* Cursor Pseud-Encoding */
	    vcs->has_cursor_encoding = 1;
	    break;
#endif
        case -254: /* xencenter */
            vcs->is_xencenter = 1;
            break;
        case -255: /* vncviewer */
            vcs->is_vncviewer = 1;
            break;
	case -257:
	    vcs->has_pointer_type_change = 1;
	    break;
	default:
	    break;
	}
    }

    check_pointer_type_change(vcs,
			      vs->ds->mouse_is_absolute(vs->ds->mouse_opaque));
}

static void set_pixel_format(struct VncClientState *vcs,
			     int bits_per_pixel, int depth,
			     int big_endian_flag, int true_color_flag,
			     int red_max, int green_max, int blue_max,
			     int red_shift, int green_shift, int blue_shift)
{
    struct VncState *vs = vcs->vs;
    int host_big_endian_flag;

#ifdef WORDS_BIGENDIAN
    host_big_endian_flag = 1;
#else
    host_big_endian_flag = 0;
#endif
    if (!true_color_flag) {
    fail:
	vnc_client_error(vcs);
        return;
    }
    dprintf("pixel format bpp %d depth %d vs depth %d\n", bits_per_pixel,
	    depth, vs->depth);
    if (bits_per_pixel == 32 && 
        host_big_endian_flag == big_endian_flag &&
        red_max == 0xff && green_max == 0xff && blue_max == 0xff &&
        red_shift == 16 && green_shift == 8 && blue_shift == 0 &&
        bits_per_pixel == vs->depth * 8) {
        vcs->write_pixels = vnc_write_pixels_copy;
        vcs->send_hextile_tile = send_hextile_tile_32;
        dprintf("set pixel format bpp %d depth %d copy\n", bits_per_pixel,
		vs->depth);
    } else if (bits_per_pixel == 16 && 
        host_big_endian_flag == big_endian_flag &&
        red_max == 31 && green_max == 63 && blue_max == 31 &&
        red_shift == 11 && green_shift == 5 && blue_shift == 0 &&
        bits_per_pixel == vs->depth * 8) {
        vcs->write_pixels = vnc_write_pixels_copy;
        vcs->send_hextile_tile = send_hextile_tile_16;
        dprintf("set pixel format bpp %d depth %d copy\n", bits_per_pixel,
		vs->depth);
    } else if (bits_per_pixel == 8 && 
        red_max == 7 && green_max == 7 && blue_max == 3 &&
        red_shift == 5 && green_shift == 2 && blue_shift == 0 &&
        bits_per_pixel == vs->depth * 8) {
        vcs->write_pixels = vnc_write_pixels_copy;
        vcs->send_hextile_tile = send_hextile_tile_8;
        dprintf("set pixel format bpp %d depth %d copy\n", bits_per_pixel,
		vs->depth);
    } else 
    {
        /* generic and slower case */
        if (bits_per_pixel != 8 &&
            bits_per_pixel != 16 &&
            bits_per_pixel != 32)
            goto fail;
        if (vcs->vs->depth == 4) {
            vcs->send_hextile_tile = send_hextile_tile_generic_32;
        } else if (vcs->vs->depth == 2) {
            vcs->send_hextile_tile = send_hextile_tile_generic_16;
        } else {
            vcs->send_hextile_tile = send_hextile_tile_generic_8;
        }
        vcs->pix_big_endian = big_endian_flag;
        vcs->write_pixels = vnc_write_pixels_generic;
        dprintf("set pixel format bpp %d depth %d generic\n", bits_per_pixel,
                vs->depth);
    }
    vcs->red_shift = red_shift;
    vcs->red_max = red_max;
    vcs->green_shift = green_shift;
    vcs->green_max = green_max;
    vcs->blue_shift = blue_shift;
    vcs->blue_max = blue_max;
    vcs->pix_bpp = bits_per_pixel / 8;
    dprintf("sending cursor %d for pixel format change\n", vcs->csock);
    vcs->vpm.vpm_cursor_update = 1;
    vnc_write_pending(vcs);
}

static int protocol_client_msg(struct VncClientState *vcs, uint8_t *data,
			       size_t len)
{
    struct VncState *vs = vcs->vs;
    int i;
    uint16_t limit;

    switch (data[0]) {
    case 0:
	if (len == 1)
	    return 20;

	set_pixel_format(vcs, read_u8(data, 4), read_u8(data, 5),
			 read_u8(data, 6), read_u8(data, 7),
			 read_u16(data, 8), read_u16(data, 10),
			 read_u16(data, 12), read_u8(data, 14),
			 read_u8(data, 15), read_u8(data, 16));
	break;
    case 2:
	if (len == 1)
	    return 4;

	if (len == 4) {
	    uint16_t v;
	    v = read_u16(data, 2);
	    if (v)
		return 4 + v * 4;
	}

	limit = read_u16(data, 2);
	for (i = 0; i < limit; i++) {
	    int32_t val = read_s32(data, 4 + (i * 4));
	    memcpy(data + 4 + (i * 4), &val, sizeof(val));
	}

	set_encodings(vcs, (int32_t *)(data + 4), limit);

	/* encodings available, immiedately update the cursor - if supported */
	if (VCS_ACTIVE(vcs)) {
	    dprintf("sending cursor %d for encodings change\n",
		    vcs->csock);
	    vcs->vpm.vpm_cursor_update = 1;
	    vnc_write_pending(vcs);
	}

	break;
    case 3:
	if (len == 1)
	    return 10;

	framebuffer_update_request(vcs, read_u8(data, 1), read_u16(data, 2),
				   read_u16(data, 4), read_u16(data, 6),
				   read_u16(data, 8));
	break;
    case 4:
	if (len == 1)
	    return 8;

	vs->timer_interval = VNC_REFRESH_INTERVAL_BASE;
	vs->ds->set_timer(vs->timer, vs->ds->get_clock() + vs->timer_interval);
	key_event(vs, read_u8(data, 1), read_u32(data, 4));
	break;
    case 5:
	if (len == 1)
	    return 6;

	vs->timer_interval = VNC_REFRESH_INTERVAL_BASE;
	vs->ds->set_timer(vs->timer, vs->ds->get_clock() + vs->timer_interval);
	pointer_event(vcs, read_u8(data, 1), read_u16(data, 2),
		      read_u16(data, 4));
	break;
    case 6:
	if (len == 1)
	    return 8;

	if (len == 8) {
	    uint32_t v;
	    v = read_u32(data, 4);
	    if (v)
		return 8 + v;
	}

	client_cut_text(vs, read_u32(data, 4), (char *)(data + 8));
	break;
    case 254: // Special case, sending keyboard scan codes
	if (len == 1)
	    return 8;

	vs->timer_interval = VNC_REFRESH_INTERVAL_BASE;
	vs->ds->set_timer(vs->timer, vs->ds->get_clock() + vs->timer_interval);
	scan_event(vs, read_u8(data, 1), read_u32(data, 4));
	break;
    default:
	dprintf("Msg: %d\n", data[0]);
	vnc_client_error(vcs);
	break;
    }

    vnc_read_when(vcs, protocol_client_msg, 1);
    return 0;
}

static int protocol_client_init(struct VncClientState *vcs, uint8_t *data,
				size_t len)
{
    struct VncState *vs = vcs->vs;
    size_t l;
    char pad[3] = { 0, 0, 0 };

    if (vs->ds->hw_update)
	vs->ds->hw_update(vs->ds->hw_opaque);

    dprintf("client init\n");
    vnc_write_u16(vcs, vs->ds->width);
    vnc_write_u16(vcs, vs->ds->height);
    vcs->width = vs->ds->width;
    vcs->height = vs->ds->height;

    vnc_write_u8(vcs, vs->depth * 8); /* bits-per-pixel */
    if (vs->depth == 4) vnc_write_u8(vcs, 24); /* depth */
    else  vnc_write_u8(vcs, vs->depth * 8);
#ifdef WORDS_BIGENDIAN
    vnc_write_u8(vcs, 1);             /* big-endian-flag */
#else
    vnc_write_u8(vcs, 0);             /* big-endian-flag */
#endif
    vnc_write_u8(vcs, 1);             /* true-color-flag */
    if (vs->depth == 4) {
	vnc_write_u16(vcs, 0xFF);     /* red-max */
	vnc_write_u16(vcs, 0xFF);     /* green-max */
	vnc_write_u16(vcs, 0xFF);     /* blue-max */
	vnc_write_u8(vcs, 16);        /* red-shift */
	vnc_write_u8(vcs, 8);         /* green-shift */
	vnc_write_u8(vcs, 0);         /* blue-shift */
        vcs->send_hextile_tile = send_hextile_tile_32;
    } else if (vs->depth == 2) {
	vnc_write_u16(vcs, 31);       /* red-max */
	vnc_write_u16(vcs, 63);       /* green-max */
	vnc_write_u16(vcs, 31);       /* blue-max */
	vnc_write_u8(vcs, 11);        /* red-shift */
	vnc_write_u8(vcs, 5);         /* green-shift */
	vnc_write_u8(vcs, 0);         /* blue-shift */
        vcs->send_hextile_tile = send_hextile_tile_16;
    } else if (vs->depth == 1) {
        /* XXX: change QEMU pixel 8 bit pixel format to match the VNC one ? */
	vnc_write_u16(vcs, 7);        /* red-max */
	vnc_write_u16(vcs, 7);        /* green-max */
	vnc_write_u16(vcs, 3);        /* blue-max */
	vnc_write_u8(vcs, 5);         /* red-shift */
	vnc_write_u8(vcs, 2);         /* green-shift */
	vnc_write_u8(vcs, 0);         /* blue-shift */
        vcs->send_hextile_tile = send_hextile_tile_8;
    }
    vcs->write_pixels = vnc_write_pixels_copy;

    vnc_write(vcs, pad, 3);           /* padding */

    l = strlen(vs->title); 
    vnc_write_u32(vcs, l);
    vnc_write(vcs, vs->title, l);

    vnc_flush(vcs);

    vnc_read_when(vcs, protocol_client_msg, 1);

    return 0;
}

static int protocol_response(struct VncClientState *vcs,
			     uint8_t *client_response, size_t len)
{
    unsigned char cryptchallenge[AUTHCHALLENGESIZE];
    unsigned char key[8];
    int passwdlen, i, j;

    memcpy(cryptchallenge, challenge, AUTHCHALLENGESIZE);

    /* Calculate the sent challenge */
    passwdlen = strlen(vncpasswd);
    for (i=0; i<8; i++)
	key[i] = i<passwdlen ? vncpasswd[i] : 0;
    deskey(key, EN0);
    for (j = 0; j < AUTHCHALLENGESIZE; j += 8)
	des(cryptchallenge+j, cryptchallenge+j);

    /* Check the actual response */
    if (memcmp(cryptchallenge, client_response, AUTHCHALLENGESIZE) != 0) {
	/* password error */
	vnc_write_u32(vcs, 1);
	vnc_write_u32(vcs, 22);
	vnc_write(vcs, "Authentication failure", 22);
	vnc_flush(vcs);
	fprintf(stderr, "VNC Password error.\n");
	vnc_client_error(vcs);
	return 0;
    }

    dprintf("protocol response\n");
    vnc_write_u32(vcs, 0);
    vnc_flush(vcs);

    vnc_read_when(vcs, protocol_client_init, 1);

    return 0;
}

static int protocol_version(struct VncClientState *vcs, uint8_t *version,
			    size_t len)
{
    extern char vncpasswd[64];
    extern unsigned char challenge[AUTHCHALLENGESIZE];
    char local[13];
    int  support, maj, min;

    memcpy(local, version, 12);
    local[12] = 0;

    /* protocol version check */
    if (sscanf(local, "RFB %03d.%03d\n", &maj, &min) != 2) {
	fprintf(stderr, "Protocol version error.\n");
	vnc_client_error(vcs);
	return 0;
    }

    support = 0;
    if (maj == 3) {
	if (min == 3 || min ==4) {
	    support = 1;
	}
    }

    if (! support) {
	fprintf(stderr, "Client uses unsupported protocol version %d.%d.\n",
		maj, min);
	vnc_client_error(vcs);
	return 0;
    }

    dprintf("authentication\n");
    if (*vncpasswd == '\0') {
	/* AuthType is None */
	vnc_write_u32(vcs, 1);
	vnc_flush(vcs);
	vnc_read_when(vcs, protocol_client_init, 1);
    } else {
	/* AuthType is VncAuth */
	vnc_write_u32(vcs, 2);

	/* Challenge-Responce authentication */
	/* Send Challenge */
	make_challenge(challenge, AUTHCHALLENGESIZE);
	vnc_write(vcs, challenge, AUTHCHALLENGESIZE);
	vnc_flush(vcs);
	vnc_read_when(vcs, protocol_response, AUTHCHALLENGESIZE);
    }

    return 0;
}

static void vnc_listen_read(void *opaque)
{
    VncState *vs = opaque;
    struct VncClientState *vcs;
    struct sockaddr_in addr;
    socklen_t addrlen = sizeof(addr);
    int new_sock;
    int i;

    new_sock = accept(vs->lsock, (struct sockaddr *)&addr, &addrlen);
    if (new_sock == -1)
	return;

    for (i = 0; i < MAX_CLIENTS; i++)
	if (!VCS_INUSE(vs->vcs[i]))
	    break;
    if (i == MAX_CLIENTS)
	goto fail;

    if (vs->vcs[i] == NULL) {
	vs->vcs[i] = calloc(1, sizeof(struct VncClientState));
	if (vs->vcs[i] == NULL)
	    goto fail;
    }

    vcs = vs->vcs[i];
    vcs->vs = vs;
    vs->ds->idle = 0;
    vcs->update_row = qemu_reallocz(vcs->update_row, vs->ds->height *
				    sizeof(vcs->update_row[0]));
    vcs->csock = new_sock;
#ifndef _WIN32
    socket_set_nonblock(vcs->csock);
#endif
    vs->ds->set_fd_handler(vcs->csock, NULL, vnc_client_read, NULL, vcs);
    dprintf("rfb greeting\n");
    vnc_write(vcs, "RFB 003.003\n", 12);
    vnc_flush(vcs);
    vnc_read_when(vcs, protocol_version, 12);
    vcs->is_vncviewer = 0;
    vcs->is_xencenter = 0;
    vcs->has_resize = 0;
    vcs->has_hextile = 0;
    vcs->last_x = -1;
    vcs->last_y = -1;
    vnc_timer_init(vs);		/* XXX */
    return;

 fail:
    closesocket(new_sock);
    return;
}

int vnc_display_init(DisplayState *ds, struct sockaddr *addr,
		     int find_unused, char *title, char *keyboard_layout)
{
    struct sockaddr_in *iaddr = NULL;
    int reuse_addr, ret;
    socklen_t addrlen;
    VncState *vs;

    vs = qemu_mallocz(sizeof(VncState));
    if (!vs)
	exit(1);

    ds->opaque = vs;
#if 0
    vnc_state = vs;
    vs->display = arg;
#endif

    vs->lsock = -1;
    if (!ds->dpy_update_sdl) {
        ds->depth = 32;
        ds->idle = 1;
    }
    vs->depth = ds->depth / 8;
    switch (ds->depth) {
        case 8:
            vs->depth = 1;
            vs->red_max1 = 7;
            vs->green_max1 = 7;
            vs->blue_max1 = 3;
            vs->red_shift1 = 5;
            vs->green_shift1 = 2;
            vs->blue_shift1 = 0;
            break;
        case 16:
            vs->depth = 2;
            vs->red_max1 = 31;
            vs->green_max1 = 63;
            vs->blue_max1 = 31;
            vs->red_shift1 = 11;
            vs->green_shift1 = 5;
            vs->blue_shift1 = 0;
            break;
        case 32:
            vs->depth = 4;
            vs->red_max1 = 255;
            vs->green_max1 = 255;
            vs->blue_max1 = 255;
            vs->red_shift1 = 16;
            vs->green_shift1 = 8;
            vs->blue_shift1 = 0;
            break;
        default:
            break;
    }
    vs->ds = ds;

    memset(vs->vcs, 0, sizeof(vs->vcs));

    if (!keyboard_layout)
	keyboard_layout = "en-us";

    vs->kbd_layout_name = strdup(keyboard_layout);
    vs->kbd_layout = init_keyboard_layout(keyboard_layout);
    if (!vs->kbd_layout)
	exit(1);
    vs->modifiers_state[0x45] = 1; /* NumLock on - on boot */

    vs->title = strdup(title ?: "");

    vs->ds->dpy_update = vnc_dpy_update;
    vs->ds->dpy_resize = vnc_dpy_resize;
    if (ds->dpy_resize_sdl)
        vs->ds->dpy_resize_shared = NULL;
    else
        vs->ds->dpy_resize_shared = vnc_dpy_resize_shared;
    vs->ds->dpy_refresh = vnc_dpy_refresh;
    vs->ds->dpy_set_server_text = vnc_set_server_text;
    vs->ds->dpy_bell = vnc_send_bell;

    if (ds->dpy_resize_sdl)
        vnc_dpy_resize_shared(ds, ds->width, ds->height, ds->depth, ds->linesize, ds->data);
    else
        vnc_dpy_resize(vs->ds, 640, 400);

#if !defined(_WIN32) && !defined(__MINIOS__) 
    if (addr->sa_family == AF_UNIX) {
	addrlen = sizeof(struct sockaddr_un);

	vs->lsock = socket(PF_UNIX, SOCK_STREAM, 0);
	if (vs->lsock == -1) {
	    fprintf(stderr, "Could not create socket\n");
	    exit(1);
	}
    } else
#endif
    if (addr->sa_family == AF_INET) {
	iaddr = (struct sockaddr_in *)addr;
	addrlen = sizeof(struct sockaddr_in);

	vs->lsock = socket(PF_INET, SOCK_STREAM, 0);
	if (vs->lsock == -1) {
	    fprintf(stderr, "Could not create socket\n");
	    exit(1);
	}

	iaddr->sin_port = htons(ntohs(iaddr->sin_port) + 5900);

	reuse_addr = 1;
	ret = setsockopt(vs->lsock, SOL_SOCKET, SO_REUSEADDR,
			 (const char *)&reuse_addr, sizeof(reuse_addr));
	if (ret == -1) {
	    fprintf(stderr, "setsockopt() failed\n");
	    exit(1);
	}
    } else {
	fprintf(stderr, "Invalid socket family %x\n", addr->sa_family);
	exit(1);
    }

    while (bind(vs->lsock, addr, addrlen) == -1) {
	if (errno == EADDRINUSE && find_unused && addr->sa_family == AF_INET) {
	    iaddr->sin_port = htons(ntohs(iaddr->sin_port) + 1);
	    continue;
	}
	fprintf(stderr, "bind() failed\n");
	exit(1);
    }

    if (listen(vs->lsock, 1) == -1) {
	fprintf(stderr, "listen() failed\n");
	exit(1);
    }

    ret = vs->ds->set_fd_handler(vs->lsock, NULL, vnc_listen_read, NULL, vs);
    if (ret == -1)
	exit(1);

    if (addr->sa_family == AF_INET)
	return ntohs(iaddr->sin_port);
    else
	return 0;
}

int vnc_start_viewer(int port)
{
#ifndef _WIN32
    int pid, i, open_max;
    char s[16];

    sprintf(s, ":%d", port);

    switch (pid = fork()) {
    case -1:
	fprintf(stderr, "vncviewer failed fork\n");
	exit(1);

    case 0:	/* child */
	open_max = sysconf(_SC_OPEN_MAX);
	for (i = 0; i < open_max; i++)
	    if (i != STDIN_FILENO &&
		i != STDOUT_FILENO &&
		i != STDERR_FILENO)
		close(i);
	execlp("vncviewer", "vncviewer", s, NULL);
	fprintf(stderr, "vncviewer execlp failed\n");
	exit(1);

    default:
	return pid;
    }
#else
    fprintf(stderr, "vncviewer start not supported\n");
    exit(1);
#endif
}

unsigned int seed;

static int make_challenge(unsigned char *random, int size)
{
 
    set_seed(&seed);
    get_random(size, random);

    return 0;
}

static void set_seed(unsigned int *seedp)
{
    *seedp += (unsigned int)(time(NULL)+getpid()+getpid()*987654+rand());
    srand(*seedp);

    return;
}

static void get_random(int len, unsigned char *buf)
{
    int i;

    for (i=0; i<len; i++)
	buf[i] = (int) (256.0*rand()/(RAND_MAX+1.0));

    return;
}

void vnc_keymap_change(DisplayState *ds, char *keymap)
{
    VncState *vs = ds->opaque;
    kbd_layout_t *new_layout;

    if (!strcmp(keymap, vs->kbd_layout_name))
        return;

    new_layout = init_keyboard_layout(keymap);
    if (!new_layout) {
        fprintf(stderr, "Failed to initialise new keyboard layout\n");
        return;
    }

    fprintf(stderr, "Initialise new keyboard layout %s\n", keymap);

    free(vs->kbd_layout_name);
    free_keyboard_layout(vs->kbd_layout);

    vs->kbd_layout_name = strdup(keymap);
    vs->kbd_layout = new_layout;
}

