/*
 * This file is subject to the terms and conditions of the GNU General
 * Public License.  See the file "COPYING" in the main directory of
 * this archive for more details.
 *
 * Copyright (C) 2006 Christian Limpach
 * Copyright (C) 2006 XenSource Ltd.
 *
 */

#include "qemu-common.h"
#include "qemu-char.h"

#include "block_int.h"
#include <unistd.h>
#if !defined(QEMU_KXEN)
#ifndef CONFIG_STUBDOM
#include <sys/ipc.h>
#include <sys/shm.h>
#endif
#endif
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <assert.h>
#if !defined(_WIN32)
#include <err.h>
#endif

#include "exec-all.h"
#include "sysemu.h"

#include "hw.h"
#include "pci.h"
#include "console.h"
#include "qemu-timer.h"
#include "qemu-xen.h"
#include "privsep.h"


#if !defined(QEMU_KXEN)
/* The token used to identify the set_clipboard watch. */
#define SET_CLIPBOARD_TOKEN "set_clipboard"

/* The token used to identify the report_clipboard watch. */
#define REPORT_CLIPBOARD_TOKEN "report_clipboard"

/* The path to the keymap declaration, inside /local/domain/<domid>. */
#define KEYMAP_PATH "keymap"

/* The token used to identify the keymap watch. */
#define KEYMAP_TOKEN "keymap"

struct xs_handle *xsh = NULL;
static char *media_filename[MAX_DRIVES+1];
static QEMUTimer *insert_timer = NULL;
static char *xenbus_param_paths[MAX_DRIVES+1];

static struct {
    int need_reset;
    char *data;
    int off;
} set_clipboard_state;
#endif
 
int xenstore_find_device(BlockDriverState *bs)
{
#if !defined(QEMU_KXEN)
    int i;

    for (i = 0; i < MAX_DRIVES + 1; i++) {
	if (drives_table[i].bdrv == bs)
	    return i;
    }
#endif
    return -1;
}

void xenstore_do_eject(BlockDriverState *bs)
{
#if !defined(QEMU_KXEN)
    int i;

    i = xenstore_find_device(bs);
    if (i == -1) {
	fprintf(logfile, "couldn't find disk to eject.\n");
	return;
    }
    privsep_eject_cd(i);
#endif
}

void xenstore_set_device_locked(BlockDriverState *bs)
{
#if !defined(QEMU_KXEN)
    int i;

    i = xenstore_find_device(bs);
    if (i == -1) {
	fprintf(logfile, "xenstore_set_device_locked: couldn't find disk.\n");
	return;
    }
    if (bs->locked)
        privsep_lock_cd(i);
    else
        privsep_unlock_cd(i);
#endif
}

int
xenstore_pv_driver_build_blacklisted(uint16_t product_nr,
                                     uint32_t build_nr)
{
#if !defined(QEMU_KXEN)
    char *buf = NULL;
    char *tmp;
    const char *product;

    switch (product_nr) {
    case 1:
        product = "xensource-windows";
        break;
    default:
        /* Don't know what product this is -> we can't blacklist
         * it. */
        return 0;
    }
    if (asprintf(&buf, "/mh/driver-blacklist/%s/%d", product, build_nr) < 0)
        return 0;
    tmp = xs_read(xsh, XBT_NULL, buf, NULL);
    free(tmp);
    free(buf);
    if (tmp == NULL)
        return 0;
    else
        return 1;
#endif
    return 0;			/* XXX */
}

#if !defined(QEMU_KXEN)
#define UWAIT_MAX (30*1000000) /* thirty seconds */
#define UWAIT     (100000)     /* 1/10th second  */

static int pasprintf(char **buf, const char *fmt, ...)
{
    va_list ap;
    int ret = 0;

    if (*buf)
        free(*buf);
    va_start(ap, fmt);
    if (vasprintf(buf, fmt, ap) == -1) {
        buf = NULL;
        ret = -1;
    }
    va_end(ap);
    return ret;
}

static void insert_media(void *opaque)
{
    int i;
    BlockDriverState *bs;

    for (i = 0; i < MAX_DRIVES + 1; i++) {
        bs = drives_table[i].bdrv;
        if (media_filename[i] && bs && bs->filename[0] == '\0') {
            bdrv_open2(bs, media_filename[i], 0, &bdrv_raw);
            pstrcpy(bs->filename, sizeof(bs->filename), media_filename[i]);
            free(media_filename[i]);
            media_filename[i] = NULL;
        }
    }
}
#endif

void xenstore_check_new_media_present(int timeout)
{
#if !defined(QEMU_KXEN)

    if (insert_timer == NULL)
        insert_timer = qemu_new_timer(rt_clock, insert_media, NULL);
    qemu_mod_timer(insert_timer, qemu_get_clock(rt_clock) + timeout);
#endif
}

#if !defined(QEMU_KXEN)
static void waitForDevice(char *fn)
{ 
    struct stat sbuf;
    int status;
    int uwait = UWAIT_MAX;

    do {
        status = stat(fn, &sbuf);
        if (!status) break;
        usleep(UWAIT);
        uwait -= UWAIT;
    } while (uwait > 0);

    return;
}

static int any_hdN;

static int parse_drive_name(const char *dev, DriveInfo *out) {
    /* alway sleaves out->bdrv unchanged */
    /* on success, returns 0 and fills in out->type, ->bus, ->unit */
    /* if drive name not understood, returns -1 and *out may be garbage */
    int ch, max, per_bus;

    if (!strncmp(dev, "fd", 2) && strlen(dev) == 3) {
        ch = dev[2];
        out->type = IF_FLOPPY;
        max = 2;
        per_bus = 2;
    } else {
        /* Change xvdN to look like hdN */            ch = dev[3];
        if (!any_hdN && !strncmp(dev, "xvd", 3) && strlen(dev) == 4) {
            ch = dev[3];
            fprintf(logfile, "Using %s for guest's hd%c\n", dev, ch);
            out->type = IF_IDE;
        } else if (!strncmp(dev, "hd", 2) && strlen(dev) == 3) {
            ch = dev[2];
            out->type = IF_IDE;
        } else if (!strncmp(dev, "sd", 2) && strlen(dev) == 3) {
            ch = dev[2];
            out->type = IF_SCSI;
        } else {
            fprintf(stderr, "qemu: ignoring not-understood drive `%s'\n", dev);
            return -1;
        }

        if (out->type == IF_SCSI) {
            max = MAX_SCSI_DEVS;
            per_bus = max;
        } else {
            max = 4;
            per_bus = 2;
        }
    }

    ch = ch - 'a';
    if (ch >= max) {
        fprintf(stderr, "qemu: drive `%s' out of range\n", dev);
        return -1;
    }

    out->bus = ch / per_bus;
    out->unit = ch % per_bus;
    
    return 0;
}       

static int drive_name_to_index(const char *name) {
    DriveInfo tmp;
    int ret;

    ret = parse_drive_name(name, &tmp);  if (ret) return -1;
    ret = drive_get_index(tmp.type, tmp.bus, tmp.unit);
    return ret;
}
#endif

#define DIRECT_PCI_STR_LEN 160
char direct_pci_str[DIRECT_PCI_STR_LEN];
void xenstore_parse_domain_config(int hvm_domid)
{
#if !defined(QEMU_KXEN)
    char **e = NULL;
    char *buf = NULL, *path;
    char *fpath = NULL, *bpath = NULL,
        *dev = NULL, *params = NULL, *type = NULL, *mode = NULL, *drv = NULL;
    char *name = NULL;
    char *serial_c;
    char serial_temp[PATH_MAX - 5];
    int i, any_hdN = 0, ret;
    unsigned int len, num, hd_index, pci_devid = 0;
    BlockDriverState *bs;
    BlockDriver *format;

    for(i = 0; i < MAX_DRIVES + 1; i++)
        media_filename[i] = NULL;

    xsh = xs_daemon_open();
    if (xsh == NULL) {
        fprintf(logfile, "Could not contact xenstore for domain config\n");
        return;
    }

    name = xenstore_vm_read(domid, "uuid", NULL);
    if (name) {
	if (snprintf(domain_name, DOMAIN_NAME_LEN - 1, "HVMXEN-%s",
		     name) > 0) {
	    domain_name[DOMAIN_NAME_LEN - 1] = 0;
            cpu_set_log_syslog(domain_name);
	    cpu_set_log(0);
	}
    }

    /* read the console output filename */
    serial_c = xs_read(xsh, XBT_NULL, "/local/logconsole/@", NULL);
    if (serial_c) {
        serial_devices[0] = qemu_mallocz(PATH_MAX);
	strcpy(serial_devices[0], "file:");
	if (strstr(serial_c, "%s"))
	    snprintf(serial_temp, sizeof(serial_temp), serial_c, name);
	else
	    snprintf(serial_temp, sizeof(serial_temp), serial_c, domid);
	strncat(serial_devices[0], serial_temp, sizeof(serial_temp));
	free(serial_c);
    }
    /* we don't care if it failed or not */

    if (name)
 	free(name);

    path = xs_get_domain_path(xsh, hvm_domid);
    if (path == NULL) {
        fprintf(logfile, "xs_get_domain_path() error\n");
        goto out;
    }

    if (pasprintf(&buf, "%s/" KEYMAP_PATH, path) == -1)
	goto out;

    xs_watch(xsh, buf, KEYMAP_TOKEN); // Ignore failure -- we can muddle on.

    if (pasprintf(&buf, "%s/device/vbd", path) == -1)
        goto out;

    e = xs_directory(xsh, XBT_NULL, buf, &num);
    if (e == NULL)
        num = 0;

    for (i = 0; i < num; i++) {
        /* read the backend path */
        if (pasprintf(&buf, "%s/device/vbd/%s/backend", path, e[i]) == -1)
            continue;
        free(bpath);
        bpath = xs_read(xsh, XBT_NULL, buf, &len);
        if (bpath == NULL)
            continue;
        /* read the name of the device */
        if (pasprintf(&buf, "%s/dev", bpath) == -1)
            continue;
        free(dev);
        dev = xs_read(xsh, XBT_NULL, buf, &len);
        if (dev == NULL)
            continue;
        if (!strncmp(dev, "hd", 2)) {
            any_hdN = 1;
            break;
        }
    }
        
    for (i = 0; i < num; i++) {
	format = NULL; /* don't know what the format is yet */

        /* read the backend path */
        if (pasprintf(&buf, "%s/device/vbd/%s/backend", path, e[i]) == -1)
            continue;
        free(bpath);
        bpath = xs_read(xsh, XBT_NULL, buf, &len);
        if (bpath == NULL)
            continue;
        /* read the name of the device */
        if (pasprintf(&buf, "%s/dev", bpath) == -1)
            continue;
        free(dev);
        dev = xs_read(xsh, XBT_NULL, buf, &len);
        if (dev == NULL)
            continue;
	if (nb_drives >= MAX_DRIVES) {
	    fprintf(stderr, "qemu: too many drives, skipping `%s'\n", dev);
	    continue;
	}
	ret = parse_drive_name(dev, &drives_table[nb_drives]);
	if (ret)
	    continue;
        /* read the type of the device */
        if (pasprintf(&buf, "%s/device/vbd/%s/device-type", path, e[i]) == -1)
            continue;
        free(type);
        type = xs_read(xsh, XBT_NULL, buf, &len);

	free(params);
	params = NULL;
	if (pasprintf(&buf, "%s/type", bpath) == -1)
	    continue;
	free(drv);
	drv = xs_read(xsh, XBT_NULL, buf, NULL);
	fprintf(logfile, "dev %s drv %s\n", e[i], drv);
	if (drv && !strcmp(drv, "tap")) {
	    /* 
	     * check if device has a phantom vbd; the phantom is hooked
	     * to the frontend device (for ease of cleanup), so lookup 
	     * the frontend device, and see if there is a phantom_vbd
	     * if there is, we will use resolution as the filename
	     */
	    if (pasprintf(&buf, "%s/device/vbd/%s/phantom_vbd", path, e[i]) ==
		-1)
		continue;
	    free(fpath);
	    fpath = xs_read(xsh, XBT_NULL, buf, &len);
	    if (fpath) {
		if (pasprintf(&buf, "%s/dev", fpath) == -1)
		    continue;
		params = xs_read(xsh, XBT_NULL, buf , &len);
		if (params) {
		    /* 
		     * wait for device, on timeout silently fail
		     * because we will fail to open below
		     */
		    waitForDevice(params);
		}
	    }
	    if (params == NULL) {
		if (pasprintf(&buf, "%s/backdev-path", bpath) == -1)
		    continue;
		fprintf(logfile, "=== path %s\n", buf);
		while (params == NULL) {
		    params = xs_read(xsh, XBT_NULL, buf, &len);
		    if (params == NULL)
			usleep(UWAIT);
		}
		fprintf(logfile, "=== params %s\n", params);
	    }
	    format = &bdrv_raw;
        }
	if (params == NULL) {
	    if (pasprintf(&buf, "%s/params", bpath) == -1)
		continue;
	    free(params);
	    params = xs_read(xsh, XBT_NULL, buf, &len);
	}

        /* Prefix with /dev/ if needed */
        if (!strcmp(drv, "phy") && params[0] != '/') {
            char *newparams = malloc(5 + strlen(params) + 1);
            sprintf(newparams, "/dev/%s", params);
            free(params);
            params = newparams;
	    format = &bdrv_raw;
        }
	free(mode);
	if (pasprintf(&buf, "%s/mode", bpath) == -1)
	    mode = NULL;
	else
	    mode = xs_read(xsh, XBT_NULL, buf, &len);

        bs = bdrv_new(dev);
        if (type) {
            if (!strcmp(type, "cdrom"))
                bdrv_set_type_hint(bs, BDRV_TYPE_CDROM);
            if (!strcmp(type, "floppy"))
                bdrv_set_type_hint(bs, BDRV_TYPE_FLOPPY);
            if (!strcmp(type, "cdrom") || !strcmp(type, "floppy")) {
		if (pasprintf(&buf, "%s/params", bpath) != -1) {
		    privsep_set_cd_backend(nb_drives, bpath);
                    xs_watch(xsh, buf, dev);
		    xenbus_param_paths[nb_drives] = strdup(buf);
		}
            }
        }

        if (mode && strchr(mode, 'w') == NULL)
            bdrv_set_read_only(bs);

        /* open device now if media present */
#ifdef CONFIG_STUBDOM
        if (pasprintf(&buf, "%s/device/vbd/%s", path, e[i]) == -1)
            continue;
	if (bdrv_open2(bs, buf, 0 /* snapshot */, &bdrv_vbd) == 0) {
	    pstrcpy(bs->filename, sizeof(bs->filename), params);
	} else
#endif

        if (params[0]) {
	    if (!format) {
		if (!drv) {
		    fprintf(stderr, "qemu: type (image format) not specified for vbd '%s' or image '%s'\n", buf, params);
		    continue;
		}
		if (!strcmp(drv,"qcow")) {
		    /* autoguess qcow vs qcow2 */
		} else if (!strcmp(drv,"file")) {
		    format = &bdrv_raw;
		} else if (!strcmp(drv,"phy")) {
		    format = &bdrv_raw;
		} else {
		    format = bdrv_find_format(drv);
		    if (!format) {
			fprintf(stderr, "qemu: type (image format) '%s' unknown for vbd '%s' or image '%s'\n", drv, buf, params);
			continue;
		    }
		}
	    }
            pstrcpy(bs->filename, sizeof(bs->filename), params);
            if (bdrv_open2(bs, params, 0 /* snapshot */, format) < 0)
                fprintf(stderr, "qemu: could not open vbd '%s' or hard disk image '%s' (drv '%s' format '%s')\n", buf, params, drv ? drv : "?", format ? format->format_name : "0");
        }

        drives_table[nb_drives].bdrv = bs;
        if (!strcmp(type, "floppy")) {
           drives_table[nb_drives].type = IF_FLOPPY;
           drives_table[nb_drives].unit = 0;
        }
        nb_drives++;
    }

#ifdef CONFIG_STUBDOM
    if (pasprintf(&buf, "%s/device/vkbd", path) == -1)
        goto out;

    free(e);
    e = xs_directory(xsh, XBT_NULL, buf, &num);

    if (e) {
        for (i = 0; i < num; i++) {
            if (pasprintf(&buf, "%s/device/vkbd/%s", path, e[i]) == -1)
                continue;
            xenfb_connect_vkbd(buf);
        }
    }

    if (pasprintf(&buf, "%s/device/vfb", path) == -1)
        goto out;

    free(e);
    e = xs_directory(xsh, XBT_NULL, buf, &num);

    if (e) {
        for (i = 0; i < num; i++) {
            if (pasprintf(&buf, "%s/device/vfb/%s", path, e[i]) == -1)
                continue;
            xenfb_connect_vfb(buf);
        }
    }
#endif


    /* Set a watch for log-dirty requests from the migration tools */
    if (pasprintf(&buf, "%s/logdirty/next-active", path) != -1) {
        xs_watch(xsh, buf, "logdirty");
        fprintf(logfile, "Watching %s\n", buf);
    }

    /* Set a watch for suspend requests from the migration tools */
    if (pasprintf(&buf, 
                  "/local/domain/0/device-model/%u/command", domid) != -1) {
        xs_watch(xsh, buf, "dm-command");
        fprintf(logfile, "Watching %s\n", buf);
    }
    xs_watch(xsh, "@releaseDomain", "releaseDomain");

    if (pasprintf(&buf, "%s/data/set_clipboard", path) == -1)
        goto out;
    xs_watch(xsh, buf, SET_CLIPBOARD_TOKEN);

    if (pasprintf(&buf, "%s/data/report_clipboard", path) == -1)
        goto out;
    xs_watch(xsh, buf, REPORT_CLIPBOARD_TOKEN);


    /* get the pci pass-through parameter */
    if (pasprintf(&buf, "/local/domain/0/backend/pci/%u/%u/num_devs",
                  hvm_domid, pci_devid) == -1)
        goto out;

    free(params);
    params = xs_read(xsh, XBT_NULL, buf, &len);
    if (params == NULL)
        goto out;
    num = atoi(params);

    for ( i = 0; i < num; i++ ) {
        if (pasprintf(&buf, "/local/domain/0/backend/pci/%u/%u/dev-%d",
                    hvm_domid, pci_devid, i) != -1) {
            free(dev);
            dev = xs_read(xsh, XBT_NULL, buf, &len);

            if ( strlen(dev) + strlen(direct_pci_str) > DIRECT_PCI_STR_LEN ) {
                fprintf(stderr, "qemu: too many pci pass-through devices\n");
                memset(direct_pci_str, 0, DIRECT_PCI_STR_LEN);
                goto out;
            }

            /* append to direct_pci_str */
            if ( dev ) {
                strcat(direct_pci_str, dev);
                strcat(direct_pci_str, "-");
            }
        }
    }

 out:
    free(mode);
    free(type);
    free(params);
    free(dev);
    free(bpath);
    free(buf);
    free(path);
    free(e);
    free(drv);
#endif
    return;
}

int xenstore_fd(void)
{
#if !defined(QEMU_KXEN)
    if (xsh)
        return xs_fileno(xsh);
#endif
    return -1;
}

#if !defined(QEMU_KXEN)
unsigned long *logdirty_bitmap = NULL;
unsigned long logdirty_bitmap_size;
extern int vga_ram_size, bios_size;

void xenstore_process_logdirty_event(void)
{
#ifdef CONFIG_STUBDOM
    /* XXX we just can't use shm. */
    return;
#else
    char *act;
    static char *active_path = NULL;
    static char *next_active_path = NULL;
    static char *seg = NULL;
    unsigned int len;
    int i;

    fprintf(logfile, "Triggered log-dirty buffer switch\n");

    if (!seg) {
        char *path, *p, *key_ascii, key_terminated[17] = {0,};
        key_t key;
        int shmid;

        /* Find and map the shared memory segment for log-dirty bitmaps */
        if (!(path = xs_get_domain_path(xsh, domid))) {            
            fprintf(logfile, "Log-dirty: can't get domain path in store\n");
            exit(1);
        }
        if (!(path = realloc(path, strlen(path) 
                             + strlen("/logdirty/next-active") + 1))) {
            fprintf(logfile, "Log-dirty: out of memory\n");
            exit(1);
        }
        strcat(path, "/logdirty/");
        p = path + strlen(path);
        strcpy(p, "key");
        
        key_ascii = xs_read(xsh, XBT_NULL, path, &len);
        if (!key_ascii) {
            /* No key yet: wait for the next watch */
            free(path);
            return;
        }
        strncpy(key_terminated, key_ascii, 16);
        free(key_ascii);
        key = (key_t) strtoull(key_terminated, NULL, 16);

        /* Figure out how bit the log-dirty bitmaps are */
        logdirty_bitmap_size = xc_memory_op(xc_handle, 
                                            XENMEM_maximum_gpfn, &domid) + 1;
        logdirty_bitmap_size = ((logdirty_bitmap_size + HOST_LONG_BITS - 1)
                                / HOST_LONG_BITS); /* longs */
        logdirty_bitmap_size *= sizeof (unsigned long); /* bytes */

        /* Map the shared-memory segment */
        if ((shmid = shmget(key, 
                            2 * logdirty_bitmap_size, 
                            S_IRUSR|S_IWUSR)) == -1 
            || (seg = shmat(shmid, NULL, 0)) == (void *)-1) {
            fprintf(logfile, "Log-dirty: can't map segment %16.16llx (%s)\n",
                    (unsigned long long) key, strerror(errno));
            exit(1);
        }

        fprintf(logfile, "Log-dirty: mapped segment at %p\n", seg);

        /* Double-check that the bitmaps are the size we expect */
        if (logdirty_bitmap_size != *(uint32_t *)seg) {
            fprintf(logfile, "Log-dirty: got %u, calc %lu\n", 
                    *(uint32_t *)seg, logdirty_bitmap_size);
            return;
        }

        /* Remember the paths for the next-active and active entries */
        strcpy(p, "active");
        if (!(active_path = strdup(path))) {
            fprintf(logfile, "Log-dirty: out of memory\n");
            exit(1);
        }
        strcpy(p, "next-active");
        if (!(next_active_path = strdup(path))) {
            fprintf(logfile, "Log-dirty: out of memory\n");
            exit(1);
        }
        free(path);
    }
    
    /* Read the required active buffer from the store */
    act = xs_read(xsh, XBT_NULL, next_active_path, &len);
    if (!act) {
        fprintf(logfile, "Log-dirty: can't read next-active\n");
        exit(1);
    }

    /* Switch buffers */
    i = act[0] - '0';
    if (i != 0 && i != 1) {
        fprintf(logfile, "Log-dirty: bad next-active entry: %s\n", act);
        exit(1);
    }
    logdirty_bitmap = (unsigned long *)(seg + i * logdirty_bitmap_size);

    /* Ack that we've switched */
    privsep_ack_logdirty_switch(act);
    free(act);
#endif
}


/* Accept state change commands from the control tools */
static void xenstore_process_dm_command_event(void)
{
    char *command = NULL, *par = NULL;
    unsigned int len;

    command = privsep_read_dm("command");
    if (!command)
        goto out;
    len = strlen(command);
    
    if (!strncmp(command, "save", len)) {
        fprintf(logfile, "dm-command: pause and save state\n");
        xen_pause_requested = 1;
    } else if (!strncmp(command, "continue", len)) {
        fprintf(logfile, "dm-command: continue after state save\n");
        xen_pause_requested = 0;
#ifdef CONFIG_PASSTHROUGH
    } else if (!strncmp(command, "pci-rem", len)) {
        fprintf(logfile, "dm-command: hot remove pass-through pci dev \n");

        par = privsep_read_dm("parameter");
        if (!par)
            goto out;

        do_pci_del(par);
        free(par);
    } else if (!strncmp(command, "pci-ins", len)) {
        fprintf(logfile, "dm-command: hot insert pass-through pci dev \n");

        par = privsep_read_dm("parameter");
        if (!par)
            goto out;

        do_pci_add(par);
        free(par);
#endif
    } else {
        fprintf(logfile, "dm-command: unknown command\"%*s\"\n", len, command);
    }

 out:
    free(command);
}
#endif

void xenstore_record_dm_state(char *state)
{
#if !defined(QEMU_KXEN)
    privsep_record_dm("state", state);
#endif
}

#if !defined(QEMU_KXEN)
static void set_clipboard_event(void)
{
    char *old;
    int this_time;
    char *d_path, *cb_path;

    /* Something happened on the state machine used for setting the
       guest clipboard. */

    cb_path = NULL;
    d_path = xs_get_domain_path(xsh, domid);
    if (d_path) {
        if (asprintf(&cb_path, "%s/data/set_clipboard", d_path) == -1)
            cb_path = NULL;
    }
    free(d_path);
    if (!cb_path)
        return;

    old = xs_read(xsh, XBT_NULL, cb_path, NULL);
    if (old != NULL) {
        /* Still waiting for the guest to ack the last operation. */
        free(old);
        free(cb_path);
        return;
    }

    if (set_clipboard_state.need_reset) {
        xs_write(xsh, XBT_NULL, cb_path, "", 0);
        free(cb_path);
        set_clipboard_state.need_reset = 0;
        return;
    }

    if (set_clipboard_state.data == NULL) {
        free(cb_path);
        return;
    }
    this_time = MIN(1024,
                    strlen(set_clipboard_state.data) -
                    set_clipboard_state.off);
    xs_write(xsh,
             XBT_NULL,
             cb_path,
             set_clipboard_state.data + set_clipboard_state.off,
             this_time);
    free(cb_path);
    if (this_time != 0) {
        set_clipboard_state.off += this_time;
    } else {
        free(set_clipboard_state.data);
        set_clipboard_state.data = NULL;
        set_clipboard_state.off = 0;
    }
}

/* Something happened on the state machine used by the guest to update
   the VNC clipboard. */
static void report_clipboard_event(void)
{
    static int failed;
    static char *accumulation;
    static int acc_len;
    static int acc_buf_size;
    int nacc_buf_size;
    unsigned len;
    char *chunk;
    char *nacc_buf;
    char *d_path, *cb_path;

    cb_path = NULL;
    d_path = xs_get_domain_path(xsh, domid);
    if (d_path) {
        if (asprintf(&cb_path, "%s/data/report_clipboard", d_path) == -1)
            cb_path = NULL;
    }
    free(d_path);
    if (!cb_path)
        return;

    chunk = xs_read(xsh, XBT_NULL, cb_path, &len);
    if (!chunk) {
        free(cb_path);
        return;
    }
    if (len == 0) {
        if (accumulation != NULL && !failed)
            set_clipboard(accumulation);
        else
            free(accumulation);
        failed = 0;
        acc_len = 0;
        accumulation = NULL;
        acc_buf_size = 0;
    } else {
        if (len + acc_len + 1 > acc_buf_size) {
            nacc_buf_size = ((len + acc_len + 1) + 63) & ~63;
            nacc_buf = realloc(accumulation, nacc_buf_size);
        } else {
            nacc_buf = accumulation;
            nacc_buf_size = acc_buf_size;
        }
        if (nacc_buf) {
            acc_buf_size = nacc_buf_size;
            memcpy(nacc_buf + acc_len, chunk, len);
            nacc_buf[acc_len + len] = 0;
            acc_len += len;
            accumulation = nacc_buf;
        } else {
            failed = 1;
        }
    }
    free(chunk);

    xs_rm(xsh, XBT_NULL, cb_path);
    free(cb_path);
}
#endif

void xenstore_process_event(void *opaque)
{
#if !defined(QEMU_KXEN)
    char **vec, *offset, *image = NULL;
    unsigned int len, num, hd_index;
    xc_dominfo_t dominfo;

    vec = xs_read_watch(xsh, &num);
    if (!vec)
        return;

    if (!strcmp(vec[XS_WATCH_TOKEN], "logdirty")) {
        xenstore_process_logdirty_event();
        goto out;
    }

    if (!strcmp(vec[XS_WATCH_TOKEN], "releaseDomain") &&
            (xc_domain_getinfo(xc_handle, domid, 1, &dominfo) != 1 ||
             dominfo.domid != domid || dominfo.dying)) {
        exit_requested = 1;
        goto out;
    }

    if (!strcmp(vec[XS_WATCH_TOKEN], "dm-command")) {
        xenstore_process_dm_command_event();
        goto out;
    }

    if (!strcmp(vec[XS_WATCH_TOKEN], SET_CLIPBOARD_TOKEN)) {
        set_clipboard_event();
        goto out;
    }
    if (!strcmp(vec[XS_WATCH_TOKEN], REPORT_CLIPBOARD_TOKEN)) {
        report_clipboard_event();
        goto out;
    }

#ifndef CONFIG_STUBDOM
    if (!strcmp(vec[XS_WATCH_TOKEN], KEYMAP_TOKEN)) {
	image = xs_read(xsh, XBT_NULL, vec[XS_WATCH_PATH], &len);
 	if (image == NULL)
 	    goto out;
	vnc_keymap_change(opaque, image);
	goto out;
    }
#endif

    if ((strncmp(vec[XS_WATCH_TOKEN], "hd", 2) &&
         strncmp(vec[XS_WATCH_TOKEN], "fd", 2)) ||
        strlen(vec[XS_WATCH_TOKEN]) != 3)
        goto out;

    hd_index = drive_name_to_index(vec[XS_WATCH_TOKEN]);
    if (hd_index == -1) {
	fprintf(stderr,"medium change watch on `%s' -"
		" unknown device, ignored\n", vec[XS_WATCH_TOKEN]);
	goto out;
    }

    image = xs_read(xsh, XBT_NULL, vec[XS_WATCH_PATH], &len);

    fprintf(stderr,"medium change watch on `%s' (index: %d): %s\n",
	    vec[XS_WATCH_TOKEN], hd_index, image ? image : "<none>");

    if (image == NULL)
        goto out;  /* gone */

    if (!strcmp(image, drives_table[hd_index].bdrv->filename))
        goto out;  /* identical */

    drives_table[hd_index].bdrv->filename[0] = '\0';
    bdrv_close(drives_table[hd_index].bdrv);
    if (media_filename[hd_index]) {
        free(media_filename[hd_index]);
        media_filename[hd_index] = NULL;
    }

    if (image[0]) {
        media_filename[hd_index] = strdup(image);
    }

    if (!strncmp(vec[XS_WATCH_TOKEN], "fd", 2) && insert_timer == NULL)
        insert_media(NULL);

 out:
    free(image);
    free(vec);
#endif
}

void xenstore_write_vncport(int display)
{
#if !defined(QEMU_KXEN)
    char *buf = NULL, *path;
    char *portstr = NULL;

    if (xsh == NULL)
        return;

    path = xs_get_domain_path(xsh, domid);
    if (path == NULL) {
        fprintf(logfile, "xs_get_domain_path() error\n");
        goto out;
    }

    if (pasprintf(&buf, "%s/console/vnc-port", path) == -1)
        goto out;

    if (pasprintf(&portstr, "%d", display) == -1)
        goto out;

    if (xs_write(xsh, XBT_NULL, buf, portstr, strlen(portstr)) == 0)
        fprintf(logfile, "xs_write() vncport failed\n");

 out:
    free(portstr);
    free(buf);
#endif
}

void xenstore_read_vncpasswd(int domid, char *pwbuf, size_t pwbuflen)
{
#if !defined(QEMU_KXEN)
    char *buf = NULL, *path, *uuid = NULL, *passwd = NULL;
    unsigned int i, len;

    pwbuf[0] = '\0';

    if (xsh == NULL)
        return;

    path = xs_get_domain_path(xsh, domid);
    if (path == NULL) {
        fprintf(logfile, "xs_get_domain_path() error. domid %d.\n", domid);
        return;
    }

    pasprintf(&buf, "%s/vm", path);
    free(path);
    uuid = xs_read(xsh, XBT_NULL, buf, &len);
    if (uuid == NULL) {
        fprintf(logfile, "xs_read(): uuid get error. %s.\n", buf);
        free(buf);
        return;
    }

    pasprintf(&buf, "%s/vncpasswd", uuid);
    free(uuid);
    passwd = xs_read(xsh, XBT_NULL, buf, &len);
    if (passwd == NULL) {
        fprintf(logfile, "xs_read(): vncpasswd get error. %s.\n", buf);
        free(buf);
        return;
    }

    if (len >= pwbuflen)
    {
        fprintf(logfile, "xenstore_read_vncpasswd(): truncated password to avoid buffer overflow\n");
        len = pwbuflen - 1;
    }

    for (i=0; i<len; i++)
        pwbuf[i] = passwd[i];
    pwbuf[len] = '\0';
    passwd[0] = '\0';
    if (xs_write(xsh, XBT_NULL, buf, passwd, 1) == 0)
        fprintf(logfile, "xs_write() vncpasswd failed.\n");

    free(passwd);
    free(buf);
#endif
}


#if !defined(QEMU_KXEN)
/*
 * get all device instances of a certain type
 */
char **xenstore_domain_get_devices(struct xs_handle *handle,
                                   const char *devtype, unsigned int *num)
{
    char *path;
    char *buf = NULL;
    char **e  = NULL;

    path = xs_get_domain_path(handle, domid);
    if (path == NULL)
        goto out;

    if (pasprintf(&buf, "%s/device/%s", path,devtype) == -1)
        goto out;

    e = xs_directory(handle, XBT_NULL, buf, num);

 out:
    free(path);
    free(buf);
    return e;
}

/*
 * Check whether a domain has devices of the given type
 */
int xenstore_domain_has_devtype(struct xs_handle *handle, const char *devtype)
{
    int rc = 0;
    unsigned int num;
    char **e = xenstore_domain_get_devices(handle, devtype, &num);
    if (e)
        rc = 1;
    free(e);
    return rc;
}

/*
 * Function that creates a path to a variable of an instance of a
 * certain device
 */
static char *get_device_variable_path(const char *devtype, const char *inst,
                                      const char *var)
{
    char *buf = NULL;
    if (pasprintf(&buf, "/local/domain/0/backend/%s/%d/%s/%s",
                  devtype,
                  domid,
                  inst,
                  var) == -1) {
        free(buf);
        buf = NULL;
    }
    return buf;
}

char *xenstore_backend_read_variable(struct xs_handle *handle,
                                     const char *devtype, const char *inst,
                                     const char *var)
{
    char *value = NULL;
    char *buf = NULL;
    unsigned int len;

    buf = get_device_variable_path(devtype, inst, var);
    if (NULL == buf)
        goto out;

    value = xs_read(handle, XBT_NULL, buf, &len);

    free(buf);

 out:
    return value;
}

/*
  Read the hotplug status variable from the backend given the type
  of device and its instance.
*/
char *xenstore_read_hotplug_status(struct xs_handle *handle,
                                   const char *devtype, const char *inst)
{
    return xenstore_backend_read_variable(handle, devtype, inst,
                                          "hotplug-status");
}

/*
   Subscribe to the hotplug status of a device given the type of device and
   its instance.
   In case an error occurrs, a negative number is returned.
 */
int xenstore_subscribe_to_hotplug_status(struct xs_handle *handle,
                                         const char *devtype,
                                         const char *inst,
                                         const char *token)
{
    int rc = 0;
    char *path = get_device_variable_path(devtype, inst, "hotplug-status");

    if (path == NULL)
        return -1;

    if (0 == xs_watch(handle, path, token))
        rc = -2;

    free(path);

    return rc;
}

/*
 * Unsubscribe from a subscription to the status of a hotplug variable of
 * a device.
 */
int xenstore_unsubscribe_from_hotplug_status(struct xs_handle *handle,
                                             const char *devtype,
                                             const char *inst,
                                             const char *token)
{
    int rc = 0;
    char *path;
    path = get_device_variable_path(devtype, inst, "hotplug-status");
    if (path == NULL)
        return -1;

    if (0 == xs_unwatch(handle, path, token))
        rc = -2;

    free(path);

    return rc;
}

char *xenstore_vm_read(int domid, char *key, unsigned int *len)
{
    char *buf = NULL, *path = NULL, *value = NULL;

    if (xsh == NULL)
        goto out;

    path = xs_get_domain_path(xsh, domid);
    if (path == NULL) {
        fprintf(logfile, "xs_get_domain_path(%d): error\n", domid);
        goto out;
    }

    pasprintf(&buf, "%s/vm", path);
    free(path);
    path = xs_read(xsh, XBT_NULL, buf, NULL);
    if (path == NULL) {
        fprintf(logfile, "xs_read(%s): read error\n", buf);
        goto out;
    }

    pasprintf(&buf, "%s/%s", path, key);
    value = xs_read(xsh, XBT_NULL, buf, len);
    if (value == NULL) {
        fprintf(logfile, "xs_read(%s): read error\n", buf);
        goto out;
    }

 out:
    free(path);
    free(buf);
    return value;
}
#endif

/* Advertise through xenstore that the device model is up and the
   domain can be started. */
void xenstore_dm_finished_startup(void)
{
#if !defined(QEMU_KXEN)
    char *path;
    char *buf = NULL;

    path = xs_get_domain_path(xsh, domid);
    if (!path) {
        fprintf(logfile, "xs_get_domain_path() failed.\n");
        goto out;
    }
    if (pasprintf(&buf, "%s/device-misc/dm-ready", path) == -1) {
        fprintf(logfile, "pasprintf failed to get path.\n");
        goto out;
    }
    if (xs_write(xsh, XBT_NULL, buf, "1", 1) == 0)
        fprintf(logfile, "xs_write() dm-ready failed\n");
 out:
    free(buf);
    free(path);
#endif
}

#if !defined(QEMU_KXEN)
#ifndef CONFIG_STUBDOM
void
xenstore_drop_privileges(void)
{
    if (!xs_restrict(xsh, domid))
        err(1, "failed to restrict xenstore access");
}
#endif
#endif

void
xenstore_set_guest_clipboard(const char *text, size_t len)
{
#if !defined(QEMU_KXEN)
    char *dup;

    if (set_clipboard_state.data)
        set_clipboard_state.need_reset = 1;
    dup = malloc(len + 1);
    if (!dup)
        return;
    memcpy(dup, text, len);
    dup[len] = 0;

    set_clipboard_state.off = 0;
    free(set_clipboard_state.data);
    set_clipboard_state.data = dup;

    set_clipboard_event();
#endif
}

#if !defined(QEMU_KXEN)
/*
 * Create a store entry for a device (e.g., monitor, serial/parallel lines).
 * The entry is <domain-path><storeString>/tty and the value is the name
 * of the pty associated with the device.
 */
static int store_dev_info(const char *devName, int domid,
                          CharDriverState *cState, const char *storeString)
{
#ifdef CONFIG_STUBDOM
    fprintf(logfile, "can't store dev %s name for domid %d in %s from a stub domain\n", devName, domid, storeString);
    return ENOSYS;
#else
    int xc_handle;
    struct xs_handle *xs;
    char *path;
    char *newpath;
    char *pts;
    char namebuf[128];
    int ret;

    /*
     * Only continue if we're talking to a pty
     */
    if (!cState->chr_getname) return 0;
    ret = cState->chr_getname(cState, namebuf, sizeof(namebuf));
    if (ret < 0) {
        fprintf(logfile, "ptsname failed (for '%s'): %s\n",
                storeString, strerror(errno));
        return 0;
    }
    if (memcmp(namebuf, "pty ", 4)) return 0;
    pts = namebuf + 4;

    /* We now have everything we need to set the xenstore entry. */
    xs = xs_daemon_open();
    if (xs == NULL) {
        fprintf(logfile, "Could not contact XenStore\n");
        return -1;
    }

    xc_handle = xc_interface_open();
    if (xc_handle == -1) {
        fprintf(logfile, "xc_interface_open() error\n");
        return -1;
    }

    path = xs_get_domain_path(xs, domid);
    if (path == NULL) {
        fprintf(logfile, "xs_get_domain_path() error\n");
        return -1;
    }
    newpath = realloc(path, (strlen(path) + strlen(storeString) +
                             strlen("/tty") + 1));
    if (newpath == NULL) {
        free(path); /* realloc errors leave old block */
        fprintf(logfile, "realloc error\n");
        return -1;
    }
    path = newpath;

    strcat(path, storeString);
    strcat(path, "/tty");
    if (!xs_write(xs, XBT_NULL, path, pts, strlen(pts))) {
        fprintf(logfile, "xs_write for '%s' fail", storeString);
        return -1;
    }

    free(path);
    xs_daemon_close(xs);
    close(xc_handle);

    return 0;
#endif
}
#endif

void xenstore_store_serial_port_info(int i, CharDriverState *chr,
				     const char *devname) {
#if !defined(QEMU_KXEN)
    char buf[16];

    snprintf(buf, sizeof(buf), "/serial/%d", i);
    store_dev_info(devname, domid, chr, buf);
    if (i == 0) /* serial 0 is also called the console */
        store_dev_info(devname, domid, chr, "/console");
#endif
}

#if !defined(QEMU_KXEN)
unsigned int
xenstore_get_cores_per_socket(int domid)
{
    char *buf = NULL, *path = NULL, *s;
    unsigned int cores_per_socket = 0;

    path = xs_get_domain_path(xsh, domid);
    if (path == NULL) {
        fprintf(logfile, "xs_get_domain_path(%d): error\n", domid);
        goto out;
    }
    pasprintf(&buf, "%s/platform/cores-per-socket", path);

    s = xs_read(xsh, XBT_NULL, buf, NULL);
    if (s) {
	if (sscanf(s, "%d", &cores_per_socket) != 1)
	    cores_per_socket = 0;
	free(s);
    }

  out:
    free(path);
    free(buf);
    return cores_per_socket;
}

#endif
