/*
 *  err.h
 *
 *  Copyright 2009 Citrix Systems, Inc. All rights reserved.
 *
 */

#ifndef _ERR_H_
#define _ERR_H_

#include <errno.h>
#include <stdarg.h>
#include <stdio.h>
#include <string.h>

#include <windows.h>

#ifndef ERR_NO_PROGNAME
extern const char *_progname;

#define DECLARE_PROGNAME const char *_progname
#else
#define _progname "err"
#endif

static inline void
vwarn(const char *fmt, va_list ap)
{
    (void)fprintf(stderr, "%s: ", _progname);
    if (fmt) {
        (void)vfprintf(stderr, fmt, ap);
        (void)fprintf(stderr, ": %s", strerror(errno));
    }
    (void)fprintf(stderr, "\n");
}

static inline void
vwarnx(const char *fmt, va_list ap)
{
    (void)fprintf(stderr, "%s: ", _progname);
    if (fmt)
        (void)vfprintf(stderr, fmt, ap);
    (void)fprintf(stderr, "\n");
}

static inline void
warn(const char *fmt, ...)
{
    va_list ap;

    va_start(ap, fmt);
    vwarn(fmt, ap);
    va_end(ap);
}

static inline void
warnx(const char *fmt, ...)
{
    va_list ap;

    va_start(ap, fmt);
    vwarnx(fmt, ap);
    va_end(ap);
}

static inline void
verr(int eval, const char *fmt, va_list ap)
{

    vwarn(fmt, ap);
    exit(eval);
    /* NOTREACHED */
}

static inline void
verrx(int eval, const char *fmt, va_list ap)
{

    vwarnx(fmt, ap);
    exit(eval);
    /* NOTREACHED */
}

static inline void
err(int eval, const char *fmt, ...)
{
    va_list ap;

    va_start(ap, fmt);
    verr(eval, fmt, ap);
    /* NOTREACHED */
}

static inline void
errx(int eval, const char *fmt, ...)
{
    va_list ap;

    va_start(ap, fmt);
    verrx(eval, fmt, ap);
    /* NOTREACHED */
}

static inline void
Wwarnv(const char *fmt, va_list ap)
{
    LPVOID lpMsgBuf;

    FormatMessage(FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM |
		  FORMAT_MESSAGE_IGNORE_INSERTS, NULL, GetLastError(),
		  MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), (LPTSTR)&lpMsgBuf,
		  0, NULL);

    (void)fprintf(stderr, "%s: ", _progname);
    if (fmt)
        (void)vfprintf(stderr, fmt, ap);
    (void)fprintf(stderr, ": %s", lpMsgBuf);

    LocalFree(lpMsgBuf);
}

static inline void
Wwarn(const char *fmt, ...)
{
    va_list ap;

    va_start(ap, fmt);
    Wwarnv(fmt, ap);
    va_end(ap);
}

static inline void
Werr(int eval, const char *fmt, ...)
{
    va_list ap;

    va_start(ap, fmt);
    Wwarnv(fmt, ap);
    exit(eval);
    /* NOTREACHED */
}

static inline const char *
getprogname(void)
{
    return _progname;
}

static inline void
setprogname(const char *name)
{
#ifndef ERR_NO_PROGNAME
    _progname = strrchr(name, '/');
    if (_progname)
        return;
    _progname = strrchr(name, '\\');
    if (_progname)
        return;
    _progname = name;
#endif
}

#endif

