/*
 *  xc_windows.c
 *  hxen
 *
 *  Copyright 2009 Citrix Systems, Inc. All rights reserved.
 *
 */

#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>

#include <windows.h>

#include "hxenctllib.h"

#include "xc_private.h"

int xc_use_foreign_mappings = 1;

static HANDLE *h = NULL;
static int nrh = 0;

static int
bad_handle(int xc_handle)
{
    return h[xc_handle] == INVALID_HANDLE_VALUE;
}

int xc_memory_op_new(int xc_handle, int cmd, void *arg)
{
    int ret;
    struct hxen_memop_desc kmd;

    if (bad_handle(xc_handle)) {
	errno = EBADF;
	return -1;
    }

    kmd.kmd_cmd = cmd;
    set_xen_guest_handle(kmd.kmd_kma, arg);

    ret = hxen_memop(h[xc_handle], &kmd);
    return ret;
}

int do_hvm_op(int xc_handle, unsigned cmd, void *arg)
{
    int ret;
    struct hxen_hvmop_desc khd;

    if (bad_handle(xc_handle)) {
	errno = EBADF;
	return -1;
    }

    khd.khd_cmd = cmd;
    set_xen_guest_handle(khd.khd_kha, arg);

    ret = hxen_hvmop(h[xc_handle], &khd);
    return ret;
}

int xc_interface_open(void)
{
    int fd;

    for (fd = 0; fd < nrh; fd++)
	if (h[fd] == INVALID_HANDLE_VALUE)
	    break;

    if (fd == nrh) {
	HANDLE *nh = h;
	int nnrh = 1 + nrh * 2;
	nh = realloc(h, nnrh * sizeof(HANDLE));
	if (nh == NULL) {
	    PERROR("Could not realloc interface handle array");
	    return -1;
	}
	h = nh;
	while (nrh < nnrh) {
	    h[nrh] = INVALID_HANDLE_VALUE;
	    nrh++;
	}
    }

    h[fd] = hxen_open(0, TRUE);
    if (h[fd] == INVALID_HANDLE_VALUE) {
	PERROR("Could not obtain handle on privileged command interface");
	return -1;
    }

    return fd;
}

int xc_interface_close(int xc_handle)
{
    if (bad_handle(xc_handle)) {
	errno = EBADF;
	return -1;
    }
    hxen_close(h[xc_handle]);
    h[xc_handle] = INVALID_HANDLE_VALUE;
    return 0;
}

void *xc_map_foreign_range(int xc_handle, uint32_t dom,
                           int size, int prot,
                           unsigned long mfn)
{
    int ret, i;
    struct hxen_mmapbatch_desc kmd;

    if (bad_handle(xc_handle))
	return NULL;

    if (xc_use_foreign_mappings) {
	kmd.kmd_num = size >> PAGE_SHIFT;
	kmd.kmd_domid = dom;
	kmd.kmd_arr = malloc(kmd.kmd_num * sizeof(*kmd.kmd_arr));
	if (!kmd.kmd_arr)
	    return NULL;
	for (i = 0; i < kmd.kmd_num; i++)
	    kmd.kmd_arr[i] = mfn + i;
	ret = hxen_mmapbatch(h[xc_handle], &kmd);
	free(kmd.kmd_arr);
	if (ret != 0)
	    kmd.kmd_addr = 0;
	return (void *)(uintptr_t)kmd.kmd_addr;
    } else {
#if 0
	if ((mfn >> KXEN_VMAPPING_SHIFT) !=
	    ((mfn + ((size - 1) >> PAGE_SHIFT)) >> KXEN_VMAPPING_SHIFT)) {
	    errno = EINVAL;
	    return NULL;
	}
	return phys_addr_vmap(mfn << PAGE_SHIFT);
#else
	return NULL;
#endif
    }
}

void
xc_munmap(int xc_handle, void *addr, int num)
{
    struct hxen_munmap_desc kmd;

    if (bad_handle(xc_handle))
	return;

    if (!xc_use_foreign_mappings)
	return;

    kmd.kmd_num = num >> PAGE_SHIFT;
    kmd.kmd_addr = (uint64_t)(uintptr_t)addr;
    (void)hxen_munmap(h[xc_handle], &kmd);
}

int do_xen_hypercall(int xc_handle, privcmd_hypercall_t *hypercall)
{
    if (bad_handle(xc_handle)) {
	errno = EBADF;
	return -1;
    }

    return hxen_hypercall(h[xc_handle],
			  (struct hxen_hypercall_desc *)hypercall);
}
