/*
 *  hxen.c
 *  hxen
 *
 *  Copyright 2009 Citrix Systems, Inc. All rights reserved.
 * 
 */

#include "hxen.h"

#include <ntddk.h>
#include <wdmsec.h>

#define hxen_driver_load DriverEntry

struct device_extension *hxen_devext = NULL;

uint8_t *hxen_hv = NULL;
size_t hxen_size = 0;

#include <initguid.h>
//
// Since this driver is a legacy driver and gets installed as a service
// (without an INF file),  we will define a class guid for use in
// IoCreateDeviceSecure function. This would allow  the system to store
// Security, DeviceType, Characteristics and Exclusivity information of the
// deviceobject in the registery under
// HKLM\SYSTEM\CurrentControlSet\Control\Class\ClassGUID\Properties.
// This information can be overridden by an Administrator giving them the
// ability to control access to the device beyond what is initially allowed
// by the driver developer.
//
DEFINE_GUID (GUID_DEVCLASS_KXEN,
	     0xc87b189a, 0xb648, 0x4b80, 0xb4, 0xb6,
	     0x26, 0x9e, 0x73, 0xb2, 0x56, 0x4c);

NTSTATUS
hxen_create_close(
    __in PDEVICE_OBJECT DeviceObject,
    __in PIRP Irp
    )
{
    NTSTATUS status;
    struct device_extension *devext;
    IO_STACK_LOCATION *irpstack;

    devext = DeviceObject->DeviceExtension;
    irpstack = IoGetCurrentIrpStackLocation(Irp);
    status = STATUS_SUCCESS;

    switch(irpstack->MajorFunction) {
    case IRP_MJ_CREATE:
	dprintk("hxen_create_close: create\n");
	Irp->IoStatus.Information = 0;
	break;
    case IRP_MJ_CLOSE:
	dprintk("hxen_create_close: close\n");
	Irp->IoStatus.Information = 0;
	break;
    default:
	fail_msg("hxen_create_close: invalid irp function %x\n",
		 irpstack->MajorFunction);
	status = STATUS_INVALID_PARAMETER;
	break;
    }

    Irp->IoStatus.Status = status;
    IoCompleteRequest(Irp, IO_NO_INCREMENT);

    return status;
}

NTSTATUS
hxen_cleanup(
    __in PDEVICE_OBJECT DeviceObject,
    __in PIRP Irp
)
{
    dprintk("hxen_cleanup\n");

    Irp->IoStatus.Information = 0;
    Irp->IoStatus.Status = STATUS_SUCCESS;
    IoCompleteRequest(Irp, IO_NO_INCREMENT);

    dprintk("hxen_cleanup done\n");

    return STATUS_SUCCESS;
}

void
hxen_driver_unload(
    __in PDRIVER_OBJECT DriverObject
    )
{
    DEVICE_OBJECT *devobj = DriverObject->DeviceObject;
    UNICODE_STRING devicename_dos;

    dprintk("hxen_driver_unload\n");

    hxen_unload();

    (void)RtlInitUnicodeString(&devicename_dos, KXEN_DEVICE_NAME_DOS_U);
    IoDeleteSymbolicLink(&devicename_dos);
    ASSERT(!devobj->AttachedDevice);
    IoDeleteDevice(devobj);

    dprintk("hxen_driver_unload done\n");
    return;
}

NTSTATUS
hxen_driver_load(
    __in PDRIVER_OBJECT DriverObject,
    __in PUNICODE_STRING RegistryPath
    )
{
    NTSTATUS status = STATUS_SUCCESS;
    UNICODE_STRING devicename, devicename_dos, secdesc;
    DEVICE_OBJECT *devobj;
    struct device_extension *devext;

    UNREFERENCED_PARAMETER (RegistryPath);

    dprintk("hxen_driver_load\n");

    (void)RtlInitUnicodeString(&devicename, KXEN_DEVICE_NAME_U);

    //
    // We will create a secure deviceobject so that only processes running
    // in admin and local system account can access the device. Refer
    // "Security Descriptor String Format" section in the platform
    // SDK documentation to understand the format of the sddl string.
    // We need to do because this is a legacy driver and there is no INF
    // involved in installing the driver. For PNP drivers, security descriptor
    // is typically specified for the FDO in the INF file.
    //
    // Security Descriptor
    //
    // D: means it's a DACL (Discretionary Access Control List), 
    // P  means it's protected.
    //
    // ACEs are enclosed in parameters and have 6 fields
    //  ACE type                                A       Allowed
    //  ACE flags                               .
    //  Permission                              GA      Generic All
    //  Object Type                             .
    //  Inherited Object Type                   .
    //  Trustee                                 BA      Built-in Administrators
    //
    // Details http://msdn.microsoft.com/en-us/library/aa379567(VS.85).aspx
    // http://blogs.dirteam.com/blogs/jorge/archive/2008/03/26/parsing-sddl-strings.aspx
    //

    (void)RtlInitUnicodeString(&secdesc, L"D:P(A;;GA;;;SY)(A;;GA;;;BA)");

    status = IoCreateDeviceSecure(DriverObject, sizeof(struct device_extension),
				  &devicename, FILE_DEVICE_UNKNOWN,
				  FILE_DEVICE_SECURE_OPEN, FALSE,
				  &secdesc, (LPCGUID)&GUID_DEVCLASS_KXEN,
				  &devobj);
    if (!NT_SUCCESS(status))
	return status;

    (void)RtlInitUnicodeString(&devicename_dos, KXEN_DEVICE_NAME_DOS_U);

    status = IoCreateSymbolicLink(&devicename_dos, &devicename);
    if (!NT_SUCCESS(status)) {
        IoDeleteDevice(devobj);
        return status;
    }

    devext = devobj->DeviceExtension;
    hxen_devext = devext;
#ifdef DBG
    kdbgdevext = devext;
#endif
    RtlZeroMemory(devext, sizeof(struct device_extension));

    DriverObject->MajorFunction[IRP_MJ_CREATE] = hxen_create_close;
    DriverObject->MajorFunction[IRP_MJ_CLOSE] = hxen_create_close;
    DriverObject->MajorFunction[IRP_MJ_CLEANUP] = hxen_cleanup;

    devext->de_FastIoDispatch.SizeOfFastIoDispatch = sizeof(FAST_IO_DISPATCH);
    devext->de_FastIoDispatch.FastIoDeviceControl = hxen_ioctl;
    DriverObject->FastIoDispatch = &devext->de_FastIoDispatch;

    DriverObject->DriverUnload = hxen_driver_unload;

    KeInitializeSpinLock(&devext->de_biglock);

    dprintk("hxen_driver_load\n");
    return status;
}
