/*
 *  hxen_cpu.c
 *  hxen
 *
 *  Copyright 2009 Citrix Systems, Inc. All rights reserved.
 *
 */

#include "hxen.h"

#define KXEN_DEFINE_SYMBOLS_PROTO
#include <hxen_link.h>
KXEN_PROTOTYPES(extern)

unsigned long hxen_cpu_vm = 0;

#if defined(__x86_64__)
#define to_affinity(x) ((ULONGLONG)1 << (x))
#else
#define to_affinity(x) (1 << (x))
#endif

unsigned long
hxen_first_cpu(void)
{
    KAFFINITY affinity;
    unsigned long host_cpu;

    affinity = KeQueryActiveProcessors();
    for (host_cpu = 0; host_cpu < MAXIMUM_PROCESSORS; host_cpu++) {
	if (affinity & to_affinity(host_cpu))
	    break;
    }
    ASSERT(host_cpu != MAXIMUM_PROCESSORS);
    return host_cpu;
}

void
hxen_cpu_pin(unsigned long host_cpu)
{
    KAFFINITY affinity;

    affinity = to_affinity(host_cpu);
    KeSetSystemAffinityThread(affinity);
}

void
hxen_cpu_pin_current(void)
{
    unsigned long cpu;

    cpu = KeGetCurrentProcessorNumber();
    hxen_cpu_pin(cpu);
}

void
hxen_cpu_pin_first(void)
{
    unsigned long cpu;

    cpu = hxen_first_cpu();
    hxen_cpu_pin(cpu);
}

void
hxen_cpu_pin_vcpu(struct vm_info *vi)
{
    KeSetTargetProcessorDpc(&vi->vi_timer_dpc, (CCHAR)vi->vi_host_cpu);
    KeSetTargetProcessorDpc(&vi->vi_ipi_dpc, (CCHAR)vi->vi_host_cpu);
    hxen_cpu_pin(vi->vi_host_cpu);
}

void
hxen_cpu_unpin(void)
{

    KeRevertToUserAffinityThread();
}

void
hxen_set_cpu_active_mask(void *mask, int mask_size)
{
    KAFFINITY affinity;
    unsigned long host_cpu;

    affinity = KeQueryActiveProcessors();
    ASSERT(sizeof(affinity) <= mask_size);
    memcpy(mask, &affinity, sizeof(affinity));

    for (host_cpu = hxen_first_cpu() + 1; host_cpu < MAXIMUM_PROCESSORS;
	 host_cpu++) {
	if (affinity & to_affinity(host_cpu))
	    break;
    }
    ASSERT(host_cpu != MAXIMUM_PROCESSORS);
    hxen_cpu_vm = host_cpu;
}

void __cdecl
hxen_on_each_cpu(void (*fn)(void *), void *arg)
{
    KAFFINITY affinity;
    CCHAR host_cpu;

    affinity = KeQueryActiveProcessors();
    for (host_cpu = MAXIMUM_PROCESSORS - 1; host_cpu >= 0; host_cpu--) {
	if ((affinity & to_affinity(host_cpu)) == 0)
	    continue;
	hxen_cpu_pin(host_cpu);
	dprintk("hxen_on_each_cpu on cpu %d: %p(%p)\n", host_cpu, fn, arg);
	try {
	    fn(arg);
	} except (KXEN_EXCEPTION_EXECUTE_HANDLER) {
	}
    }
    hxen_cpu_unpin();
}
