/*
 *  hxenctl.c
 *  hxen
 *
 *  Copyright 2009 Citrix Systems, Inc. All rights reserved.
 *
 */

#include <err.h>
#include <fcntl.h>
#include <getopt.h>
#include <stddef.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>

#include <sys/stat.h>

#include <public/version.h>
#include <hxen_ioctl.h>
#include "hxenctllib.h"

DECLARE_PROGNAME;

void
usage(const char *progname)
{
    fprintf(stderr, "usage: %s [-R] [-L] [-l file] [-i] [-v] "
	    "[-k keys] [-s] [-u] [-U]\n", progname);
    exit(1);
}

void
do_unload_driver(void)
{
    int ret;

    ret = hxen_manage_driver(FALSE, FALSE);
    if (ret)
	warnx("unload driver failed");
}

int
main(int argc, char **argv)
{
    int ret;
    int init = 0;
    int shutdown = 0;
    char *load = NULL;
    int unload = 0;
    int version = 0;
    char *keys = NULL;
    int load_driver = 0;
    int reload_driver = 0;
    int unload_driver = 0;
    HANDLE kh;

    setprogname(argv[0]);

    while (1) {
	int c, index = 0;
	static struct option long_options[] = {
	    {"help",          no_argument,       NULL, 'h'},
	    {"load",          required_argument, NULL, 'l'},
	    {"init",          no_argument,       NULL, 'i'},
	    {"shutdown",      no_argument,       NULL, 's'},
	    {"unload",        no_argument,       NULL, 'u'},
	    {"version",       no_argument,       NULL, 'v'},
	    {"keys",          required_argument, NULL, 'k'},
	    {"load-driver",   no_argument,       NULL, 'L'},
	    {"reload-driver", no_argument,       NULL, 'R'},
	    {"unload-driver", no_argument,       NULL, 'U'},
	    {NULL,   0,                 NULL, 0}
	};

	c = getopt_long(argc, argv, "hl:isuvk:LRU", long_options, &index);
	if (c == -1)
	    break;

	switch (c) {
	case 'h':
	    usage(argv[0]);
	    /* NOTREACHED */
	case 'l':
	    load = optarg;
	    break;
	case 'i':
	    init = 1;
	    break;
	case 's':
	    shutdown = 1;
	    break;
	case 'u':
	    unload = 1;
	    break;
	case 'v':
	    version = 1;
	    break;
	case 'k':
	    keys = optarg;
	    break;
	case 'L':
	    load_driver = 1;
	    break;
	case 'R':
	    reload_driver = 1;
	    load_driver = 1;
	    break;
	case 'U':
	    unload_driver = 1;
	    break;
	}
    }

    if (unload_driver) {
	ret = atexit(do_unload_driver);
	if (ret)
	    err(1, "at_exit");
    }

    if (reload_driver) {
	ret = hxen_manage_driver(FALSE, TRUE);
	if (ret)
	    errx(1, "reload driver unload failed");
    }

    if (load_driver) {
	ret = hxen_manage_driver(TRUE, FALSE);
	if (ret)
	    errx(1, "load driver failed");
    }

    if (init == 0 && shutdown == 0 && load == NULL && unload == 0 &&
	version == 0 && keys == NULL) {
	if (load_driver || unload_driver)
	    exit(0);
	usage(argv[0]);
	/* NOTREACHED */
    }

    kh = hxen_open(0, load != NULL);
    if (kh == INVALID_HANDLE_VALUE)
	errx(1, "hxen_open failed");

    if (load) {
	ret = hxen_load(kh, load);
	if (ret)
	    errx(1, "load failed");
    }

    if (init) {
	ret = hxen_init(kh);
	if (ret)
	    errx(1, "init failed");
    }

    if (version) {
	ret = hxen_output_version_info(kh, stdout);
	if (ret)
	    errx(1, "version failed");
    }

    if (keys) {
	ret = hxen_trigger_keyhandler(kh, keys);
	if (ret)
	    errx(1, "trigger keyhandler failed");
    }

    if (shutdown) {
	ret = hxen_shutdown(kh);
	if (ret)
	    errx(1, "shutdown failed");
    }

    if (unload) {
	ret = hxen_unload(kh);
	if (ret)
	    errx(1, "unload failed");
    }

    hxen_close(kh);

    return 0;
}
