/*
 *  hxenctllib.c
 *  hxen
 *
 *  Copyright 2009 Citrix Systems, Inc. All rights reserved.
 *
 */

#include <err.h>
#include <fcntl.h>
#include <getopt.h>
#include <stddef.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>

#include <sys/stat.h>

#include "hxen_def.h"

#include <winioctl.h>
#include "hxen_ioctl.h"

#include <public/xen.h>
#include <public/version.h>

#include "hxenctllib.h"

int
hxen_manage_driver(BOOLEAN install, BOOLEAN fail_ok)
{
    SC_HANDLE scm_handle = NULL;
    SC_HANDLE scs_handle = NULL;
    int ret = -1;
    char pathbuf[MAX_PATH];

    scm_handle = OpenSCManager(NULL, NULL, SC_MANAGER_ALL_ACCESS);
    if (!scm_handle) {
	Wwarn("OpenSCManager");
	return ret;
    }

    if (install) {
	ret = GetCurrentDirectory(sizeof(pathbuf), pathbuf);
	if (ret == 0) {
	    Wwarn("GetCurrentDirectory");
	    goto error;
	}
	(void)strncat(pathbuf, "\\" KXEN_DRIVER_NAME ".sys", sizeof(pathbuf));
	scs_handle = CreateService(scm_handle, KXEN_DRIVER_NAME,
				   KXEN_DRIVER_NAME, SERVICE_ALL_ACCESS,
				   SERVICE_KERNEL_DRIVER, SERVICE_DEMAND_START,
				   SERVICE_ERROR_NORMAL,
				   pathbuf, NULL, NULL, NULL, NULL, NULL);
	if (scs_handle) {
	    CloseServiceHandle(scs_handle);
	    scs_handle = OpenService(scm_handle, KXEN_DRIVER_NAME,
				     SERVICE_ALL_ACCESS);
	}
	if (scs_handle == NULL && GetLastError() == ERROR_SERVICE_EXISTS)
	    scs_handle = OpenService(scm_handle, KXEN_DRIVER_NAME,
				     SERVICE_ALL_ACCESS);
	if (scs_handle == NULL) {
	    if (fail_ok) {
		ret = 0;
		goto out;
	    }
	    Wwarn("CreateService");
	    goto error;
	}
	ret = !StartService(scs_handle, 0, NULL);
	if (ret && GetLastError() != ERROR_SERVICE_ALREADY_RUNNING) {
	    if (fail_ok) {
		ret = 0;
		goto out;
	    }
	    Wwarn("StartService %s", pathbuf);
	    goto error;
	}
    } else {
	SERVICE_STATUS service_status;

	scs_handle = OpenService(scm_handle, KXEN_DRIVER_NAME,
				 SERVICE_ALL_ACCESS);
	if (scs_handle == NULL) {
	    if (fail_ok) {
		ret = 0;
		goto out;
	    }
	    Wwarn("OpenService");
	    goto error;
	}
	ret = !ControlService(scs_handle, SERVICE_CONTROL_STOP,
			      &service_status);
	if (ret) {
	    if (fail_ok) {
		ret = 0;
		goto out;
	    }
	    Wwarn("ControlService");
	    goto error;
	}

	DeleteService(scs_handle);
    }

    ret = 0;
  out:
    if (scs_handle) {
	if (ret)
	    DeleteService(scs_handle);
	CloseServiceHandle(scs_handle);
    }
    if (scm_handle)
	CloseServiceHandle(scm_handle);
    return ret;
  error:
    ret = -1;
    goto out;
}

HANDLE
hxen_open(int index, BOOLEAN install_driver)
{
    HANDLE h;
    int ret;

    /* FILE_FLAG_OVERLAPPED to makey go go fast! */
    h = CreateFile("\\\\.\\" KXEN_DEVICE_NAME, GENERIC_READ, 0, NULL,
		   OPEN_EXISTING, FILE_FLAG_OVERLAPPED, NULL);
    if (h != INVALID_HANDLE_VALUE)
	return h;
    if (!install_driver || index != 0) {
	Wwarn("CreateFile \\\\.\\" KXEN_DEVICE_NAME);
	return INVALID_HANDLE_VALUE;
    }

    ret = hxen_manage_driver(TRUE, FALSE);
    if (ret)
	return INVALID_HANDLE_VALUE;

    /* try again */
    return hxen_open(index, FALSE);
}

void
hxen_close(HANDLE h)
{
    if (h != INVALID_HANDLE_VALUE)
	CloseHandle(h);
}

static int
hxen_ioctl(HANDLE h, unsigned long ctl, ...)
{
    va_list ap;
    int func;
    int ret;
    void *InBuffer = NULL;
    void *OutBuffer = NULL;
    unsigned long InBufferLength = 0;
    unsigned long OutBufferLength = 0;
    unsigned long retval;

    va_start(ap, ctl);

    func = FUNCTION_FROM_CTL_CODE(ctl);

    if (func & KXEN_FLAG_INBUFFER) {
	InBuffer = va_arg(ap, void *);
	InBufferLength = va_arg(ap, unsigned long);
    }

    if (func & KXEN_FLAG_OUTBUFFER) {
	OutBuffer = va_arg(ap, void *);
	OutBufferLength = va_arg(ap, unsigned long);
    }

    ret = !DeviceIoControl(h, ctl, InBuffer, InBufferLength,
			   OutBuffer, OutBufferLength, &retval, NULL);
    if (ret) {
	Wwarn("DeviceIoControl %lx", ctl);
	errno = retval;
	ret = -1;
    } else
	ret = (int)retval;

    va_end(ap);

    return ret;
}

int
hxen_init(HANDLE h)
{
    int ret;

    ret = hxen_ioctl(h, KXENINIT);
    if (ret < 0) {
	warn("ioctl(KXENINIT)");
	goto out;
    }

    ret = 0;
out:
    return ret;
}

int
hxen_shutdown(HANDLE h)
{
    int ret;

    ret = hxen_ioctl(h, KXENSHUTDOWN);
    if (ret < 0) {
	warn("ioctl(KXENSHUTDOWN)");
	goto out;
    }

    ret = 0;
out:
    return ret;
}

int
hxen_load(HANDLE h, const char *filename)
{
    int ret;
    int file_fd = -1;
    struct stat statbuf;
    struct hxen_load_desc kld;
    uint8_t *uvaddr = NULL;

    fprintf(stderr, "loading hypervisor binary \"%s\"\n", filename);

    memset(&kld, 0, sizeof(kld));

    ret = open(filename, O_RDONLY | O_BINARY);
    if (ret < 0) {
	warn("open %s", filename);
	goto out;
    }
    file_fd = ret;

    ret = fstat(file_fd, &statbuf);
    if (ret) {
	warn("fstat %s", filename);
	goto out;
    }

    kld.kld_size = statbuf.st_size;
    uvaddr = (uint8_t *)malloc(kld.kld_size);
    if (uvaddr == NULL) {
	warn("malloc %d", kld.kld_size);
	ret = -1;
	goto out;
    }
    set_xen_guest_handle(kld.kld_uvaddr, uvaddr);

    ret = read(file_fd, uvaddr, kld.kld_size);
    if (ret != kld.kld_size) {
	warn("read %d from %s, got %d", kld.kld_size, filename, ret);
	goto out;
    }

    ret = hxen_ioctl(h, KXENLOAD, &kld, sizeof(kld));
    if (ret < 0) {
	warn("ioctl(KXENLOAD)");
	goto out;
    }

    ret = 0;
out:
    if (uvaddr)
	free(uvaddr);
    if (file_fd >= 0)
	close(file_fd);
    return ret;
}

int
hxen_unload(HANDLE h)
{
    int ret;

    ret = hxen_ioctl(h, KXENUNLOAD);
    if (ret < 0) {
	warn("ioctl(KXENUNLOAD)");
	goto out;
    }

    ret = 0;
out:
    return ret;
}

int
hxen_output_version_info(HANDLE h, FILE *f)
{
    int ret;
    struct hxen_version_desc kvd;
    struct hxen_hypercall_desc khd;
    xen_extraversion_t xen_extraversion;

    ret = hxen_ioctl(h, KXENVERSION, &kvd, sizeof(kvd));
    if (ret < 0) {
	warn("ioctl(KXENVERSION)");
	goto out;
    }
    if (f)
	fprintf(f, "hxen driver version %d.%d%c%s\n",
		kvd.kvd_driver_version_major, kvd.kvd_driver_version_minor,
		kvd.kvd_driver_version_tag[0] ? '-' : '\n',
		kvd.kvd_driver_version_tag);

    khd.khd_op = __HYPERVISOR_xen_version;
    khd.khd_arg[0] = XENVER_extraversion;
    khd.khd_arg[1] = (uint64_t)(uintptr_t)&xen_extraversion;
    ret = hxen_hypercall(h, &khd);
    if (ret < 0) {
	warn("hypercall(HYPERVISOR_xen_version,XENVER_extraversion)");
	ret = -1;
	goto out;
    }

    khd.khd_op = __HYPERVISOR_xen_version;
    khd.khd_arg[0] = XENVER_version;
    ret = hxen_hypercall(h, &khd);
    if (ret < 0) {
	warn("hypercall(HYPERVISOR_xen_version,XENVER_version)");
	ret = -1;
	goto out;
    }

    if (f)
	fprintf(f, "hxen core version %d.%d%.*s\n", ret >> 16, (uint16_t)ret,
		XEN_EXTRAVERSION_LEN, xen_extraversion);

    ret = 0;
  out:
    return ret;
}

int
hxen_trigger_keyhandler(HANDLE h, const char *keys)
{
    int ret;

    ret = hxen_ioctl(h, KXENKEYHANDLER, (void *)keys, strlen(keys));
    if (ret < 0)
	warn("ioctl(KXENKEYHANDLER)");

    return ret;
}

int
hxen_hypercall(HANDLE h, struct hxen_hypercall_desc *khd)
{
    int ret;

    ret = hxen_ioctl(h, KXENHYPERCALL, khd, sizeof(*khd));
    if (ret < 0)
	warn("ioctl(KXENHYPERCALL)");

    return ret;
}

int
hxen_hvmop(HANDLE h, struct hxen_hvmop_desc *khd)
{
    int ret;

    ret = hxen_ioctl(h, KXENHVMOP, khd, sizeof(*khd));
    if (ret < 0) {
	errno = -ret;
	warn("ioctl(KXENHVMOP,%x) failed: %d", khd->khd_cmd, errno);
    }

    return ret;
}

int
hxen_memop(HANDLE h, struct hxen_memop_desc *kmd)
{
    int ret;

    ret = hxen_ioctl(h, KXENMEMOP, kmd, sizeof(*kmd));
    if (ret < 0)
	warn("ioctl(KXENMEMOP)");

    return ret;
}

int
hxen_domctl(HANDLE h, struct hxen_domctl_desc *kdd)
{
    int ret;

    ret = hxen_ioctl(h, KXENDOMCTL, kdd, sizeof(*kdd));
    if (ret < 0)
	warn("ioctl(KXENDOMCTL)");

    return ret;
}

int
hxen_mmapbatch(HANDLE h, struct hxen_mmapbatch_desc *kmd)
{
    int ret;

    ret = hxen_ioctl(h, KXENMMAPBATCH, kmd, sizeof(*kmd));
    if (ret < 0)
	warn("ioctl(KXENMMAPBATCH)");

    return ret;
}

int
hxen_munmap(HANDLE h, struct hxen_munmap_desc *kmd)
{
    int ret;

    ret = hxen_ioctl(h, KXENMUNMAP, kmd, sizeof(*kmd));
    if (ret < 0)
	warn("ioctl(KXENMUNMAP)");

    return ret;
}

int
hxen_vmappings(HANDLE h, struct hxen_vmappings_desc *kmd)
{
    int ret;

    ret = hxen_ioctl(h, KXENVMAPPINGS, kmd, sizeof(*kmd));
    if (ret < 0)
	warn("ioctl(KXENVMAPPINGS)");

    return ret;
}

int
hxen_execute(HANDLE h)
{
    int ret;

    ret = hxen_ioctl(h, KXENEXECUTE);
    if (ret < 0)
	warn("ioctl(KXENEXECUTE)");

    return ret;
}

int
hxen_set_ioemu_events(HANDLE h, struct hxen_ioemu_events_desc *kied)
{
    int ret;

    ret = hxen_ioctl(h, KXENSETIOEMUEVENTS, kied, sizeof(*kied));
    if (ret < 0)
	warn("ioctl(KXENSETIOEMUEVENTS)");

    return ret;
}
