        page    ,132
        title   tprhandler - APIC TPR access handlers

;++
;
; Copyright (c) XenSource inc. All rights reserved.
;
; The system HAL is patched to not access the APIC (Advanced Programmable
; Interrupt Controller) TPR (Thread PRiority) register directly but to 
; instead call into the routines in this module.
;
; This allows us to manage the TPR locally rather than involving a trip
; to the hypervisor.  At the time of writing, a two processor (virtual)
; system running Windows 2003 SP1 accesses the TPR in excess of 110,000
; times per second.
;
; The instructions that access the TPR are a minimum of 5 bytes long
; which allows us to patch them with calls into the appropriate routines
; in this module.
;
; There are around 40 sites in the HAL which modify the TPR, some are
; only used at boot, we are only worried about those used frequently.
; 11 different instructions are used (in Windows 2003 SP1).
;
; code                  mnemonic                     sites   frequency
; c7058000feffc1000000  mov dword [0FFFE0080h],0C1h  3       many
; c7058000feffff000000  mov dword [0FFFE0080h],0FFh  2       3
; c7058000feff41000000  mov dword [0FFFE0080h],41h   5       very many
; 891d8000feff          mov dword [0FFFE0080h],ebx   1       2
; 890d8000feff          mov dword [0FFFE0080h],ecx   4       very many
; 89358000feff          mov dword [0FFFE0080h],esi   1       moderate
; a38000feff            mov dword [FFFE0080h],eax    4       many
; a18000feff            mov eax,dword [FFFE0080h]    10      very many
; 8b0d8000feff          mov ecx,dword [0FFFE0080h]   4       many
; 8b158000feff          mov edx,dword [0FFFE0080h]   5       many
; 8b358000feff          mov esi,dword [0FFFE0080h]   1       moderate
;
; Due to infrequency, we don't do the store 0xFF or ebx versions.
;
; Note: Unlike the real local APIC which is mapped at the same Virtual
; address regardless of processor, the virtualized writable apic is 
; at a different VA for each processor.  We need to disable interrupts
; before accessing that page or we risk a context switch moving us
; to another processor between determining which processor we are on
; and using that as an index for the information we want.
;

        .686p

        .model  FLAT

        extrn   @_tpr_store@4:near
        extrn   @_tpr_load@0:near

PROCSTART   macro   Func
            public  _tpr_&Func
_tpr_&Func  proc
endm

PROCEND     macro   Func
_tpr_&Func  endp
endm

        .code 

        PROCSTART store_const_0x3d

        pushfd                  ; save flags
        cli                     ; disable interrupts

        ; save volatile registers

        push    edx
        push    ecx
        push    eax

        ; set up arguments to general store handler

        mov     ecx, 03dh       ; value to set 

        jmp     short _tpr_store_common

        PROCEND store_const_0x3d

        PROCSTART store_const_0xd1

        pushfd                  ; save flags
        cli                     ; disable interrupts

        ; save volatile registers

        push    edx
        push    ecx
        push    eax

        ; set up arguments to general store handler

        mov     ecx, 0d1h       ; value to set

        jmp     short _tpr_store_common

        PROCEND store_const_0xd1

        PROCSTART store_const_0x41

        pushfd                  ; save flags
        cli                     ; disable interrupts

        ; save volatile registers

        push    edx
        push    ecx
        push    eax

        ; set up arguments to general store handler

        mov     ecx, 041h       ; value to set

_tpr_store_common:
        call    @_tpr_store@4

        ; restore volatile registers, interrupt state and return.
        ; note: we do not use this path if an interrupt was injected.
        ; note: the above note is not currently true but hopefully
        ;       will be one day.

        pop     eax
        pop     ecx
        pop     edx
        popfd
        ret

        PROCEND store_const_0x41

        PROCSTART store_const_0xc1

        pushfd                  ; save flags
        cli                     ; disable interrupts

        ; save volatile registers

        push    edx
        push    ecx
        push    eax

        ; set up arguments to general store handler

        mov     ecx, 0c1h       ; value to set

        jmp     _tpr_store_common

        PROCEND store_const_0xc1

        PROCSTART store_ecx

        pushfd                  ; save flags
        cli                     ; disable interrupts

        ; save volatile registers

        push    edx
        push    ecx
        push    eax

        ; set up arguments to general store handler


        jmp     _tpr_store_common

        PROCEND store_ecx

        PROCSTART store_eax

        pushfd                  ; save flags
        cli                     ; disable interrupts

        ; save volatile registers

        push    edx
        push    ecx
        push    eax

        ; set up arguments to general store handler

        mov     ecx, eax        ; value to set

        jmp     _tpr_store_common

        PROCEND store_eax

        PROCSTART store_esi

        pushfd                  ; save flags
        cli                     ; disable interrupts

        ; save volatile registers

        push    edx
        push    ecx
        push    eax

        ; set up arguments to general store handler

        mov     ecx, esi        ; value to set

        jmp     _tpr_store_common

        PROCEND store_esi

        PROCSTART load_eax

        ; save volatile registers and disable interrupts

        pushfd
        cli
        push    edx
        push    ecx

        call    @_tpr_load@0

        ; restore volatile registers

        pop     ecx
        pop     edx
        popfd

        ret

        PROCEND load_eax

        PROCSTART load_ecx

        ; save volatile registers and disable interrupts

        pushfd
        cli
        push    edx
        push    eax

        call    @_tpr_load@0

        ; move result to desired location

        mov     ecx, eax

        ; restore volatile registers

        pop     eax
        pop     edx
        popfd

        ret

        PROCEND load_ecx

        PROCSTART load_edx

        ; save volatile registers and disable interrupts

        pushfd
        cli
        push    ecx
        push    eax

        call    @_tpr_load@0

        ; move result to desired location

        mov     edx, eax

        ; restore volatile registers

        pop     eax
        pop     ecx
        popfd

        ret

        PROCEND load_edx

        PROCSTART load_esi

        ; save volatile registers and disable interrupts

        pushfd
        cli
        push    edx
        push    ecx
        push    eax

        call    @_tpr_load@0

        ; move result to desired location

        mov     esi, eax

        ; restore volatile registers

        pop     eax
        pop     ecx
        pop     edx
        popfd

        ret

        PROCEND load_esi

        ; keFlushEntireTbPatch and kxFlushEntireTbPatch are landing points for 
        ; the branch out of kernel code.  They differ in the restoration of
        ; registers and return instruction (to match the patched routine).

        extrn   _XenFlushCount:dword
        extrn   @_XenFlushEntireTb@0:near

        PROCSTART keFlushEntireTbPatch

        lock    inc dword ptr [_XenFlushCount+0]
        call    @_XenFlushEntireTb@0

        pop     esi
        pop     ebx
        pop     ebp
        ret     8

        PROCEND keFlushEntireTbPatch

        PROCSTART kxFlushEntireTbPatch

        lock    inc dword ptr [_XenFlushCount+4]
        call    @_XenFlushEntireTb@0

        pop     edi
        pop     esi
        pop     ebx
        pop     ecx
        pop     ebp
        ret

        PROCEND kxFlushEntireTbPatch

        ; keFlushSingleTbPatch is called in place of a call to KiIpiSendPacket
        ; gives us the advantage of not needing to save volatile registers.
        ; Also, we can just use the return instuction to fix the stack.

        extrn   @_XenFlushEntireTbRaw@0:near

        PROCSTART keFlushSingleTbPatch

        lock    inc dword ptr [_XenFlushCount+8]
        call    @_XenFlushEntireTbRaw@0

        ret     20

        PROCEND keFlushSingleTbPatch

        ; keFlushMultipleTbPatch is similar to keFlushSingleTb except that
        ; the code being patched is in line (the call to KiIpiSendPacket in
        ; KeFlushSingleTb is out of line) which means we need to mess with
        ; the return address to avoid the code we don't want to execute when
        ; we return.  ick.  To help simplify this case, rather
        ; than also skipping the ipi cleanup code we force the z flag to
        ; be set so the caller will think this is a single cpu system and
        ; skip the cleanup itself.

        PROCSTART keFlushMultipleTbPatch

        lock    inc dword ptr [_XenFlushCount+12]
        mov     eax, dword ptr [esp+8]
        add     dword ptr [esp], eax    ; adjust return address
        call    @_XenFlushEntireTbRaw@0
        cmp     eax, eax                ; set eflags.z

        ret     20

        PROCEND keFlushMultipleTbPatch

        ; keAcquireQueuedSpinLockAtDpcLevel
        ; keAcquireQueuedSpinLock
        ; kfAcquireSpinLock
        ;
        ; Both routines are patched in the "spin waiting for lock logic" to
        ; call thru here on the way to XenSource's wait loop which will yield
        ; to the hypervisor if the lock is not acuired in a reasonable time.
        ;
        ; In the former, ecx has the address of the lock queue structure, the
        ; bit of interest is the low bit at ecx+4.  Once the bit is clear, the
        ; patched code executes a return instruction, nothing needs cleaning up.
        ;
        ; KeAcquireQueuedSpinLock raises IRQL to DISPATCH level and the previous
        ; IRQL is returned.  It has been pushed to the stack.  

        extrn   @XenWaitForZeroBit@8:near

        PROCSTART keAcquireQueuedSpinLockAtDpcLevel

        push    1
        pop     edx
        lea     ecx, [ecx+4]
        jmp     @XenWaitForZeroBit@8

        PROCEND keAcquireQueuedSpinLockAtDpcLevel

        PROCSTART keAcquireQueuedSpinLock

        push    1
        pop     edx
        lea     ecx, [eax+4]
        call    @XenWaitForZeroBit@8
        pop     eax
        ret

        PROCEND keAcquireQueuedSpinLock

        PROCSTART keAcquireSpinLockRaiseToSynch

        push    eax
        push    ecx
        push    1
        pop     edx
        call    @XenWaitForZeroBit@8
        xor     eax, eax            ; set ZF
        pop     ecx
        pop     eax
        ret

        PROCEND keAcquireSpinLockRaiseToSynch

        ; kdCheckDebuggerEnabled
        ;
        ; This routine replaces the instruction in KdPollBreakIn which checks
        ; to see if the kernel debugger is enabled and only returns true once
        ; per clock tick (1/100th second).  This occurs on a timer interrupt
        ; and causes us to go to dom0 too often if the timer frequency gets
        ; cranked up, this routine slows the debugger poll back to the regular
        ; rate.
        ;
        ; The return value is the state of the eflags register.  ZF means
        ; no debugger, at least not this time.

        extrn   __gp_kdDebuggerEnabled:dword
        extrn   __g_kdCheckDebuggerTick:dword
        extrn   _KeTickCount:dword

        PROCSTART kdCheckDebuggerEnabled
 
        mov     eax, [__gp_KdDebuggerEnabled]
        mov     ecx, [_KeTickCount]

        ; If the debugger actually isn't enabled, always return zf.

        cmp     byte ptr [eax],0
        je      short @f

        ; Debugger enabled, has KeTickCount advanced?

        mov     ecx, [ecx]
        cmp     ecx, [__g_kdCheckDebuggerTick]
        je      short @f

        ; Yes, record the new tick value and return with ZF clear.

        mov     [__g_kdCheckDebuggerTick], ecx
@@:     ret

        PROCEND kdCheckDebuggerEnabled

        ; mov_to_cr8 / mov_from_cr8
        ;
        ; AMD processors support access to cr8 in 32 bit mode by using a
        ; lock prefix on a mov to/from cr0 instruction.  cr8 is equivalent
        ; to the APIC TPR.
        ;
        ; Note: The assembler doesn't allow the lock prefix so we have to
        ; fake it.

        public  @mov_to_cr8@4
@mov_to_cr8@4   proc

        db      0f0h
        mov     cr0, ecx
        ret

@mov_to_cr8@4 endp


        public  @mov_from_cr8@0
@mov_from_cr8@0 proc

        db      0f0h
        mov     eax, cr0
        ret

@mov_from_cr8@0 endp

        end
