/* Various wrappers so that we can share code between ordinary xenvbd
   and ntbootdd.sys xenvbd */
/* Note that the ntbootdd versions of these functions are often much
   less functional than the normal version.  In particular, WaitEvent
   always returns timeout and InsertQueueDpc runs the DPC immediately
   rather than deferring to anything.  It happens that this is the
   correct behaviour for all of the code which currently uses these
   wrappers. */
/* (Remember that you can't reschedule from ntbootdd) */
#ifndef SCSIBOOT_H__
#define SCSIBOOT_H__

#include <stdarg.h>

//#include "xscompat.h"

//BOOLEAN GetAustereMode(VOID);
#define GetAustereMode() (0)
VOID SetAustereMode(BOOLEAN x);

#define AustereMode GetAustereMode()


BOOLEAN ChechXenHypervisor(void);
VOID XenCpuid(ULONG leaf, ULONG *peax, ULONG *pebx, ULONG *pecx,
            ULONG *pedx);
VOID _cpuid(ULONG leaf, ULONG *peax, ULONG *pebx, ULONG *pecx,
            ULONG *pedx);
void XenTraceFlush(void);
void XenTraceSetLevels(const int *levels);
ULONG HvmGetLogRingSize(void);
NTSTATUS HvmGetLogRing(void *buffer, ULONG size);
ULONG XmExtractTailOfLog(char *outbuf, ULONG max_size);
void UnplugIoemu(void);
struct _OSVERSIONINFOEXW;
VOID XenutilGetVersionInfo(struct _OSVERSIONINFOEXW *out);

/**********************************************************************/
/* Memory allocation functions */
PVOID _XmAllocateMemory(size_t size, const char *file, int line);
PVOID _XmAllocateZeroedMemory(size_t size, const char *file, int line);

/* Allocate x bytes of non-paged pool.  Guaranteed to be page aligned
   if x >= PAGE_SIZE. */
#define XmAllocateMemory(x) _XmAllocateMemory((x), __FILE__, __LINE__)

/* Like XmAllocateMemory(), but zero the memory on success. */
#define XmAllocateZeroedMemory(x) _XmAllocateZeroedMemory((x), __FILE__, __LINE__)

/* Like XmAllocateMemory, but also return the physical address of the
   memory. */
PVOID XmAllocatePhysMemory(size_t size, PHYSICAL_ADDRESS *pa);
/* XmFreeMemory(x) releases memory obtained via XmAllocateMemory or
   XmAllocatePhysMemory. */
VOID XmFreeMemory(PVOID ptr);


/*********************************************************************/
/* Standard debug support functions */

/* The first argument must be a format.  We include it in the ... so
   that __VA_ARGS__ works even if XmBugCheck()'s format doesn't have
   any arguments. */
DECLSPEC_NORETURN void _XmBugCheck(const char *file, unsigned line, ...);

DECLSPEC_NORETURN void _XmAssertFail(const char *file, unsigned line,
                                     const char *expr);

DECLSPEC_NORETURN void _XmBug(const char *file, unsigned line,
                              const char *expr, ULONG_PTR val);

#define XmBugCheck(...) _XmBugCheck(__FILE__, __LINE__, __VA_ARGS__)

/* Assert (x) != 0 */
#define XM_ASSERT(x)                                         \
do {                                                         \
    if (!(x)) {                                              \
        _XmAssertFail(__FILE__, __LINE__, #x );              \
    }                                                        \
} while (0)

/* Assert x == 0 */
#define XM_BUG_ON(x)                                         \
do {                                                         \
    ULONG_PTR __bug_y = (ULONG_PTR)(x);                      \
    if (__bug_y) {                                           \
        _XmBug(__FILE__, __LINE__, #x , __bug_y);            \
    }                                                        \
} while (0)

/* Warn if x != 0 */
#define XM_WARN_ON(x)                                        \
do {                                                         \
    ULONG_PTR __warn_y = (ULONG_PTR)(x);                     \
    if (__warn_y) {                                          \
        TraceWarning(("%s is 0x%p, should be 0 at %s:%d\n",  \
                       #x , __warn_y, __FILE__, __LINE__));  \
    }                                                        \
} while (0)

/********************************************************************/
/* String manipulation functions */

size_t Xmvsnprintf(char *buf, size_t size, const char *fmt,
                   va_list args);

/* snprintf */
static __inline size_t
Xmsnprintf(char *s, size_t n, const char *fmt, ...)
{
    va_list args;
    size_t r;
    va_start(args, fmt);
    r = Xmvsnprintf(s, n, fmt, args);
    va_end(args);
    return r;
}

char *Xmasprintf(const char *fmt, ...);
char *Xmvasprintf(const char *fmt, va_list args);

/* Locks which can be acquired from interrupt context. */
struct irqsafe_lock {
    LONG lock;
};

static __inline KIRQL
acquire_irqsafe_lock(struct irqsafe_lock *l)
{
    KIRQL irql;
    KeRaiseIrql(HIGH_LEVEL, &irql);
    while (InterlockedCompareExchange(&l->lock, 1, 0) != 0)
        _mm_pause();
    _ReadWriteBarrier();
    return irql;
}

static __inline VOID
release_irqsafe_lock(struct irqsafe_lock *l, KIRQL irql)
{
    _ReadWriteBarrier();
    InterlockedExchange(&l->lock, 0);
    KeLowerIrql(irql);
}

static __inline NTSTATUS
try_acquire_irqsafe_lock(struct irqsafe_lock *l, KIRQL *pirql)
{
    KIRQL irql;
    KeRaiseIrql(HIGH_LEVEL, &irql);
    if (InterlockedCompareExchange(&l->lock, 1, 0) == 0) {
        _ReadWriteBarrier();
        *pirql = irql;
        return STATUS_SUCCESS;
    }
    KeLowerIrql(irql);
    return STATUS_UNSUCCESSFUL;
}

/* Slightly easier-to-work-with abstraction around
 * PsCreateSystemThread */
struct xm_thread {
    BOOLEAN exit; /* TRUE if the thread should exit soon */
    KEVENT event; /* signalled whenever there's work for the thread to
                     do or when exit becomes TRUE */
    PKTHREAD thread;
    NTSTATUS (*cb)(struct xm_thread *st, void *data);
    void *data;
};

/* create a new thread, invoking cb(thread, @d) in it.  Calls
 * PsTerminateThread() automatically using the return value of @cb.
 * Returns NULL on error.
 */
struct xm_thread *XmSpawnThread(NTSTATUS (*cb)(struct xm_thread *,void*),
                                void *d);
/* Tell a thread to exit, wait for it to do so, and then release all
 * associated resources.  No-op if invoked on NULL. */
void XmKillThread(struct xm_thread *t);
/* Wait on the thread's event and then clear it.  Returns -1 if the
   thread should exit and 0 otherwise.  Intended to be called from the
   thread's main loop. */
int XmThreadWait(struct xm_thread *t);



/* A few functions for manipulating linked lists. */

/* Make the list anchored at @sublist appear at the end of the list
   anchored at @list.  @sublist itself does not appear in the new
   list.  @sublist is not a valid list when this returns. */
void XmListSplice(PLIST_ENTRY list, PLIST_ENTRY sublist);

/* Take all of the list elements in @src and move them to @dest.  @src
   is reinitialised as an empty list.  Any elements currently in @dest
   are discarded. */
void XmListTransplant(PLIST_ENTRY dest, PLIST_ENTRY src);

/*****************************************************************/
/* Version checking.  We don't support running with mixed versions of
 * the drivers, but we can at least make the failure
 * comprehensible. */
/* These are stable ABIs, even though they're xensource-private. */
#define XEVTCHN_CURRENT_VERSION 1
ULONG XmGetXevtchnVersion(void);
ULONG XmGetXenutilVersion(void);
/* Used by xevtchn to tell xenutil that there's been a version
   mismatch.  This usually leads to a bugcheck.  Nobody else should be
   calling this. */
void XmXenutilVersionMismatch(ULONG xevtchn_version);
/* Used by xenvbd to tell xenutil about a version mismatch.  This
   usually leads to a bugcheck. */
void XmXenvbdVersionMismatch(ULONG xevtchn_expected_version);

/*****************************************************************/
/* Feature flags, controlled through boot.ini */
#define DEBUG_NO_PARAVIRT               0x00000001
#define DEBUG_UNUSED1                   0x00000002  // DEBUG_PATCH_PIT (removed in 4.2)             
#define DEBUG_UNUSED2                   0x00000004  // DEBUG_PATCH_CIRRUS (removed in 4.2)            
#define DEBUG_PATCH_BLUESCREEN          0x00000008
#define DEBUG_DUMP_DEVICE_DOM0          0x00000010
#define DEBUG_UNUSED3                   0x00000020  // DEBUG_PATCH_HIBERNATE (removed in 4.2)
#define DEBUG_PATCH_APIC                0x00000040
#define DEBUG_NIC_FAST_AND_LOOSE        0x00000080
#define DEBUG_VERY_LOUD                 0x00000100
#define DEBUG_BALLOON                   0x00000200
#define DEBUG_VERY_QUIET                0x00000400
#define DEBUG_PATCH_TLB_FLUSH           0x00000800
#define DEBUG_PATCH_SPINLOCKS           0x00001000
#define DEBUG_FAKE_NETIF                0x00002000
#define DEBUG_NIC_8021_P                0x00004000  // xennet6 only
#define DEBUG_PATCH_KD_POLL             0x00008000
#define DEBUG_NIC_NO_TSO                0x00010000
#define DEBUG_NIC_NO_DMA                0x00020000
#define DEBUG_FORCE_EARLY_UNPLUG        0x00040000
#define DEBUG_PATCH_2K_IDLE_DELAY       0x00080000
#define DEBUG_HA_SAFEMODE               0x00100000  // xevtchn.sys loads in safe mode but no xennet or xenvbd.
#define DEBUG_BOOT_EMULATED             0x00200000  // Use an emulated c:
#define DEBUG_HCT_DP_HACKS              0x00400000  // Stupid HCT DP test hacks
#define DEBUG_TRACE_PROCESS_CREATE      0x00800000  // Traces PID and CR3 for a new process.
#define DEBUG_TRACE_IMAGE_LOAD          0x01000000  // Traces process name and PID on executable load.
#define DEBUG_INTERNAL_XENNET           0x02000000  // Report xennet on INTERNAL bus.
#define DEBUG_TRAP_DBGPRINT             0x04000000  // Try to trap calls to DbgPrint()

BOOLEAN 
XenPVFeatureEnabled(
    ULONG FeatureFlag
    );

#ifndef INVALID_HANDLE_VALUE
#define INVALID_HANDLE_VALUE ((HANDLE)-1)
#endif

#endif /* SCSIBOOT_H__ */
