//
// xenevtchn.h
//
// Copyright (c) 2006 XenSource, Inc. - All rights reserved.
//

#ifndef _XENEVTCHN_H_
#define _XENEVTCHN_H_

#define DEBUG 1

#define TEST_DEVICE_ENUMERATION 1

#define XENEVTCHN_VERSION "1.0"

#include "ntddk.h"
#include "wchar.h"
#include "xsapi.h"
#include "xs_ioctl.h"

#define XENEVTCHN_DEVICE_NAME L"\\Device\\xenevent"
#define XENEVTCHN_FILE_NAME  L"\\DosDevices\\XenBus"

#define XENEVTCHN_PDO_SIGNATURE 0x7556b3a5
#define XENEVTCHN_FDO_SIGNATURE 0x7556b3a6

typedef struct _XENEVTCHN_DEVICE_HDR {
    ULONG   Signature;
} XENEVTCHN_DEVICE_HDR, *PXENEVTCHN_DEVICE_HDR;

struct xenbus_device_class {
    struct _XENEVTCHN_DEVICE_EXTENSION *devExt;
    LIST_ENTRY classes; /* List of all device_classes, anchored at
                           DEVICE_EXTENSION::xenbus_device_classes */
    LIST_ENTRY devices; /* Anchor for list of devices in this class,
                           threaded through
                           xenbus_device::devices_this_class */
    char *class;
};

/* See xenbus_device_lifecycle.txt */
struct xenbus_device {
    struct _XENEVTCHN_DEVICE_EXTENSION *devExt;
    LIST_ENTRY devices_this_class; /* List of xenbus_device
                                      structures, anchored at
                                      xenbus_device_class::devices */
    LIST_ENTRY all_devices; /* List of xenbus_device structures,
                               anchored at
                               DEVICE_EXTENSION::devices */
    struct xenbus_device_class *class;
    char *name;
    LONG refcount;
    PDEVICE_OBJECT pdo;
    FAST_MUTEX pdo_lock;
    PWCHAR InstanceId;
    PWCHAR DeviceId;
    PWCHAR HardwareId;
    char *frontend_path; /* path to the frontend's area in the store. */
    struct xenbus_watch_handler *bstate_watch; /* Watch on
                                                * @backend_path/state */
    LARGE_INTEGER removal_time; /* Don't report the device as removed
                                   until this time, even if
                                   present_in_xenbus has gone to zero,
                                   to avoid races in Windows
                                   userspace. */
    unsigned present_in_xenbus:1;
    unsigned reported_to_pnpmgr:1;
    unsigned lost_watch_event:1;
    unsigned failed:1;
    unsigned backend_watch_stale:1;
    unsigned pdo_ready:1;
};

#define XS_DELAY_REMOVAL_US 5000000

struct PDO_DEVICE_EXTENSION {
    XENEVTCHN_DEVICE_HDR Header;
    struct xenbus_device *dev;
    struct _XENEVTCHN_DEVICE_EXTENSION *fdo;
};

//
// The following structure contains device-specific information
// that is collected during the creation and activation process.
//
typedef struct _XENEVTCHN_DEVICE_EXTENSION {
    XENEVTCHN_DEVICE_HDR Header;
    PDRIVER_OBJECT DriverObject;
    PDEVICE_OBJECT DeviceObject;
    PDEVICE_OBJECT LowerDeviceObject;
    PDEVICE_OBJECT PhysicalDeviceObject;

    FAST_MUTEX xenbus_lock; /* Protects the device and device class
                               lists.  Leaf lock. */
    KEVENT initial_probe_complete;

    struct xm_thread *reprobe_xenbus_thread;

    LIST_ENTRY xenbus_device_classes;
    LIST_ENTRY devices;
    /* Device removal has to be delayed by a few milliseconds to work
       around a race in Windows userspace. */
    KTIMER xenbus_timer;
    KDPC xenbus_dpc;

    LIST_ENTRY ActiveHandles;
    BOOLEAN UninstEnabled;

    PKINTERRUPT Interrupt;
} XENEVTCHN_DEVICE_EXTENSION, *PXENEVTCHN_DEVICE_EXTENSION;

/* A representation of a userspace watch. */
typedef struct _USER_WATCH_HANDLE {
    struct _USER_WATCH_HANDLE *next, *prev;
    int handle; /* Handle reported to userspace. */
    PKEVENT evt;
    struct xenbus_watch_handler *wh;
} USER_WATCH_HANDLE, *PUSER_WATCH_HANDLE;

struct xenevtchn_msg_port 
{
    LIST_ENTRY list;
    LONG refcnt;

    unsigned handle;
    PKEVENT event;
    struct XenbusMsgPort *port;

    /* Protects @pending. */
    FAST_MUTEX mux;

    /* We need to keep one message around so that userspace can find
       out what size buffer it needs to allocate. */
    struct
    {
        BOOLEAN valid;
        char *source;
        unsigned source_len;
        void *message;
        unsigned message_len;
    } pending;
};

typedef struct _XENEVTCHN_ACTIVE_HANDLE {
    LIST_ENTRY ListEntry; /* This must be the first field */
    PFILE_OBJECT FileObject;
    xenbus_transaction_t xbt;
    PKEVENT suspend_event;
    KSPIN_LOCK watches_lock; /* Leaf lock */
    PUSER_WATCH_HANDLE watches;
    BOOLEAN precious; /* Warn on close */

    LONG next_port_handle;
    KSPIN_LOCK port_lock; /* Leaf lock */
    LIST_ENTRY ports; /* List of xenevtchn_msg_port, threaded on
                       * @list. */
} XENEVTCHN_ACTIVE_HANDLE, *PXENEVTCHN_ACTIVE_HANDLE;

struct evtchn_port_handler {
    struct evtchn_port_handler *next, *prev;
    int xen_port;
    PEVTCHN_HANDLER_CB handler;
    void *handler_context;

    enum {
        EVTCHN_CLASS_INTERDOMAIN,
        EVTCHN_CLASS_INTERDOMAIN_DPC,
        EVTCHN_CLASS_FIXED,
        EVTCHN_CLASS_VIRQ
    } class;
    union {
        DOMAIN_ID interdomain;
        int virq;
        struct {
            DOMAIN_ID dom;
            KDPC dpc;

            /* DPC insertion control field.  This can be either 0 (DPC
               neither queued nor running), 1 (DPC either queued or
               running, but not both), or 2 (DPC both queued and
               currently running).  During normal operation, it is
               only accessed from CPU 0, but it may be accessed from
               another CPU if the port is stopped. */
            /* (Actually, we decrement it a little bit early when the
               DPC stops running.  It is, however, guaranteed that we
               won't touch the port again from the DPC after we've
               decremented the count. (Unless it gets inserted again,
               of course.)) */
            /* Think of it this way: there are two interesting and
             * orthoginal ``real'' state bits: is the DPC queued, and
             * is the DPC currently running.  Denote queued by Q, not
             * queued by q, running by R, and not running by r (where
             * running means ``could conceivably touch the port
             * structure before returning'').  You then get a state
             * transition diagram which looks like this:
             *
             *         F
             * +<-------------<+
             * v               ^
             * |               |
             * v   I       D   ^
             * qr >--> Qr >--> qR
             *         ^       v
             *         |       | I
             *         |       |
             *         ^   F   v
             *         +<-----<QR
             *
             * Where I is the interrupt queueing the DPC, D is the DPC
             * starting, and F is the DPC finishing with the port
             * structure.  If the interrupt fires in Qr or QR then
             * nothing happens, because the DPC is already queued (so
             * the I transition can't happen, even though the
             * interrupt has fired).  Likewise, D can't happen in a q
             * state, and F can't happen in an r state, so this
             * diagram covers all possible transitions.
             *
             * We abstract the state machine like so:
             *
             * qr -> 0
             * Qr -> 1
             * qR -> 1
             * QR -> 2
             *
             * because I and F are much easier to observe than D, and
             * we don't really care about D transitions.  That gives
             * us an abstract state machine which looks like this:
             *
             *       F
             * +<----------<+
             * v            ^
             * v     I      ^      I
             * 0 >--------> 1 >---------> 2
             *              ^             v
             *              ^             v
             *              +<-----------<+
             *                     F
             *
             * In other words, we increment the count every time we
             * queue the DPC, and decrement it every time it finishes.
             */
            LONG insert_count;
        } interdomain_dpc;
    } u;
};
MAKE_WRAPPER_PRIV(EVTCHN_PORT, struct evtchn_port_handler *)

PVOID XenevtchnAllocIoMemory(ULONG nr_bytes, PHYSICAL_ADDRESS *pa);
NTSTATUS GnttabInit(void);
VOID GnttabCleanup(void);
NTSTATUS balloon_init(void);
BOOLEAN balloonActive(void);
void DescheduleVcpu(unsigned ms);
void balloon_freeze(void);
void balloon_thaw(void);
void balloon_query(XS_QUERY_BALLOON *xsqb);

ULONG64 HvmGetXenTime(void);

VOID XenevtchnShutdownIoHole(void);

NTSTATUS XenbusRecvMessageUserspace(struct XenbusMsgPort *port,
                                    PSTR *sending_domain,
                                    void **message,
                                    size_t *message_size);
void xenbus_fail_transaction(xenbus_transaction_t xbt, NTSTATUS status);
ULONG64 xm_strtoll(const char *buf, size_t len, unsigned base,
                   __inout int *err);

extern PDEVICE_OBJECT XenevtchnPdo;
extern int xenbusBusInterfaceRefcount;
extern BOOLEAN EvtchnStarted;

extern const BUS_INTERFACE_STANDARD xenbusBusInterface;

NTSTATUS
BinPatchPhase0();

NTSTATUS
BinPatchPhase1();

VOID UnpatchAPIC(VOID);
void ApicRecoverFromSuspend(void);

NTSTATUS
InstallDumpDeviceCallback();

void XsRequestInvalidateBus(PKDPC dpc, PVOID arg1, PVOID arg2, PVOID arg3);

extern unsigned suspend_count;

NTSTATUS UpdateMachineName();

void PnpRecoverFromHibernate(PXENEVTCHN_DEVICE_EXTENSION pXevtdx);

char *FindBackendPath(xenbus_transaction_t xbt, struct xenbus_device *xd);
struct xenbus_device *GetXenbusDeviceForPdo(PDEVICE_OBJECT pdo);

#if !defined(AMD64)
void FASTCALL mov_to_cr8(ULONG NewPriority);
ULONG  FASTCALL mov_from_cr8(void);
#endif

void CleanupDebugHelpers(void);
void DisconnectDebugVirq(void);
void InitDebugHelpers(void);

extern BOOLEAN XenbusFrozen;

#endif // _XENEVTCHN_H_

