#include <ntddk.h>
#include <stdarg.h>
#include <ntstrsafe.h>
#include "xsapi.h"
#include "scsiboot.h"
//#include "vbd_special.h"
#include "verinfo.h"

#define XUTIL_TAG 'LUTX'

#define DEBUG_DEFAULT_FEATURE_SET (                               \
                                    DEBUG_PATCH_APIC            | \
                                    DEBUG_BALLOON               | \
                                    DEBUG_NIC_FAST_AND_LOOSE    | \
                                    DEBUG_PATCH_TLB_FLUSH       | \
                                    DEBUG_PATCH_SPINLOCKS       | \
                                    DEBUG_PATCH_2K_IDLE_DELAY   | \
                                    DEBUG_PATCH_KD_POLL         | \
                                    DEBUG_FORCE_EARLY_UNPLUG    | \
                                    DEBUG_INTERNAL_XENNET       | \
                                    DEBUG_TRAP_DBGPRINT         | \
                                    0                             \
                                  )

static ULONG    _g_PVFeatureFlags   = DEBUG_DEFAULT_FEATURE_SET;

static BOOLEAN done_unplug; /* True if we've done a device unplug.
                               This is not reset when we hibernate, so
                               is only an approximation to whether the
                               emulated devices are actually
                               present. */

/* We keep a ring buffer of recent messages which is included in
   kernel crashdumps. */
#define LOG_MSG_BUFFER_SIZE 8192
static char LogMessageBuffer[LOG_MSG_BUFFER_SIZE];
static ULONG LogMessageBufferPtr;
extern PULONG InitSafeBootMode;

#if defined(KXEN_GUEST)
void *
_XmAllocateMemory(size_t ByteCount, const char * CallerFile, int CallerLine)
{
    return ExAllocatePoolWithTag(NonPagedPool, ByteCount, 'NEXK');
}
void
XmFreeMemory(void * Thing)
{
    ExFreePoolWithTag(Thing, 'NEXK');
}
#endif

static void
xm_thread_func(void *data)
{
    struct xm_thread *me = data;

    PsTerminateSystemThread(me->cb(me, me->data));
}

struct xm_thread *
XmSpawnThread(NTSTATUS (*cb)(struct xm_thread *xt, void *d), void *d)
{
    struct xm_thread *work;
    NTSTATUS stat;
    HANDLE tmpHandle;

    work = XmAllocateMemory(sizeof(*work));
    if (!work) return NULL;
    work->exit = FALSE;
    KeInitializeEvent(&work->event, NotificationEvent, FALSE);
    work->cb = cb;
    work->data = d;
    stat = PsCreateSystemThread(&tmpHandle, THREAD_ALL_ACCESS, NULL,
                                INVALID_HANDLE_VALUE, NULL, xm_thread_func,
                                work);
    if (!NT_SUCCESS(stat)) {
        XmFreeMemory(work);
        return NULL;
    }
    stat = ObReferenceObjectByHandle(tmpHandle, SYNCHRONIZE, NULL,
                                     KernelMode, &work->thread,
                                     NULL);
    ZwClose(tmpHandle);
    if (!NT_SUCCESS(stat)) {
        /* We can't reliably kill the thread in this case, and
           therefore can't release memory.  Instruct it to exit soon
           and hope for the best. */
        work->exit = TRUE;
        KeSetEvent(&work->event, IO_NO_INCREMENT, FALSE);
        return NULL;
    }

    return work;
}

void
XmKillThread(struct xm_thread *t)
{
    if (!t)
        return;
    TraceLoud (("Killing thread %p.\n", t));
    t->exit = TRUE;
    KeSetEvent(&t->event, IO_NO_INCREMENT, FALSE);
    KeWaitForSingleObject(t->thread, Executive, KernelMode, FALSE,
                          NULL);
    ObDereferenceObject(t->thread);
    XmFreeMemory(t);
}

int
XmThreadWait(struct xm_thread *t)
{
    KeWaitForSingleObject(&t->event,
                          Executive,
                          KernelMode,
                          FALSE,
                          NULL);
    KeClearEvent(&t->event);
    if (t->exit)
        return -1;
    else
        return 0;
}

/* Make the list anchored at @sublist appear at the end of the list
   anchored at @list.  @sublist itself does not appear in the new
   list.  @sublist is not a valid list when this returns. */
void
XmListSplice(PLIST_ENTRY list, PLIST_ENTRY sublist)
{
    PLIST_ENTRY sub_first, sub_last;

    if (IsListEmpty(sublist)) {
        /* sublist is empty -> nothing to do */
        return;
    }

    sub_first = sublist->Flink;
    sub_last = sublist->Blink;

    list->Blink->Flink = sub_first;
    sub_first->Blink = list->Blink;
    list->Blink = sub_last;
    sub_last->Flink = list;
}

/* Take all of the list elements in @src and move them to @dest.  @src
   is reinitialised as an empty list. */
void
XmListTransplant(PLIST_ENTRY dest, PLIST_ENTRY src)
{
    InitializeListHead(dest);
    XmListSplice(dest, src);
    InitializeListHead(src);
}

static ULONG XenCPUIDBaseLeaf = 0x40000000;

VOID
XenCpuid(ULONG leaf, ULONG *peax, ULONG *pebx, ULONG *pecx, ULONG *pedx)
{
    _cpuid(leaf + XenCPUIDBaseLeaf, peax, pebx, pecx, pedx);
}

BOOLEAN
ChechXenHypervisor(void)
{
    ULONG eax, ebx ='fool', ecx = 'beef', edx = 'dead';
    char signature[13];

    //
    // Check that we're running on Xen and that CPUID supports leaves up to
    // at least 0x40000002 which we need to get the hypercall page info.
    //
    // Note: The Xen CPUID leaves may have been shifted upwards by a
    // multiple of 0x100.
    //

    for (; XenCPUIDBaseLeaf <= 0x40000100; XenCPUIDBaseLeaf += 0x100)
    {
        _cpuid(XenCPUIDBaseLeaf, &eax, &ebx, &ecx, &edx);

        *(ULONG*)(signature + 0) = ebx;
        *(ULONG*)(signature + 4) = ecx;
        *(ULONG*)(signature + 8) = edx;
        signature[12] = 0;
        if ((strcmp("XenVMMXenVMM", signature) == 0) &&
                (eax >= (XenCPUIDBaseLeaf + 2)))
            return TRUE;
    }
    return FALSE;
}

NTSTATUS
XenReadRegistryKey(
    IN  UNICODE_STRING szRegPath,
    IN  UNICODE_STRING szRegValueName,
    OUT PKEY_VALUE_PARTIAL_INFORMATION *ppKeyInfoValue
    )
{
    NTSTATUS Status = STATUS_SUCCESS;
    ULONG ulKeyValueSize = 0;
    ULONG ulReturnLength;
    OBJECT_ATTRIBUTES attrs;
    HANDLE hKey;
    
    ASSERT (*ppKeyInfoValue == NULL);

    if (*ppKeyInfoValue != NULL)
    {
        TraceError (("ppKeyInfoValue invalid?\n"));
        return STATUS_INVALID_PARAMETER;
    }

    InitializeObjectAttributes(
            &attrs, 
            &szRegPath, 
            OBJ_KERNEL_HANDLE | OBJ_CASE_INSENSITIVE, 
            NULL, 
            NULL);
            
    Status = ZwOpenKey(&hKey, KEY_READ, &attrs);    
    if (NT_SUCCESS(Status))
    {
        Status = ZwQueryValueKey(
                    hKey,
                    &szRegValueName,
                    KeyValuePartialInformation,
                    *ppKeyInfoValue,
                    ulKeyValueSize,
                    &ulReturnLength
                    );

        ASSERT (Status != STATUS_SUCCESS);

        if (Status == STATUS_BUFFER_TOO_SMALL)
        {
            ulKeyValueSize = ulReturnLength + sizeof (KEY_VALUE_BASIC_INFORMATION);

            *ppKeyInfoValue = (PKEY_VALUE_PARTIAL_INFORMATION) ExAllocatePoolWithTag (NonPagedPool, ulKeyValueSize, XUTIL_TAG);
            if (*ppKeyInfoValue == NULL)
            {
                Status = STATUS_NO_MEMORY;
                goto ErrorExit;
            }   

            RtlZeroMemory (*ppKeyInfoValue, ulKeyValueSize);

            Status = ZwQueryValueKey(
                           hKey,
                           &szRegValueName,
                           KeyValuePartialInformation,
                           *ppKeyInfoValue,
                           ulKeyValueSize,
                           &ulReturnLength
                           );
        }
            
        ZwClose(hKey);
    }   

    return Status;

ErrorExit:

    ZwClose (hKey);

    return Status;
}

BOOLEAN
XenPVFeatureEnabled(
    IN ULONG    FeatureFlag
)
{
    return (BOOLEAN)((_g_PVFeatureFlags & FeatureFlag) != 0);
}

static ULONG
wcstoix(__in PWCHAR a, __out PBOOLEAN err)
{
    ULONG acc = 0;
    if (err) *err = FALSE;
    if (!a || !a[0]) *err = TRUE;

    while (a[0] && a[0] != L' ') {
        if (a[0] >= L'0' && a[0] <= L'9')
            acc = acc * 16 + a[0] - L'0';
        else if (a[0] >= L'a' && a[0] <= L'f')
            acc = acc * 16 + a[0] - L'a' + 10;
        else if (a[0] >= L'A' && a[0] <= L'F')
            acc = acc * 16 + a[0] - L'A' + 10;
        else
        {
            if (err) *err = TRUE;
            break;
        }
        
        a++;
    }
    return acc;
}

/* Read a DWORD out of xenevtchn's parameters key. */
static NTSTATUS
ReadRegistryParameterDword(const WCHAR *value, ULONG *out)
{
    UNICODE_STRING path;
    UNICODE_STRING valueStr;
    NTSTATUS stat;
    PKEY_VALUE_PARTIAL_INFORMATION pKeyValueInfo;

    pKeyValueInfo = NULL;

    RtlInitUnicodeString(&valueStr, value);
    RtlInitUnicodeString(&path,
                         L"\\Registry\\Machine\\SYSTEM\\CurrentControlSet\\Services\\xenevtchn\\Parameters");
    stat = XenReadRegistryKey(path, valueStr, &pKeyValueInfo);
    if (NT_SUCCESS(stat))
    {
        if (pKeyValueInfo->Type == REG_DWORD)
        {
            *out = *((ULONG*)pKeyValueInfo->Data);
        }
        else
        {
            TraceError(("Expected %wZ to be a DWORD, actually type %x\n",
                        &valueStr, pKeyValueInfo->Type));
            stat = STATUS_INVALID_PARAMETER;
        }
        ExFreePool(pKeyValueInfo);
    }
    return stat;
}

/* Read a flags field out of the registry.  Returns 0 if there is any
   error reading the key.  @value says which value to read; it is
   pulled out of the xenevtchn service's parameters key. */
static ULONG
ReadRegistryFlagsOrZero(PWCHAR value)
{
    ULONG res;
    NTSTATUS stat;

    stat = ReadRegistryParameterDword(value, &res);
    if (NT_SUCCESS(stat))
        return res;
    else
        return 0;
}

static VOID
XenParseBootParams(void)
{
    NTSTATUS        Status;
    PKEY_VALUE_PARTIAL_INFORMATION pKeyValueInfo = NULL;
    UNICODE_STRING  usControlPath;
    UNICODE_STRING  usStartOptions;
    UNICODE_STRING  usNOPVBoot;
    UNICODE_STRING  registryPath;

    PWCHAR          wzReturnValue = NULL;
    PWCHAR          wzPVstring;
    
    ULONG           features;
    BOOLEAN         bError;
    BOOLEAN         bDefault;

    //
    // XXX: HACK: Should never be called at anything other then
    // passive level but we cant import KeGetCurrentIRql in the xenvbd
    // and we try to call this again in dump_xenvbd at high irql which
    // is bad so the check is here to keep the compiler/linker happy.
    //
    if (KeGetCurrentIrql() != PASSIVE_LEVEL)
    {
        return;
    }

    Status = ReadRegistryParameterDword(L"FeatureFlags", &features);
    if (NT_SUCCESS(Status))
        _g_PVFeatureFlags = features;
    _g_PVFeatureFlags |= ReadRegistryFlagsOrZero(L"SetFlags");
    _g_PVFeatureFlags &= ~ReadRegistryFlagsOrZero(L"ClearFlags");

    if (*InitSafeBootMode > 0) {
        _g_PVFeatureFlags |= DEBUG_NO_PARAVIRT;
    }   

    //
    // If the "NOPVBoot" value us in our service's key then do *not*
    // go into pv mode.
    //
    RtlInitUnicodeString(&usNOPVBoot, L"NOPVBoot");
    RtlInitUnicodeString(&registryPath,
                         L"\\Registry\\Machine\\SYSTEM\\CurrentControlSet\\Services\\xenevtchn");
    Status = XenReadRegistryKey(registryPath, usNOPVBoot, &pKeyValueInfo);
    if (NT_SUCCESS(Status)) 
    {
        _g_PVFeatureFlags |= DEBUG_NO_PARAVIRT;
        ExFreePool(pKeyValueInfo);
        pKeyValueInfo = NULL;
    }

    //
    // Normally we want to assert paged code here but whatever. see above.
    //

    RtlInitUnicodeString(&usControlPath, L"\\Registry\\Machine\\SYSTEM\\CurrentControlSet\\Control");
    RtlInitUnicodeString(&usStartOptions, L"SystemStartOptions");

    Status = XenReadRegistryKey(usControlPath, usStartOptions, &pKeyValueInfo);
    if (! NT_SUCCESS(Status))
    {
        TraceError (("XENUTIL!XenParseBootParams: Failed with Status = %d\n", Status));
        goto _Cleanup;
    }

    //
    // A hack to null terminate the string
    //

    wzReturnValue = ExAllocatePoolWithTag ( PagedPool, 
                                            pKeyValueInfo->DataLength 
                                             + sizeof(WCHAR),
                                            XUTIL_TAG);
    if (wzReturnValue == NULL) 
    {
        ASSERT (FALSE);
        goto _Cleanup;
    }
    
    RtlCopyMemory (wzReturnValue, pKeyValueInfo->Data,
                   pKeyValueInfo->DataLength);
    wzReturnValue[pKeyValueInfo->DataLength/sizeof(WCHAR)] = L'\0';

    bError = FALSE;
    bDefault = FALSE;
    features = 0;

    //
    // find the /PV=X string where X is a hex string of features
    //
        
    wzPVstring = wcsstr (wzReturnValue, L" PV");

    if (wzPVstring != NULL)
    {

        switch (wzPVstring[3])
        {
        case L'=':
            features = wcstoix(&wzPVstring[4], &bError);
            break;

        case L'^':
                if (wzPVstring[4] == L'=')
                {
                    features = wcstoix(&wzPVstring[5], &bError);
                    features = _g_PVFeatureFlags ^ features;
                }
                else
                {
                    bError = TRUE;
                }
                break;

            case L'~':
                if (wzPVstring[4] == L'=')
                {
                    features = wcstoix(&wzPVstring[5], &bError);
                    features = _g_PVFeatureFlags & ~features;
                }
                else
                {
                    bError = TRUE;
                }
                break;

            case L'|':
                if (wzPVstring[4] == L'=')
                {
                    features = wcstoix(&wzPVstring[5], &bError);
                    features = _g_PVFeatureFlags | features;
                }
                else
                {
                    bError = TRUE;
                }
                break;

            case L'\0':
            case L'\t':
            case L' ':
                bDefault = TRUE;
                break;

            default:
                bError = TRUE;
                break;
        }

        if (!bError)
        {
            if (!bDefault)
            {
                TraceNotice (("XENUTIL!XenParseBootParams: Booting PV features = %I64x\n", features));
                _g_PVFeatureFlags = features;
            }
        }
        else
        {
            TraceWarning(("XENUTIL!XenParseBootParams: error parsing /PV argument.\n"));
        }
    }

    if (_g_PVFeatureFlags & DEBUG_HA_SAFEMODE)
    {
        if (_g_PVFeatureFlags & DEBUG_NO_PARAVIRT)
        {
            //
            // If both the DEBUG_HA_SAFEMODE and DEBUG_NO_PARAVIRT flags
            // are set the clear the DEBUG_NO_PARAVIRT so we will start
            // up xenevtchn.
            //
            _g_PVFeatureFlags &= ~DEBUG_NO_PARAVIRT;
        } 
        else 
        {
            //
            // This means the DEBUG_HA_SAFEMODE flag was set but the
            // DEBUG_NO_PARAVIRT is not set, so we aren't in safe mode.
            // This is not a valid config so just clear the DEBUG_HA_SAFEMODE
            // flag
            //
            _g_PVFeatureFlags &= ~DEBUG_HA_SAFEMODE;
        }
    }

    if (_g_PVFeatureFlags & DEBUG_NO_PARAVIRT)
    {
        TraceNotice (("XENUTIL!XenParseBootParams: Booting into NON-PV mode\n"));
    }
    else if (_g_PVFeatureFlags & DEBUG_HA_SAFEMODE)
    {
        TraceNotice (("XENUTIL!XenParseBootParams: Booting into NON-PV mode (HA PV mode)\n"));
    }
    else
    {
        TraceNotice (("XENUTIL!XenParseBootParams: Booting into PV mode\n"));
    }

_Cleanup:
    if (wzReturnValue != NULL)
        {
            ExFreePool (wzReturnValue);
        }
        
    if (pKeyValueInfo != NULL)
    {
        ExFreePool (pKeyValueInfo);
    }

    if (!ChechXenHypervisor()) {
        /* Turn off all feature flags except FAKE_NETIF, turn on
         * NO_PARAVIRT */
        _g_PVFeatureFlags &= DEBUG_FAKE_NETIF;
        _g_PVFeatureFlags |= DEBUG_NO_PARAVIRT;
    }

    if (_g_PVFeatureFlags != DEBUG_DEFAULT_FEATURE_SET)
    {
        TraceNotice(("Feature flags %x\n", _g_PVFeatureFlags));
    }

    return;
}

ULONG
XmGetXenutilVersion(void)
{
    return XEVTCHN_CURRENT_VERSION;
}

void
XmXenvbdVersionMismatch(ULONG xevtchn_expected_version)
{
    TraceCritical(("Xenvbd expected xevtchn version %d, but got version %d!\n",
                   xevtchn_expected_version, XEVTCHN_CURRENT_VERSION));

    if (done_unplug)
        XmBugCheck("Can't start PV drivers, but have already disconnected emulated devices!\n");

    /* Don't do any more PV stuff.  We're almost certainly going to
       crash anyway, though. */
    _g_PVFeatureFlags = DEBUG_NO_PARAVIRT;
}

void
XmXenutilVersionMismatch(ULONG xevtchn_version)
{
    XM_ASSERT(xevtchn_version != XmGetXenutilVersion());

    /* If we're already unplugged the emulated devices then we're
       pretty much screwed. */
    if (done_unplug)
        XmBugCheck("xevtchn version %d is not compatible with xenutil version %d!\n",
                   xevtchn_version, XmGetXenutilVersion());

    /* If we haven't done the unplug yet, try to break anything else
       which might try to use the PV interfaces. */
    _g_PVFeatureFlags = DEBUG_NO_PARAVIRT;
    TraceError(("PV mode disabled due to version mismatch: xenutil %d is not compatible with xevtchn %d.\n",
                XmGetXenutilVersion(),
                xevtchn_version));
}

BOOLEAN
XenPVEnabled()
{
    if (_g_PVFeatureFlags & DEBUG_NO_PARAVIRT)
        return FALSE;
    else
        return TRUE;
}

/* Device unplugging and connecting to the log-to-dom0 port.  This is
   slightly icky, mostly because there are two different protocols.

   In the old protocol, we have two io ports on the PCI scsi
   controller, one for unplugging and one for logging.  Writing
   anything to the unplug port causes both network and IDE disks to
   get unplugged.  Log-to-dom0 is done by writing bytes to the other
   port.

   In the new protocol, we still have two ports, but they're
   hard-coded to 0x10 and 0x12, which are reserved to the motherboard
   in the ACPI tables.  You can tell the new protocol is available if
   reading 0x10 gives you the signature 0x49d2.  If the new protocol
   is available, there will be a version number which can be obtained
   by reading a byte from 0x12.  There are two versions:

   -- 0, the base protocol.  You unplug devices by writing a USHORT
      bitmask to 0x10 (0x01 -> IDE disks, 0x02 -> rtl8139 NICs, all
      other bits -> reserved).  Logging is done by writing bytes to
      0x12.

   -- 1, which is like 0 but adds a mechanism for telling qemu what
      version of the drivers we are, and for qemu to block versions
      which are known to be bad.  The drivers are expected to write a
      product id to port 0x12 as a short and their build number to
      0x10 as a long, and then check the magic on port 0x10 again.  If
      the drivers are blacklisted, it will have changed to something
      other than the magic.  The only defined product ID is 1, which
      is the Citrix Windows PV drivers.

   The old protocol still works on new toolstacks, but the new one is
   better because it means you can unplug the PCI devices before the
   PCI driver comes up.
*/

static USHORT unplug_protocol; /* 0 -> old, 1 -> new */

/* Old protocol */
static PVOID device_unplug_port_old; /* NULL -> unknown or new
                                      * protocol */

/* New protocol */
#define NEW_UNPLUG_PORT ((PVOID)(ULONG_PTR)0x10)
#define NEW_DOM0LOG_PORT ((PVOID)(ULONG_PTR)0x12)
#define UNPLUG_VERSION_PORT ((PVOID)(ULONG_PTR)0x12)
#define UNPLUG_DRIVER_VERSION_PORT ((PVOID)(ULONG_PTR)0x10)
#define UNPLUG_DRIVER_PRODUCT_PORT ((PVOID)(ULONG_PTR)0x12)
#define DEVICE_UNPLUG_PROTO_MAGIC 0x49d2
#define UNPLUG_DRIVER_PRODUCT_NUMBER 1

#define UNPLUG_ALL_IDE 1
#define UNPLUG_ALL_NICS 2
#define UNPLUG_AUX_IDE 4

/* Shared */
static PVOID dom0_debug_port;

/* Decide which protocol we're using.  This also sets up the port
   numbers for the new protocol.  The old protocol can't do that until
   the PCI resources are available. */
void
InitUnplug(void)
{
    USHORT magic;
    UCHAR version;

    magic = READ_PORT_USHORT(NEW_UNPLUG_PORT);
    if (magic == DEVICE_UNPLUG_PROTO_MAGIC) {
        unplug_protocol = 1;
        version = READ_PORT_UCHAR(UNPLUG_VERSION_PORT);
        if (version >= 1) {
            WRITE_PORT_USHORT(UNPLUG_DRIVER_PRODUCT_PORT,
                              UNPLUG_DRIVER_PRODUCT_NUMBER);
            WRITE_PORT_ULONG(UNPLUG_DRIVER_VERSION_PORT,
                             BRANDING_BUILD_NR);
            magic = READ_PORT_USHORT(NEW_UNPLUG_PORT);
            if (magic != DEVICE_UNPLUG_PROTO_MAGIC) {
                /* Okay, qemu doesn't like this version of the drivers
                   for some reason.  Turn ourselves off. */
                _g_PVFeatureFlags = DEBUG_NO_PARAVIRT;
                return;
            }
        }
        dom0_debug_port = NEW_DOM0LOG_PORT;
        TraceNotice(("PV driver logging available.\n"));
    } else {
        unplug_protocol = 0;
    }
}

void
UnplugIoemu(void)
{
    /* If PV mode is disabled then unplugging the emulated devices is
       a really bad idea. */
    XM_ASSERT(XenPVEnabled());

    if (unplug_protocol == 1) {
        if (XenPVFeatureEnabled(DEBUG_BOOT_EMULATED))
            WRITE_PORT_USHORT(NEW_UNPLUG_PORT,
                              UNPLUG_ALL_NICS|UNPLUG_AUX_IDE);
        else
            WRITE_PORT_USHORT(NEW_UNPLUG_PORT,
                              UNPLUG_ALL_NICS|UNPLUG_ALL_IDE);
        done_unplug = TRUE;
    } else if (unplug_protocol == 0 && device_unplug_port_old) {
        WRITE_PORT_ULONG(device_unplug_port_old, 0);
        done_unplug = TRUE;
    } else if (XenPVFeatureEnabled(DEBUG_FORCE_EARLY_UNPLUG)) {
        /* Icky hack so that you can use scsifilt on Rio, because
           there's some bad interaction between atapi.sys, scsifilt,
           and late unplugging. */
        WRITE_PORT_ULONG((PVOID)0xc104, 0);
        done_unplug = TRUE;
    }
}

void
InitOldUnplugProtocol(PHYSICAL_ADDRESS ioportbase, ULONG nports)
{
    if (unplug_protocol == 0 && nports >= 12) {
        device_unplug_port_old = (PVOID)(ULONG_PTR)(ioportbase.LowPart + 4);
        dom0_debug_port = (PVOID)(ULONG_PTR)(ioportbase.LowPart + 8);
        TraceNotice(("Using old device unplugging protocol.\n"));
    }
}

#define XENTRACE_KD   (1<<0) /* Messages go to kernel debugger */
#define XENTRACE_XEN  (1<<1) /* Messages go to Xen debug port */
#define XENTRACE_LOG  (1<<2) /* Messages go to ring buffer */
#define XENTRACE_DOM0 (1<<3) /* Messages got to dom0 logging port */

/* Indexed by instances of XenTraceLevel */
static int XenTraceDispositions[] = {
    0,
    XENTRACE_LOG|XENTRACE_KD,
    XENTRACE_LOG|XENTRACE_KD|XENTRACE_XEN,
    XENTRACE_LOG|XENTRACE_KD|XENTRACE_XEN|XENTRACE_DOM0,
    XENTRACE_LOG|XENTRACE_KD|XENTRACE_XEN|XENTRACE_DOM0,
    XENTRACE_LOG|XENTRACE_KD|XENTRACE_XEN|XENTRACE_DOM0,
    XENTRACE_LOG|XENTRACE_KD|XENTRACE_XEN|XENTRACE_DOM0,
    XENTRACE_LOG|XENTRACE_XEN
};

/* Map from a XenTraceLevel to a string. */
static const char *const XenTraceTags[] = {
    "[  LOUD  ]",
    "[VERBOSE ]",
    "[  INFO  ]",
    "[ NOTICE ]",
    "[WARNING ]",
    "[ ERROR  ]",
    "[CRITICAL]",
    "[   KD   ]"
};

static VOID
SendMessageToRing(IN PCHAR msg)
{
    ULONG len;
    ULONG ptr;

    len = (LONG)strlen(msg)+1;
    ptr = LogMessageBufferPtr;
    LogMessageBufferPtr += len;
    ptr %= LOG_MSG_BUFFER_SIZE;
    if (ptr + len >= LOG_MSG_BUFFER_SIZE) {
        memcpy(LogMessageBuffer + ptr, msg, LOG_MSG_BUFFER_SIZE - ptr);
        memcpy(LogMessageBuffer, msg + LOG_MSG_BUFFER_SIZE - ptr,
               len + ptr - LOG_MSG_BUFFER_SIZE);
    } else {
        memcpy(LogMessageBuffer + ptr, msg, len);
    }
}

/* We really do want to use a hard-coded port number here, even though
   it upsets prefast. */
#pragma warning(disable : 28138)
static VOID
XenHvmOutString(IN PCHAR msg)
{
    int x;
    for (x = 0; msg[x]; x++)
        WRITE_PORT_UCHAR((PVOID)(ULONG_PTR)0xe9, msg[x]);
}
#pragma warning(default : 28138)

static VOID
SendMessageToDom0(IN PCHAR msg)
{
    int x;
    if (dom0_debug_port) {
        for (x = 0; msg[x]; x++)
            WRITE_PORT_UCHAR(dom0_debug_port, msg[x]);
    }
}

void
___XenTrace(XenTraceLevel level, __in_ecount(module_size) PCSTR module,
            size_t module_size, PCSTR fmt, va_list args)
{
    /* 256 is about the upper end of what can safely be allocated on
       the stack. */
    char buf[256];
    static struct irqsafe_lock lock;
    KIRQL irql;

    if (!XenTraceDispositions[level])
        return;

    memcpy(buf, XenTraceTags[level], 10);
    buf[10] = '[';
    memcpy(buf + 11, module, module_size);
    buf[10+module_size] = ']';
#define Xmvsnprintf _vsnprintf
    Xmvsnprintf(buf+11+module_size, sizeof(buf)-11-module_size, fmt, args);
    buf[sizeof(buf)-1] = 0;

    if ( (XenTraceDispositions[level] & XENTRACE_KD) &&
         KeGetCurrentIrql() <= DISPATCH_LEVEL ) {
        KdPrint(("%s", buf));
    }
    irql = acquire_irqsafe_lock(&lock);
    if ( XenTraceDispositions[level] & XENTRACE_XEN ) {
        XenHvmOutString(buf);
    }
    if ( XenTraceDispositions[level] & XENTRACE_LOG ) {
        SendMessageToRing(buf);
    }
    if ( (XenTraceDispositions[level] & XENTRACE_DOM0) &&
         dom0_debug_port > 0 ) {
        SendMessageToDom0(buf);
    }
    release_irqsafe_lock(&lock, irql);
}

void
XenTraceSetLevels(const int *levels)
{
    int x;

    for (x = 0; x < 7; x++) {
        TraceNotice (("Set trace level %s to %x\n", XenTraceTags[x],
                      levels[x]));
        XenTraceDispositions[x] = levels[x];
    }
}

ULONG
HvmGetLogRingSize(void)
{
    if (LogMessageBufferPtr >= LOG_MSG_BUFFER_SIZE)
        return LOG_MSG_BUFFER_SIZE;
    else
        return LogMessageBufferPtr;
}

NTSTATUS
HvmGetLogRing(void *buffer, ULONG size)
{
    ULONG ptr;
    ptr = LogMessageBufferPtr;
    if (ptr > LOG_MSG_BUFFER_SIZE) {
        if (size != LOG_MSG_BUFFER_SIZE) {
            return STATUS_INVALID_USER_BUFFER;
        } else {
            ptr %= LOG_MSG_BUFFER_SIZE;
            memcpy(buffer, LogMessageBuffer + ptr, LOG_MSG_BUFFER_SIZE - ptr);
            memcpy((void *)((ULONG_PTR)buffer + LOG_MSG_BUFFER_SIZE - ptr),
                   LogMessageBuffer, ptr);
            return STATUS_SUCCESS;
        }
    } else {
        if (size != ptr) {
            return STATUS_INVALID_USER_BUFFER;
        } else {
            memcpy(buffer, LogMessageBuffer, ptr);
            return STATUS_SUCCESS;
        }
    }
}

/* Extract the last @max_size bytes from the log and copy them to
   @outbuf.  Returns the number of bytes copied. */
ULONG
XmExtractTailOfLog(char *outbuf, ULONG max_size)
{
    ULONG end, start;

    if (max_size > LOG_MSG_BUFFER_SIZE)
        max_size = LOG_MSG_BUFFER_SIZE;
    end = LogMessageBufferPtr;
    if (end > max_size) {
        start = end - max_size;
    } else {
        start = 0;
    }
    if (start / LOG_MSG_BUFFER_SIZE == end / LOG_MSG_BUFFER_SIZE) {
        memcpy(outbuf, LogMessageBuffer + (start % LOG_MSG_BUFFER_SIZE),
               end - start);
    } else {
        memcpy(outbuf, LogMessageBuffer + (start % LOG_MSG_BUFFER_SIZE),
               LOG_MSG_BUFFER_SIZE - (start % LOG_MSG_BUFFER_SIZE));
        memcpy(outbuf + LOG_MSG_BUFFER_SIZE - (start % LOG_MSG_BUFFER_SIZE),
               LogMessageBuffer,
               end % LOG_MSG_BUFFER_SIZE);
    }
    return end - start;
}

/* Dump the entire debug log ring to dom0 if possible. */
void
XenTraceFlush(void)
{
    unsigned x;
    ULONG ptr = LogMessageBufferPtr;
    if (dom0_debug_port) {
        SendMessageToDom0("Debug ring:\n");
        if (ptr > LOG_MSG_BUFFER_SIZE) {
            for (x = ptr % LOG_MSG_BUFFER_SIZE; x < LOG_MSG_BUFFER_SIZE; x++){
                if (LogMessageBuffer[x])
                    WRITE_PORT_UCHAR(dom0_debug_port, LogMessageBuffer[x]);
            }
        }
        for (x = 0; x < ptr % LOG_MSG_BUFFER_SIZE; x++) {
            if (LogMessageBuffer[x])
                WRITE_PORT_UCHAR(dom0_debug_port, LogMessageBuffer[x]);
        }
        SendMessageToDom0("\nEnd of debug ring.\n");
    }
}

void _XmBugCheck(const char *file, unsigned line, ...)
{
    va_list args;
    const char *fmt;

    TraceCritical(("%s:%d: Bug check.\n", file, line));
    va_start(args, line);
    fmt = va_arg(args, const char *);
    ___XenTrace(XenTraceLevelCritical,
                XENTARGET,
                sizeof(XENTARGET),
                fmt,
                args);
    va_end(args);
    KeBugCheckEx(0xf7dead, (ULONG_PTR)file, line, 0, 0);
}

void _XmAssertFail(const char *file, unsigned line, const char *expr)
{
    _XmBugCheck(file, line, "Assertion %s failed at %s:%d.\n", expr, file, line);
}

void _XmBug(const char *file, unsigned line, const char *expr, ULONG_PTR val)
{
    TraceCritical(("BUG: %s is 0x%p, should be 0 at %s:%d\n",
                   expr, val, file, line)); 
    KeBugCheckEx(0xf7deae, (ULONG_PTR)file, line, val, 0);
}


#if !defined(KXEN_GUEST)
NTSTATUS
DllInitialize(PUNICODE_STRING RegistryPath)
{
    UNREFERENCED_PARAMETER(RegistryPath);

    XenParseBootParams();

    if (XenPVFeatureEnabled (DEBUG_NO_PARAVIRT))
        return STATUS_SUCCESS;

    if (XenPVFeatureEnabled (DEBUG_VERY_LOUD))
    {
        int dispositions[7] = {-1,-1,-1,-1,-1,-1,-1};
        XenTraceSetLevels(dispositions);
    }

    if (XenPVFeatureEnabled (DEBUG_VERY_QUIET))
    {
        int dispositions[7] = {0};
        TraceNotice(("Disable all logging...\n"));
        XenTraceSetLevels(dispositions);
    }

    InitUnplug();

    if (XenPVEnabled() &&
        (!XenPVFeatureEnabled(DEBUG_HA_SAFEMODE)))
    {
        UnplugIoemu();
    }

   return STATUS_SUCCESS;
}

NTSTATUS
DriverEntry(PDRIVER_OBJECT dev, PUNICODE_STRING reg_path)
{
    UNREFERENCED_PARAMETER(dev);
    UNREFERENCED_PARAMETER(reg_path);
    return STATUS_SUCCESS;
}
#endif
