/*++

Copyright (c) Citrix Corporation.  All rights reserved.

Module Name:

    tenkei2.c

Abstract:

    This is the main program (console/test) for the tenkei2 project.  This 
    program is used to start and stop VMs and will load/unload the VMM if
    required.

Environment:

    User mode Win32 console application

--*/

#include <windows.h>
#include <winioctl.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <strsafe.h>

#include "tenkei2.h"
#include <tk2_ioctl.h>

#define HV_SIZE_MIN 512
#define HV_SIZE_MAX (512*1024*1024)

//
// Globals
//

HANDLE hDevice;
BOOLEAN ExitFlag = FALSE;

BOOLEAN
LoadBlob(
    PMEMDESC Descriptor
    )
{
    HANDLE blob;
    PVOID buffer;
    LARGE_INTEGER blobSize;
    ULONG bytesRead;

    RtlZeroMemory(Descriptor, sizeof(*Descriptor));

    blob = CreateFile("hxen.elf",
                      GENERIC_READ,
                      FILE_SHARE_READ,
                      NULL,
                      OPEN_EXISTING,
                      0,
                      NULL);
    if (blob == INVALID_HANDLE_VALUE) {
        printf("Open of blob file failed, err %d\n", GetLastError());
        return FALSE;
    }

    if (!GetFileSizeEx(blob, &blobSize)) {
        printf("Failed to get size of blob file, err %d\n", GetLastError());
        return FALSE;
    }

    // Note: We expect HV_SIZE_MAX to be less than 4GB, otherwise we
    // can't interchangably use blobSize.u.LowPart below.

    if ((blobSize.QuadPart < HV_SIZE_MIN) ||
        (blobSize.QuadPart > HV_SIZE_MAX)) {
        printf("Blob file size out of expected range.\n");
        return FALSE;
    }

    buffer = malloc((SIZE_T)blobSize.u.LowPart);
    if (!buffer)
    {
        printf("Failed to allocate %d bytes for blob.\n",
                blobSize.u.LowPart);
        return FALSE;
    }

    if (!ReadFile(blob,
                  buffer,
                  blobSize.u.LowPart,
                  &bytesRead,
                  NULL)) {
        printf("Blob read failed, err %d\n", GetLastError());
        return FALSE;
    }

    if (bytesRead != blobSize.u.LowPart) {
        printf("Size mismatch, read %d bytes, expected %d\n",
               bytesRead,
               blobSize.u.LowPart);
        return FALSE;
    }
    Descriptor->Base = buffer;
    Descriptor->Length = bytesRead;
    CloseHandle(blob);

    printf("Loaded blob, %d bytes.\n", bytesRead);

    return TRUE;
}

BOOLEAN
InstallBlob(
    HANDLE   hDevice,
    PMEMDESC Blob
    )
{
    BOOLEAN result;
    ULONG bytesCopied;

    result = Ioctl(hDevice,
                   TENKEI2CTL_loadhv,
                   Blob->Base,
                   Blob->Length,
                   NULL,
                   0,
                   &bytesCopied);
    free(Blob->Base);
    Blob->Base = NULL;
    Blob->Length = 0;
    printf("Copied %d bytes to kernel space.\n", bytesCopied);
    return TRUE;
}

//
// Main function
//

VOID __cdecl
main(
    __in ULONG argc,
    __in_ecount(argc) PCHAR argv[]
    )
{
    BOOLEAN result;
    ULONG driverVersion;
    ULONG returnedLength;
    PUCHAR guestMemory;

    printf("Hello\n");
    if ((hDevice = OpenDriver()) == INVALID_HANDLE_VALUE) {
        ExitProcess(1);
        return;
    }

    result = Ioctl(hDevice,
                   TENKEI2CTL_get_ver,
                   NULL,
                   0,
                   &driverVersion,
                   sizeof(driverVersion),
                   &returnedLength);

    if (result) {
        printf("IOCTL GET VERSION returned 0x%08x\n", driverVersion);
    } else {
        printf("IOCTL GET VERSION FAILED, ERROR %d\n", GetLastError());
    }

#define GUEST_SIZE (512*1024*1024)
    guestMemory = malloc(GUEST_SIZE+4096);
    guestMemory = (PUCHAR)(((ULONG_PTR)guestMemory + 4095) & ~4095);

    result = Ioctl(hDevice,
                   TENKEI2CTL_lock_guest_mem,
                   guestMemory,
                   GUEST_SIZE,
                   NULL,
                   0,
                   &returnedLength);

    printf("IOCTL lock guest mem %s\n", result ? "succeeded" : "FAILED!!!");

#if 0
    result = Ioctl(hDevice,
                   TENKEI2CTL_vcpu_run,
                   NULL, 0, NULL, 0, &returnedLength);

    printf("IOCTL run vcpu %s\n", result ? "succeeded" : "FAILED!!!");
#endif

    getchar();

    CloseDriver(hDevice);

    //
    // Unload the driver.  Ignore any errors.
    //

    ManageDriver(DRIVER_NAME,
                 NULL,
                 DRIVER_FUNC_REMOVE
                 );

    ExitProcess(0);
}
