/*
 * Copyright (c) 2006, XenSource Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#ifndef XEN_VBD_H
#define XEN_VBD_H

#include "xen_common.h"
#include "xen_driver_type.h"
#include "xen_vbd_decl.h"
#include "xen_vbd_mode.h"
#include "xen_vdi_decl.h"
#include "xen_vm_decl.h"


/*
 * The VBD class. 
 *  
 * A virtual block device.
 */


/**
 * Free the given xen_vbd.  The given handle must have been allocated
 * by this library.
 */
extern void
xen_vbd_free(xen_vbd vbd);


typedef struct xen_vbd_set
{
    size_t size;
    xen_vbd *contents[];
} xen_vbd_set;

/**
 * Allocate a xen_vbd_set of the given size.
 */
extern xen_vbd_set *
xen_vbd_set_alloc(size_t size);

/**
 * Free the given xen_vbd_set.  The given set must have been allocated
 * by this library.
 */
extern void
xen_vbd_set_free(xen_vbd_set *set);


typedef struct xen_vbd_record
{
    xen_vbd handle;
    char *uuid;
    struct xen_vm_record_opt *vm;
    struct xen_vdi_record_opt *vdi;
    char *device;
    enum xen_vbd_mode mode;
    enum xen_driver_type driver;
    double io_bandwidth_incoming_kbs;
    double io_bandwidth_outgoing_kbs;
} xen_vbd_record;

/**
 * Allocate a xen_vbd_record.
 */
extern xen_vbd_record *
xen_vbd_record_alloc(void);

/**
 * Free the given xen_vbd_record, and all referenced values.  The given
 * record must have been allocated by this library.
 */
extern void
xen_vbd_record_free(xen_vbd_record *record);


typedef struct xen_vbd_record_opt
{
    bool is_record;
    union
    {
        xen_vbd handle;
        xen_vbd_record *record;
    } u;
} xen_vbd_record_opt;

/**
 * Allocate a xen_vbd_record_opt.
 */
extern xen_vbd_record_opt *
xen_vbd_record_opt_alloc(void);

/**
 * Free the given xen_vbd_record_opt, and all referenced values.  The
 * given record_opt must have been allocated by this library.
 */
extern void
xen_vbd_record_opt_free(xen_vbd_record_opt *record_opt);


typedef struct xen_vbd_record_set
{
    size_t size;
    xen_vbd_record *contents[];
} xen_vbd_record_set;

/**
 * Allocate a xen_vbd_record_set of the given size.
 */
extern xen_vbd_record_set *
xen_vbd_record_set_alloc(size_t size);

/**
 * Free the given xen_vbd_record_set, and all referenced values.  The
 * given set must have been allocated by this library.
 */
extern void
xen_vbd_record_set_free(xen_vbd_record_set *set);



typedef struct xen_vbd_record_opt_set
{
    size_t size;
    xen_vbd_record_opt *contents[];
} xen_vbd_record_opt_set;

/**
 * Allocate a xen_vbd_record_opt_set of the given size.
 */
extern xen_vbd_record_opt_set *
xen_vbd_record_opt_set_alloc(size_t size);

/**
 * Free the given xen_vbd_record_opt_set, and all referenced values. 
 * The given set must have been allocated by this library.
 */
extern void
xen_vbd_record_opt_set_free(xen_vbd_record_opt_set *set);


/**
 * Get the current state of the given VBD.  !!!
 */
extern xen_vbd_record *
xen_vbd_get_record(xen_session *session, xen_vbd vbd);


/**
 * Get a reference to the object with the specified UUID.  !!!
 */
extern xen_vbd 
xen_vbd_get_by_uuid(xen_session *session, char *uuid);


/**
 * Create a new VBD instance, and return its handle.
 */
extern xen_vbd 
xen_vbd_create(xen_session *session, xen_vbd_record *record);


/**
 * Get the uuid field of the given VBD.
 */
extern char *
xen_vbd_get_uuid(xen_session *session, xen_vbd vbd);


/**
 * Get the VM field of the given VBD.
 */
extern xen_vm 
xen_vbd_get_vm(xen_session *session, xen_vbd vbd);


/**
 * Get the VDI field of the given VBD.
 */
extern xen_vdi 
xen_vbd_get_vdi(xen_session *session, xen_vbd vbd);


/**
 * Get the device field of the given VBD.
 */
extern char *
xen_vbd_get_device(xen_session *session, xen_vbd vbd);


/**
 * Get the mode field of the given VBD.
 */
extern enum xen_vbd_mode 
xen_vbd_get_mode(xen_session *session, xen_vbd vbd);


/**
 * Get the driver field of the given VBD.
 */
extern enum xen_driver_type 
xen_vbd_get_driver(xen_session *session, xen_vbd vbd);


/**
 * Get the IO_bandwidth/incoming_kbs field of the given VBD.
 */
extern double 
xen_vbd_get_io_bandwidth_incoming_kbs(xen_session *session, xen_vbd vbd);


/**
 * Get the IO_bandwidth/outgoing_kbs field of the given VBD.
 */
extern double 
xen_vbd_get_io_bandwidth_outgoing_kbs(xen_session *session, xen_vbd vbd);


/**
 * Set the VM field of the given VBD.
 */
extern void
xen_vbd_set_vm(xen_session *session, xen_vbd xen_vbd, xen_vm vm);


/**
 * Set the VDI field of the given VBD.
 */
extern void
xen_vbd_set_vdi(xen_session *session, xen_vbd xen_vbd, xen_vdi vdi);


/**
 * Set the device field of the given VBD.
 */
extern void
xen_vbd_set_device(xen_session *session, xen_vbd xen_vbd, char *device);


/**
 * Set the mode field of the given VBD.
 */
extern void
xen_vbd_set_mode(xen_session *session, xen_vbd xen_vbd, enum xen_vbd_mode mode);


/**
 * Set the driver field of the given VBD.
 */
extern void
xen_vbd_set_driver(xen_session *session, xen_vbd xen_vbd, enum xen_driver_type driver);


#endif
