/* Copyright (C) 2007-2008 Google, Inc.
**
** This software is licensed under the terms of the GNU General Public
** License version 2, as published by the Free Software Foundation, and
** may be copied, distributed, and modified under those terms.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
*/
#include "vl.h"
#include "arm_pic.h"
#include "platform_device.h"

int android_audio_enabled;
char* audio_input_source = NULL;

void platform_memlog_init(uint32_t base);

static struct platform_device event0_device = {
    .name = "goldfish_events",
    .id = 0,
    .size = 0x1000,
    .irq_count = 1
};

static struct platform_device nand_device = {
    .name = "goldfish_nand",
    .id = 0,
    .size = 0x1000
};

static struct platform_device trace_device = {
    .name = "qemu_trace",
    .id = -1,
    .size = 0x1000
};

/* Board init.  */

#define TEST_SWITCH 1
#if TEST_SWITCH
uint32_t switch_test_write(void *opaque, uint32_t state)
{
    platform_switch_set_state(opaque, state);
    return state;
}
#endif

static void android_arm_init(int ram_size, int vga_ram_size,
    int boot_device, DisplayState *ds, const char **fd_filename, int snapshot,
    const char *kernel_filename, const char *kernel_cmdline,
    const char *initrd_filename,
	const int guest0_load_addr, const char *guest0_filename,
	const int guest1_load_addr, const char *guest1_filename)
{
    CPUState *env;
    void *cpu_pic;
    void *platform_pic;
    int i;

    env = cpu_init();
    cpu_arm_set_model(env, ARM_CPUID_ARM926);

    register_savevm( "cpu", 0, ARM_CPU_SAVE_VERSION, cpu_save, cpu_load, env );

    cpu_register_physical_memory(0, ram_size, IO_MEM_RAM);

    cpu_pic = arm_pic_init_cpu(env);
    platform_pic = platform_interrupt_init(0xff000000, cpu_pic, ARM_PIC_CPU_IRQ, ARM_PIC_CPU_FIQ);
    platform_device_init(platform_pic, 0xff010000, 0x7f0000, 10, 22);

    platform_device_bus_init(0xff001000, 1);

    platform_timer_and_rtc_init(0xff003000, 3);

    platform_tty_add(serial_hds[0], 0, 0xff002000, 4);
    for(i = 1; i < MAX_SERIAL_PORTS; i++) {
        //printf("android_arm_init serial %d %x\n", i, serial_hds[i]);
        if(serial_hds[i]) {
            platform_tty_add(serial_hds[i], i, 0, 0);
        }
    }

    for(i = 0; i < MAX_NICS; i++) {
        if (nd_table[i].vlan) {
            if (nd_table[i].model == NULL
                || strcmp(nd_table[i].model, "smc91c111") == 0) {
                struct platform_device *smc_device;
                smc_device = qemu_mallocz(sizeof(*smc_device));
                smc_device->name = "smc91x";
                smc_device->id = i;
                smc_device->size = 0x1000;
                smc_device->irq_count = 1;
                platform_add_device_no_io(smc_device);
                smc91c111_init(&nd_table[i], smc_device->base, platform_pic, smc_device->irq);
            } else {
                fprintf(stderr, "qemu: Unsupported NIC: %s\n", nd_table[0].model);
                exit (1);
            }
        }
    }

    platform_fb_init(ds, 0);
#ifdef HAS_AUDIO
    if (android_audio_enabled) {
        AUD_init();
        platform_audio_init(0xff004000, 0, audio_input_source);
    }
#endif
    if (bs_table[0])
        platform_mmc_init(0xff005000, 0, bs_table[0]);

    platform_memlog_init(0xff006000);

    platform_battery_init();

    platform_add_device_no_io(&event0_device);
    events_dev_init(event0_device.base, platform_pic, event0_device.irq);

#ifdef CONFIG_NAND
    platform_add_device_no_io(&nand_device);
    nand_dev_init(nand_device.base);
#endif
#ifdef CONFIG_TRACE
    extern const char *trace_filename;
    if(trace_filename != NULL) {
        platform_add_device_no_io(&trace_device);
        trace_dev_init(trace_device.base);
    }
#endif

#if TEST_SWITCH
    {
        void *sw;
        sw = platform_switch_add("test", NULL, NULL, 0);
        platform_switch_set_state(sw, 1);
        platform_switch_add("test2", switch_test_write, sw, 1);
    }
#endif

    arm_load_kernel(ram_size, kernel_filename, kernel_cmdline,
                    initrd_filename,
					guest0_load_addr, guest0_filename,
					guest1_load_addr, guest1_filename,
				 	1441);
}

QEMUMachine android_arm_machine = {
    "android_arm",
    "ARM Android Emulator",
    android_arm_init,
};
