/* Copyright (C) 2007-2008 Google, Inc.
**
** This software is licensed under the terms of the GNU General Public
** License version 2, as published by the Free Software Foundation, and
** may be copied, distributed, and modified under those terms.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
*/
#include "vl.h"
#include "platform_device.h"

enum {
    TTY_PUT_CHAR       = 0x00,
    TTY_BYTES_READY    = 0x04,
    TTY_CMD            = 0x08,

    TTY_DATA_PTR       = 0x10,
    TTY_DATA_LEN       = 0x14,

    TTY_CMD_INT_DISABLE    = 0,
    TTY_CMD_INT_ENABLE     = 1,
    TTY_CMD_WRITE_BUFFER   = 2,
    TTY_CMD_READ_BUFFER    = 3,
};

struct tty_state {
    struct platform_device dev;
    CharDriverState *cs;
    uint32_t ptr;
    uint32_t ptr_len;
    uint32_t ready;
    uint8_t data[128];
    uint32_t data_count;
};

#define  PLATFORM_TTY_SAVE_VERSION  1

static void  platform_tty_save(QEMUFile*  f, void*  opaque)
{
    struct tty_state*  s = opaque;

    qemu_put_be32( f, s->ptr );
    qemu_put_be32( f, s->ptr_len );
    qemu_put_byte( f, s->ready );
    qemu_put_byte( f, s->data_count );
    qemu_put_buffer( f, s->data, s->data_count );
}

static int  platform_tty_load(QEMUFile*  f, void*  opaque, int  version_id)
{
    struct tty_state*  s = opaque;

    if (version_id != PLATFORM_TTY_SAVE_VERSION)
        return -1;

    s->ptr        = qemu_get_be32(f);
    s->ptr_len    = qemu_get_be32(f);
    s->ready      = qemu_get_byte(f);
    s->data_count = qemu_get_byte(f);
    qemu_get_buffer(f, s->data, s->data_count);

    return 0;
}

static uint32_t platform_tty_read(void *opaque, target_phys_addr_t offset)
{
    struct tty_state *s = (struct tty_state *)opaque;
    offset -= s->dev.base;

    //printf("platform_tty_read %x %x\n", offset, size);

    switch (offset) {
        case TTY_BYTES_READY:
            return s->data_count;
    default:
        cpu_abort (cpu_single_env, "platform_tty_read: Bad offset %x\n", offset);
        return 0;
    }
}

static void platform_tty_write(void *opaque, target_phys_addr_t offset, uint32_t value)
{
    struct tty_state *s = (struct tty_state *)opaque;
    offset -= s->dev.base;

    //printf("platform_tty_read %x %x %x\n", offset, value, size);
    //fprintf(stderr, "[*]");

    switch(offset) {
        case TTY_PUT_CHAR: {
            uint8_t ch = value;
            if(s->cs)
                qemu_chr_write(s->cs, &ch, 1);
			//fprintf(stderr, "%c", ch);
        } break;

        case TTY_CMD:
            switch(value) {
                case TTY_CMD_INT_DISABLE:
                    if(s->ready) {
                        if(s->data_count > 0)
                            platform_device_set_irq(&s->dev, 0, 0);
                        s->ready = 0;
                    }
                    break;

                case TTY_CMD_INT_ENABLE:
                    if(!s->ready) {
                        if(s->data_count > 0)
                            platform_device_set_irq(&s->dev, 0, 1);
                        s->ready = 1;
                    }
                    break;

                case TTY_CMD_WRITE_BUFFER:
                    if(s->cs) {
                        int len;
                        target_ulong buf;

                        buf = s->ptr;
                        len = s->ptr_len;

                        while(len) {
                            int page_remain = TARGET_PAGE_SIZE - (buf & ~TARGET_PAGE_MASK);
                            int to_write = len;
                            uint8_t *phys = (uint8_t *)v2p(buf, 0);
                            if(to_write > page_remain)
                                to_write = page_remain;
                            qemu_chr_write(s->cs, phys, to_write);
                            buf += to_write;
                            len -= to_write;
                        }
                        //printf("platform_tty_write: got %d bytes from %x\n", s->ptr_len, s->ptr);
                    }
                    break;

                case TTY_CMD_READ_BUFFER:
                    if(s->ptr_len > s->data_count)
                        cpu_abort (cpu_single_env, "platform_tty_write: reading more data than available %d %d\n", s->ptr_len, s->data_count);
                    pmemcpy(s->ptr, s->data, s->ptr_len);
                    //printf("platform_tty_write: read %d bytes to %x\n", s->ptr_len, s->ptr);
                    if(s->data_count > s->ptr_len)
                        memmove(s->data, s->data + s->ptr_len, s->data_count - s->ptr_len);
                    s->data_count -= s->ptr_len;
                    if(s->data_count == 0 && s->ready)
                        platform_device_set_irq(&s->dev, 0, 0);
                    break;

                default:
                    cpu_abort (cpu_single_env, "platform_tty_write: Bad command %x\n", value);
            };
            break;

        case TTY_DATA_PTR:
            s->ptr = value;
            break;

        case TTY_DATA_LEN:
            s->ptr_len = value;
            break;

        default:
            cpu_abort (cpu_single_env, "platform_tty_write: Bad offset %x\n", offset);
    }
}

static int tty_can_recieve(void *opaque)
{
    struct tty_state *s = opaque;

    return (sizeof(s->data) - s->data_count);
}

static void tty_recieve(void *opaque, const uint8_t *buf, int size)
{
    struct tty_state *s = opaque;

    memcpy(s->data + s->data_count, buf, size);
    s->data_count += size;
    if(s->data_count > 0 && s->ready)
        platform_device_set_irq(&s->dev, 0, 1);
}

static CPUReadMemoryFunc *platform_tty_readfn[] = {
    platform_tty_read,
    platform_tty_read,
    platform_tty_read
};

static CPUWriteMemoryFunc *platform_tty_writefn[] = {
    platform_tty_write,
    platform_tty_write,
    platform_tty_write
};

int platform_tty_add(CharDriverState *cs, int id, uint32_t base, int irq)
{
    int ret;
    struct tty_state *s;
    static int  instance_id = 0;

    s = qemu_mallocz(sizeof(*s));
    s->dev.name = "goldfish_tty";
    s->dev.id = id;
    s->dev.base = base;
    s->dev.size = 0x1000;
    s->dev.irq = irq;
    s->dev.irq_count = 1;
    s->cs = cs;

    if(cs) {
        qemu_chr_add_read_handler(cs, tty_can_recieve, tty_recieve, s);
    }

    ret = platform_device_add(&s->dev, platform_tty_readfn, platform_tty_writefn, s);
    if(ret) {
        qemu_free(s);
    } else {
        register_savevm( "platform_tty", instance_id++, PLATFORM_TTY_SAVE_VERSION,
                         platform_tty_save, platform_tty_load, s);
    }
    return ret;
}

