/* Copyright (C) 2006-2008 Google, Inc.
**
** This software is licensed under the terms of the GNU General Public
** License version 2, as published by the Free Software Foundation, and
** may be copied, distributed, and modified under those terms.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
*/

#include <signal.h>
#include <unistd.h>
#include <string.h>
#include <sys/time.h>
#ifndef _WIN32
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#else
#include <winsock2.h>
#include <process.h>
#endif
#include "libslirp.h"
#include "sockets.h"

#include "android.h"
#include "vl.h"

#include <SDL.h>
#include <SDL_syswm.h>

#include "math.h"

#include "android.h"
#include "android_charmap.h"
#include "modem_driver.h"
#include "shaper.h"

#include "proxy_http.h"

#include "android_utils.h"
#include "android_resource.h"
#include "android_config.h"

#include "skin_image.h"
#include "skin_trackball.h"
#include "skin_keyboard.h"
#include "skin_file.h"
#include "skin_window.h"
#include "skin_keyset.h"

#include "android_timezone.h"
#include "android_gps.h"
#include "android_qemud.h"
#include "android_kmsg.h"

#include "framebuffer.h"
AndroidRotation  android_framebuffer_rotation;

#define  STRINGIFY(x)   _STRINGIFY(x)
#define  _STRINGIFY(x)  #x

#define  VERSION_STRING  STRINGIFY(ANDROID_VERSION_MAJOR)"."STRINGIFY(ANDROID_VERSION_MINOR)

#define  KEYSET_FILE    "default.keyset"
SkinKeyset*      android_keyset;

#define  D(...)  do {  if (VERBOSE_CHECK(init)) dprint(__VA_ARGS__); } while (0)

extern int  control_console_start( int  port );  /* in control.c */

extern int qemu_milli_needed;

/* the default device DPI if none is specified by the skin
 */
#define  DEFAULT_DEVICE_DPI  165

static const AKeyCharmap*  android_charmap;

int    android_base_port;

#if 0
static int  arg_flashkeys;      /* forward */
#endif

static int  arg_raw_keys;       /* forward */

static void  handle_key_command( void*  opaque, SkinKeyCommand  command, int  param );

static char*  arg_onion_alpha;  /* forward */
static char*  arg_onion_rotate; /* forward */
static char*  arg_onion;        /* forward */

static char*  arg_dpi_device;   /* forward */
static char*  arg_dpi_monitor;     /* forward */

static char*  arg_window_scale; /* forward */

#ifdef CONFIG_TRACE
extern void  start_tracing(void);
extern void  stop_tracing(void);
#endif

unsigned long   android_verbose;

int   qemu_cpu_delay = 0;
int   qemu_cpu_delay_count;

/***********************************************************************/
/***********************************************************************/
/*****                                                             *****/
/*****            U T I L I T Y   R O U T I N E S                  *****/
/*****                                                             *****/
/***********************************************************************/
/***********************************************************************/

/*** APPLICATION DIRECTORY
 *** Where are we ?
 ***/

const char*  get_app_dir(void)
{
    char  buffer[1024];
    char* p   = buffer;
    char* end = p + sizeof(buffer);
    p = bufprint_app_dir(p, end);
    if (p >= end)
        return NULL;

    return strdup(buffer);
}

/***  CONFIGURATION
 ***/

#ifdef _WIN32
#  define  mkdir(path,mode)  _mkdir(path)
#  define  PATH_SEP   "\\"
#else
#  define  PATH_SEP   "/"
#endif

static AConfig*  emulator_config;
static char      emulator_configpath[256];

void
emulator_config_init( void )
{
    char*  end = emulator_configpath + sizeof(emulator_configpath);
    char*  p;
    void*  config;

    emulator_config = aconfig_node("","");

    p = bufprint_config_file(emulator_configpath, end, "emulator.cfg");
    if (p >= end) {
        dprint( "### Warning: emulator configuration path too long" );
        emulator_configpath[0] = 0;
        return;
    }

    config = load_text_file( emulator_configpath );
    if (config == NULL)
        dprint( "warning: cannot load emulator configuration at '%s'\n",
                emulator_configpath );
    else
        aconfig_load( emulator_config, config );
}

/* only call this function on normal exits, so that ^C doesn't save the configuration */
void
emulator_config_done( void )
{
    int     save = 0;  /* only save config if we see changes */
    AConfig*  guid_node;
    char    guid_value[32];
    AConfig*  window_node;
    int     prev_x, prev_y, win_x, win_y;

    if (!emulator_configpath[0]) {
        D("no config path ?");
        return;
    }

    /* compare window positions */
    {
        SDL_WM_GetPos( &win_x, &win_y );
        prev_x = win_x - 1;
        prev_y = win_y - 1;

        window_node = aconfig_find( emulator_config, "window" );
        if (window_node == NULL) {
            aconfig_set( emulator_config, "window", "" );
            window_node = aconfig_find( emulator_config, "window" );
            save = 1;
        }

        prev_x = (int)aconfig_unsigned( window_node, "x", 0 );
        prev_y = (int)aconfig_unsigned( window_node, "y", 0 );

        save = (prev_x != win_x) || (prev_y != win_y);

       /* beware: if the new window position is definitely off-screen,
        * we don't want to save it in the configuration file. this can
        * happen for example on Linux where certain window managers
        * will mimize a window by moving it to coordinates like
        * (-5000,-5000)
        */
        if ( !SDL_WM_IsFullyVisible(0) ) {
            D( "not saving new emulator window position since it is not fully visible" );
            save = 0;
        }
        else if (save)
            D( "emulator window position changed and will be saved as (%d, %d)", win_x, win_y );
    }

    /* if there is no guid node, create one with the current time in
     * milliseconds. we use a name that is sufficiently descriptive to
     * indicate that the value doesn't correspond to any system or
     * user-specific data
     */
#define  GUID_NAME  "unique-id"

    guid_node = aconfig_find( emulator_config, GUID_NAME );
    if (!guid_node) {
        struct timeval  tm;
        gettimeofday( &tm, NULL );
        sprintf( guid_value, "%lld", (long long)tm.tv_sec*1000 + tm.tv_usec/1000 );
        save = 1;
        aconfig_set( emulator_config, GUID_NAME, guid_value );
    }

    if (save) {
        char  xbuf[16], ybuf[16];

        sprintf( xbuf, "%d", win_x );
        sprintf( ybuf, "%d", win_y );

        aconfig_set( window_node, "x", xbuf );
        aconfig_set( window_node, "y", ybuf );

        /* do we need to create the $HOME/.android directory ? */
        if ( access(emulator_configpath, F_OK) != 0 ) {
            char*  dir = path_parent(emulator_configpath, 1);
            if (dir == NULL) {
                D("invalid user-specific config directory: '%s'", emulator_configpath);
                return;
            }
            if ( access(dir, F_OK) < 0 && errno == ENOENT ) {
                int  ret = mkdir(dir, 0755);
                if (ret != 0) {
                    D("cannot create directory '%s', configuration lost", dir);
                    free(dir);
                    return;
                }
            }
            free(dir);
        }
        if ( aconfig_save_file( emulator_config, emulator_configpath ) < 0 ) {
            D( "cannot save configuration to %s", emulator_configpath);
        } else
            D( "configuration saved to %s", emulator_configpath );
    }
}

void *loadpng(const char *fn, unsigned *_width, unsigned *_height);
void *readpng(const unsigned char*  base, size_t  size, unsigned *_width, unsigned *_height);

#ifdef CONFIG_DARWIN
#  define  ANDROID_ICON_PNG  "android_icon_256.png"
#else
#  define  ANDROID_ICON_PNG  "android_icon_16.png"
#endif

static void
sdl_set_window_icon( void )
{
    static int  window_icon_set;

    if (!window_icon_set)
    {
#ifdef _WIN32
        HANDLE         handle = GetModuleHandle( NULL );
        HICON          icon   = LoadIcon( handle, MAKEINTRESOURCE(1) );
        SDL_SysWMinfo  wminfo;

        SDL_GetWMInfo(&wminfo);

        SetClassLong( wminfo.window, GCL_HICON, (LONG)icon );
#else  /* !_WIN32 */
        unsigned              icon_w, icon_h;
        size_t                icon_bytes;
        const unsigned char*  icon_data;
        void*                 icon_pixels;

        window_icon_set = 1;

        icon_data = android_icon_find( ANDROID_ICON_PNG, &icon_bytes );
        if ( !icon_data )
            return;

        icon_pixels = readpng( icon_data, icon_bytes, &icon_w, &icon_h );
        if ( !icon_pixels )
            return;

       /* the data is loaded into memory as RGBA bytes by libpng. we want to manage
        * the values as 32-bit ARGB pixels, so swap the bytes accordingly depending
        * on our CPU endianess
        */
        {
            unsigned*  d     = icon_pixels;
            unsigned*  d_end = d + icon_w*icon_h;

            for ( ; d < d_end; d++ ) {
                unsigned  pix = d[0];
#if WORDS_BIGENDIAN
                /* R,G,B,A read as RGBA => ARGB */
                pix = ((pix >> 8) & 0xffffff) | (pix << 24);
#else
                /* R,G,B,A read as ABGR => ARGB */
                pix = (pix & 0xff00ff00) | ((pix >> 16) & 0xff) | ((pix & 0xff) << 16);
#endif
                d[0] = pix;
            }
        }

        SDL_Surface* icon = sdl_surface_from_argb32( icon_pixels, icon_w, icon_h );
        if (icon != NULL) {
            SDL_WM_SetIcon(icon, NULL);
            SDL_FreeSurface(icon);
            free( icon_pixels );
        }
#endif	/* !_WIN32 */
    }
}


/***********************************************************************/
/***********************************************************************/
/*****                                                             *****/
/*****            S K I N   I M A G E S                            *****/
/*****                                                             *****/
/***********************************************************************/
/***********************************************************************/

void send_key_event(unsigned code, unsigned down)
{
    if(code == 0) {
        return;
    }
    if (VERBOSE_CHECK(keys))
        printf(">> KEY [0x%03x,%s]\n", (code & 0x1ff), down ? "down" : " up " );
    kbd_put_keycode((code & 0x1ff) | (down ? 0x200 : 0));
}



typedef struct {
    AConfig*       aconfig;
    SkinFile*      layout_file;
    SkinLayout*    layout;
    SkinKeyboard*  keyboard;
    SkinWindow*    window;
    int            win_x;
    int            win_y;
    int            show_trackball;

    SkinImage*     onion;
    SkinRotation   onion_rotation;
    int            onion_alpha;
} QEmulator;

static QEmulator   qemulator[1];

static void
qemulator_done( QEmulator*  emulator )
{
    if (emulator->window) {
        skin_window_free(emulator->window);
        emulator->window = NULL;
    }
    if (emulator->keyboard) {
        skin_keyboard_free(emulator->keyboard);
        emulator->keyboard = NULL;
    }
    emulator->layout = NULL;
    if (emulator->layout_file) {
        skin_file_free(emulator->layout_file);
        emulator->layout_file = NULL;
    }
}


static void
qemulator_setup( QEmulator*  emulator );

static void
qemulator_fb_update( void*   _emulator, int  x, int  y, int  w, int  h )
{
    QEmulator*  emulator = _emulator;

    if (emulator->window)
        skin_window_update_display( emulator->window, x, y, w, h );
}

static void
qemulator_fb_rotate( void*  _emulator, int  rotation )
{
    QEmulator*     emulator = _emulator;

    qemulator_setup( emulator );
}



static int
qemulator_init( QEmulator*     emulator,
                AConfig*       aconfig,
                const char*    basepath,
                int            x,
                int            y )
{
    emulator->aconfig     = aconfig;
    emulator->layout_file = skin_file_create_from_aconfig(aconfig, basepath);
    emulator->layout      = emulator->layout_file->layouts;
    emulator->keyboard    = skin_keyboard_create_from_aconfig(aconfig, arg_raw_keys);
    emulator->window      = NULL;
    emulator->win_x       = x;
    emulator->win_y       = y;

    /* register as a framebuffer clients for all displays defined in the skin file */
    SKIN_FILE_LOOP_PARTS( emulator->layout_file, part )
        SkinDisplay*  disp = part->display;
        if (disp->valid) {
            qframebuffer_add_client( disp->qfbuff,
                                        emulator,
                                        qemulator_fb_update,
                                        qemulator_fb_rotate,
                                        NULL );
        }
    SKIN_FILE_LOOP_END_PARTS
    return 0;
}

static int
get_device_dpi( void )
{
    int     dpi_device  = DEFAULT_DEVICE_DPI;

    if (arg_dpi_device != NULL) {
        char*  end;
        dpi_device = strtol( arg_dpi_device, &end, 0 );
        if (end == NULL || *end != 0 || dpi_device <= 0) {
            fprintf(stderr, "argument for -dpi-device must be a positive integer. Aborting\n" );
            exit(1);
        }
    }
    return  dpi_device;
}

static double
get_default_scale( ADisplayInfovoid )
{
    int     dpi_device  = get_device_dpi();
    int     dpi_monitor = -1;
    double  scale       = 0.0;

    if (arg_dpi_monitor) {
        char*   end;
        dpi_monitor = strtol( arg_dpi_monitor, &end, 0 );
        if (end == NULL || *end != 0 || dpi_monitor <= 0) {
            fprintf(stderr, "argument for -dpi-monitor must be a positive integer. Aborting\n" );
            exit(1);
        }
    }

    if (arg_window_scale) {
        if (!strcmp(arg_window_scale, "auto"))
        {
            /* do we need to get the host dpi resolution ? */
            if (dpi_monitor < 0) {
                int   xdpi, ydpi;

                if ( get_monitor_resolution( &xdpi, &ydpi ) < 0 ) {
                    fprintf(stderr, "could not get monitor DPI resolution from system. please use -dpi-monitor to specify one\n" );
                    exit(1);
                }
                D( "system reported monitor resolutions: xdpi=%d ydpi=%d\n", xdpi, ydpi);
                dpi_monitor = (xdpi + ydpi+1)/2;
            }
        }
        else
        {
            char*   end;
            scale = strtod( arg_window_scale, &end );
            if (end == NULL || *end != 0) {
                fprintf(stderr, "emulator: ignoring bad -window-scale argument '%s': %s\n", arg_window_scale,
                        "not a number or the 'auto' keyword" );
                exit(1);
            } else if ( scale < 0.1 || scale > 3. ) {
                fprintf(stderr, "emulator: ignoring bad -window-scale argument '%s': %s\n", arg_window_scale,
                        "must be between 0.1 and 3.0" );
                exit(1);
            }
        }
    }

    if (scale == 0.0 && dpi_monitor > 0)
        scale = dpi_monitor*1.0/dpi_device;

    if (scale == 0.0)
        scale = 1.0;

    return scale;
}

void
android_emulator_set_window_scale( double  scale, int  is_dpi )
{
    QEmulator*  emulator = qemulator;

    if (is_dpi)
        scale /= get_device_dpi();

    if (emulator->window)
        skin_window_set_scale( emulator->window, scale );
}


static void
qemulator_set_title( QEmulator*  emulator )
{
    char  temp[64];

    if (emulator->window == NULL)
        return;

    snprintf( temp, sizeof(temp), "Android Emulator (%d)", android_base_port );
    skin_window_set_title( emulator->window, temp );
}

/* called by the emulated framebuffer device each time the content of the
 * framebuffer has changed. the rectangle is the bounding box of all changes
 */
static void
sdl_update(DisplayState *ds, int x, int y, int w, int h)
{
    /* this function is being called from the console code that is currently inactive
    ** simple totally ignore it...
    */
    (void)ds;
    (void)x;
    (void)y;
    (void)w;
    (void)h;
}



static void
qemulator_setup( QEmulator*  emulator )
{
    if ( !emulator->window && !arg_no_window ) {
        SkinLayout*  layout = emulator->layout;
        double       scale  = get_default_scale();

        emulator->window = skin_window_create( layout, emulator->win_x, emulator->win_y, scale);
        if (emulator->window == NULL)
            return;

        {
            SkinTrackBall*           ball;
            SkinTrackBallParameters  params;

            params.diameter   = 30;
            params.ring       = 2;
            params.ball_color = 0xffe0e0e0;
            params.dot_color  = 0xff202020;
            params.ring_color = 0xff000000;

            ball = skin_trackball_create( &params );
            skin_window_set_trackball( emulator->window, ball );
        }

        if ( emulator->onion != NULL )
            skin_window_set_onion( emulator->window,
                                   emulator->onion,
                                   emulator->onion_rotation,
                                   emulator->onion_alpha );

        qemulator_set_title( emulator );
    }
}


/* called by the emulated framebuffer device each time the framebuffer
 * is resized or rotated */
static void
sdl_resize(DisplayState *ds, int w, int h, int rotation)
{
    fprintf(stderr, "weird, sdl_resize being called with framebuffer interface\n");
    exit(1);
}


static void sdl_refresh(DisplayState *ds)
{
    QEmulator*     emulator = ds->opaque;
    SDL_Event      ev;
    SkinWindow*    window   = emulator->window;
    SkinKeyboard*  keyboard = emulator->keyboard;

   /* this will eventually call sdl_update if the content of the VGA framebuffer
    * has changed */
    qframebuffer_check_updates();

    if (window == NULL)
        return;

    while(SDL_PollEvent(&ev)){
        switch(ev.type){
        case SDL_VIDEOEXPOSE:
            skin_window_redraw( window, NULL );
            break;

        case SDL_KEYDOWN:
#ifdef _WIN32
            /* special code to deal with Alt-F4 properly */
            if (ev.key.keysym.sym == SDLK_F4 &&
                ev.key.keysym.mod & KMOD_ALT) {
              goto CleanExit;
            }
#endif
#ifdef __APPLE__
            /* special code to deal with Command-Q properly */
            if (ev.key.keysym.sym == SDLK_q &&
                ev.key.keysym.mod & KMOD_META) {
              goto CleanExit;
            }
#endif
            skin_keyboard_process_event( keyboard, &ev, 1 );
            break;

        case SDL_KEYUP:
            skin_keyboard_process_event( keyboard, &ev, 0 );
            break;

        case SDL_MOUSEMOTION:
            skin_window_process_event( window, &ev );
            break;

        case SDL_MOUSEBUTTONDOWN:
        case SDL_MOUSEBUTTONUP:
            {
                int  down = (ev.type == SDL_MOUSEBUTTONDOWN);
                if (ev.button.button == 4)
                {
                    /* scroll-wheel simulates DPad up */
                    send_key_event( kKeyCodeDpadUp, down );
                }
                else if (ev.button.button == 5)
                {
                    /* scroll-wheel simulates DPad down */
                    send_key_event( kKeyCodeDpadDown, down );
                }
                else if (ev.button.button == SDL_BUTTON_LEFT) {
                    skin_window_process_event( window, &ev );
                }
#if 0
                else {
                fprintf(stderr, "... mouse button %s: button=%d state=%04x x=%d y=%d\n",
                                down ? "down" : "up  ",
                                ev.button.button, ev.button.state, ev.button.x, ev.button.y);
                }
#endif
            }
            break;

        case SDL_QUIT:
#if defined _WIN32 || defined __APPLE__
        CleanExit:
#endif
            /* only save emulator config through clean exit */
            qemulator_done( emulator );
            qemu_system_shutdown_request();
            return;
        }
    }

    skin_keyboard_flush( keyboard );
}


/* used to respond to a given keyboard command shortcut
 */
static void
handle_key_command( void*  opaque, SkinKeyCommand  command, int  down )
{
    static const struct { SkinKeyCommand  cmd; AndroidKeyCode  kcode; }  keycodes[] =
    {
        { SKIN_KEY_COMMAND_BUTTON_CALL,   kKeyCodeCall },
        { SKIN_KEY_COMMAND_BUTTON_HOME,   kKeyCodeHome },
        { SKIN_KEY_COMMAND_BUTTON_BACK,   kKeyCodeBack },
        { SKIN_KEY_COMMAND_BUTTON_HANGUP, kKeyCodeEndCall },
        { SKIN_KEY_COMMAND_BUTTON_POWER,  kKeyCodePower },
        { SKIN_KEY_COMMAND_BUTTON_SEARCH,      kKeyCodeSearch },
        { SKIN_KEY_COMMAND_BUTTON_MENU,        kKeyCodeMenu },
        { SKIN_KEY_COMMAND_BUTTON_DPAD_UP,     kKeyCodeDpadUp },
        { SKIN_KEY_COMMAND_BUTTON_DPAD_LEFT,   kKeyCodeDpadLeft },
        { SKIN_KEY_COMMAND_BUTTON_DPAD_RIGHT,  kKeyCodeDpadRight },
        { SKIN_KEY_COMMAND_BUTTON_DPAD_DOWN,   kKeyCodeDpadDown },
        { SKIN_KEY_COMMAND_BUTTON_DPAD_CENTER, kKeyCodeDpadCenter },
        { SKIN_KEY_COMMAND_BUTTON_VOLUME_UP,   kKeyCodeVolumeUp },
        { SKIN_KEY_COMMAND_BUTTON_VOLUME_DOWN, kKeyCodeVolumeDown },
        { SKIN_KEY_COMMAND_NONE, 0 }
    };
    int          nn;
    static int   tracing = 0;
    QEmulator*   emulator = opaque;


    for (nn = 0; keycodes[nn].kcode != 0; nn++) {
        if (command == keycodes[nn].cmd) {
            unsigned  code = keycodes[nn].kcode;
            if (down)
                code |= 0x200;
            kbd_put_keycode( code );
            return;
        }
    }

    // for the trackball command, handle down events to enable, and
    // up events to disable
    if (command == SKIN_KEY_COMMAND_TOGGLE_TRACKBALL) {
        skin_window_toggle_trackball( emulator->window );
        emulator->show_trackball = !emulator->show_trackball;
        //qemulator_set_title( emulator );
        return;
    }

    // only handle down events for the rest
    if (down == 0)
        return;

    switch (command)
    {
    case SKIN_KEY_COMMAND_TOGGLE_NETWORK:
        {
            qemu_net_disable = !qemu_net_disable;
            if (android_modem) {
                amodem_set_data_registration(
                        android_modem,
                qemu_net_disable ? A_REGISTRATION_UNREGISTERED
                    : A_REGISTRATION_HOME);
            }
            D( "network is now %s", qemu_net_disable ? "disconnected" : "connected" );
        }
        break;

    case SKIN_KEY_COMMAND_TOGGLE_FULLSCREEN:
        if (emulator->window) {
            skin_window_toggle_fullscreen(emulator->window);
        }
        break;

    case SKIN_KEY_COMMAND_TOGGLE_TRACING:
        {
#ifdef CONFIG_TRACE
            tracing = !tracing;
            if (tracing)
                start_tracing();
            else
                stop_tracing();
#endif
        }
        break;

    case SKIN_KEY_COMMAND_ONION_ALPHA_UP:
    case SKIN_KEY_COMMAND_ONION_ALPHA_DOWN:
        if (emulator->onion)
        {
            int  alpha = emulator->onion_alpha;

            if (command == SKIN_KEY_COMMAND_ONION_ALPHA_UP)
                alpha += 16;
            else
                alpha -= 16;

            if (alpha > 256)
                alpha = 256;
            else if (alpha < 0)
                alpha = 0;

            emulator->onion_alpha = alpha;
            skin_window_set_onion( emulator->window, emulator->onion, emulator->onion_rotation, alpha );
            skin_window_redraw( emulator->window, NULL );
            //dprint( "onion alpha set to %d (%.f %%)", alpha, alpha/2.56 );
        }
        break;

    case SKIN_KEY_COMMAND_CHANGE_LAYOUT_PREV:
    case SKIN_KEY_COMMAND_CHANGE_LAYOUT_NEXT:
        {
            SkinLayout*  layout = NULL;

            if (command == SKIN_KEY_COMMAND_CHANGE_LAYOUT_NEXT) {
                layout = emulator->layout->next;
                if (layout == NULL)
                    layout = emulator->layout_file->layouts;
            }
            else if (command == SKIN_KEY_COMMAND_CHANGE_LAYOUT_PREV) {
                layout = emulator->layout_file->layouts;
                while (layout->next && layout->next != emulator->layout)
                    layout = layout->next;
            }
            if (layout != NULL) {
                emulator->layout = layout;
                skin_window_reset( emulator->window, layout );

                if (emulator->keyboard)
                    skin_keyboard_set_rotation( emulator->keyboard,
                                                skin_layout_get_dpad_rotation( layout ) );
                qframebuffer_invalidate_all();
                qframebuffer_check_updates();
            }
        }
        break;

    default:
        /* XXX: TODO ? */
        ;
    }
}


static void sdl_at_exit(void)
{
    emulator_config_done();
    qemulator_done( qemulator );
    SDL_Quit();
}


void sdl_display_init(DisplayState *ds, int full_screen)
{
    QEmulator*    emulator = qemulator;
    SkinDisplay*  disp     = skin_layout_get_display(emulator->layout);

//    fprintf(stderr,"*** sdl_display_init ***\n");
    ds->opaque = emulator;

    if (disp->rotation & 1) {
        ds->width  = disp->rect.size.h;
        ds->height = disp->rect.size.w;
    } else {
        ds->width  = disp->rect.size.w;
        ds->height = disp->rect.size.h;
    }

    ds->dpy_update  = sdl_update;
    ds->dpy_resize  = sdl_resize;
    ds->dpy_refresh = sdl_refresh;

    skin_keyboard_enable( emulator->keyboard, 1 );
    skin_keyboard_on_command( emulator->keyboard, handle_key_command, emulator );
}


extern SkinKeyboard*  android_emulator_get_keyboard(void)
{
    return qemulator->keyboard;
}

static const char*  skin_network_speed = NULL;
static const char*  skin_network_delay = NULL;

/* list of skin aliases */
static const struct {
    const char*  name;
    const char*  alias;
} skin_aliases[] = {
    { "QVGA-L", "320x240" },
    { "QVGA-P", "240x320" },
    { "HVGA-L", "480x320" },
    { "HVGA-P", "320x480" },
    { "QVGA", "320x240" },
    { "HVGA", "320x480" },
    { NULL, NULL }
};

/* this is used by hw/events_device.c to send the charmap name to the system */
const char*    android_skin_keycharmap = NULL;

void init_skinned_ui(const char *path, const char *name)
{
    char      tmp[1024];
    AConfig*  root;
    AConfig*  n;
    int       win_x, win_y, flags;

    signal(SIGINT, SIG_DFL);
#ifndef _WIN32
    signal(SIGQUIT, SIG_DFL);
#endif

    /* we're not a game, so allow the screensaver to run */
    putenv("SDL_VIDEO_ALLOW_SCREENSAVER=1");

    flags = SDL_INIT_NOPARACHUTE;
    if (!arg_no_window)
        flags |= SDL_INIT_VIDEO;

    if(SDL_Init(flags)){
        fprintf(stderr, "SDL init failure, reason is: %s\n", SDL_GetError() );
        exit(1);
    }

    if (!arg_no_window) {
        SDL_EnableUNICODE(!arg_raw_keys);
        SDL_EnableKeyRepeat(0,0);

        sdl_set_window_icon();
    }
    else
    {
#ifndef _WIN32
       /* prevent SIGTTIN and SIGTTOUT from stopping us. this is necessary to be
        * able to run the emulator in the background (e.g. "emulator &").
        * despite the fact that the emulator should not grab input or try to
        * write to the output in normal cases, we're stopped on some systems
        * (e.g. OS X)
        */
        signal(SIGTTIN, SIG_IGN);
        signal(SIGTTOU, SIG_IGN);
#endif
    }
    atexit(sdl_at_exit);

    root = aconfig_node("", "");

    if(name) {
        /* Support skin aliases like QVGA-H QVGA-P, etc...
           But first we check if it's a directory that exist before applyin the alias */
        sprintf(tmp, "%s/%s", path, name);
        if(access(tmp, F_OK) != 0) {
            /* directory is invalid, apply alias */
            int  nn;
            for (nn = 0; ; nn++ ) {
                const char*  skin_name  = skin_aliases[nn].name;
                const char*  skin_alias = skin_aliases[nn].alias;

                if ( !skin_name )
                    break;

                if ( !strcasecmp( skin_name, name ) ) {
                    name = skin_alias;
                    break;
                }
            }
        }

        /* Magically support skins like "320x240" */
        if(isdigit(name[0])) {
            char *x = strchr(name, 'x');
            if(x && isdigit(x[1])) {
                int width = atoi(name);
                int height = atoi(x + 1);
                sprintf(tmp,"display {\n  width %d\n  height %d\n}\n", width, height);
                aconfig_load(root, strdup(tmp));
                path = ":";
                goto found_a_skin;
            }
        }

        sprintf(tmp, "%s/%s/layout", path, name);
        D("trying to load skin file '%s'", tmp);

        if(aconfig_load_file(root, tmp) >= 0) {
            sprintf(tmp, "%s/%s/", path, name);
            path = tmp;
            goto found_a_skin;
        } else {
            fprintf(stderr, "### WARNING: could not load skin file '%s', using built-in one\n", tmp);
        }
    }

    {
        const unsigned char*  layout_base;
        size_t                layout_size;

        name = "<builtin>";

        layout_base = android_resource_find( "layout", &layout_size );
        if (layout_base != NULL) {
            char*  base = malloc( layout_size+1 );
            memcpy( base, layout_base, layout_size );
            base[layout_size] = 0;

            D("parsing built-in skin layout file (size=%d)", (int)layout_size);
            aconfig_load(root, base);
            path = ":";
        } else {
            fprintf(stderr, "Couldn't load builtin skin\n");
            exit(1);
        }
    }

found_a_skin:
    {
        AConfig*  node = aconfig_find( emulator_config, "window" );

        win_x = 10;
        win_y = 10;

        if (node == NULL) {
            dprint( "broken configuration file doesn't have 'window' element" );
        } else {
            win_x = aconfig_int( node, "x", win_x );
            win_y = aconfig_int( node, "y", win_y );
        }
    }
    if ( qemulator_init( qemulator, root, path, win_x, win_y ) < 0 ) {
        fprintf(stderr, "### Error: could not load emulator skin '%s'\n", name);
        exit(1);
    }

    android_skin_keycharmap = skin_keyboard_charmap_name(qemulator->keyboard);

    /* the default network speed and latency can now be specified by the device skin */
    n = aconfig_find(root, "network");
    if (n != NULL) {
        skin_network_speed = aconfig_str(n, "speed", 0);
        skin_network_delay = aconfig_str(n, "delay", 0);
    }

#if 0
    /* create a trackball if needed */
    n = aconfig_find(root, "trackball");
    if (n != NULL) {
        SkinTrackBallParameters  params;

        params.x        = aconfig_unsigned(n, "x", 0);
        params.y        = aconfig_unsigned(n, "y", 0);
        params.diameter = aconfig_unsigned(n, "diameter", 20);
        params.ring     = aconfig_unsigned(n, "ring", 1);

        params.ball_color = aconfig_unsigned(n, "ball-color", 0xffe0e0e0);
        params.dot_color  = aconfig_unsigned(n, "dot-color",  0xff202020 );
        params.ring_color = aconfig_unsigned(n, "ring-color", 0xff000000 );

        qemu_disp->trackball = skin_trackball_create( &params );
        skin_trackball_refresh( qemu_disp->trackball );
    }
#endif

    /* add an onion overlay image if needed */
    if (arg_onion) {
        SkinImage*  onion = skin_image_find_simple( arg_onion );
        int         alpha, rotate;

        if ( arg_onion_alpha && 1 == sscanf( arg_onion_alpha, "%d", &alpha ) ) {
            alpha = (256*alpha)/100;
        } else
            alpha = 128;

        if ( arg_onion_rotate && 1 == sscanf( arg_onion_rotate, "%d", &rotate ) ) {
            rotate &= 3;
        } else
            rotate = SKIN_ROTATION_0;

        qemulator->onion          = onion;
        qemulator->onion_alpha    = alpha;
        qemulator->onion_rotation = rotate;
    }
}

/* where to look relative to appdir for system.img and friends */
const char *sysdir_paths[] = {
    "%s/lib/images/%s",    /* emulator in <sdk>, images in <sdk>/lib/images */
    "%s/%s",                /* emulator and images in the same directory */
    "%s/../%s",
    "%s/../../%s",          /* emulator in <build>/host/bin, images in <build> */
    0
};

int qemu_main(int argc, char **argv);

/* this function dumps the QEMU help */
extern void  help( void );
extern void  emulator_help( void );

static char *arg_skindir = 0;
static char *arg_skin = 0;
static char *arg_sysdir = 0;
static char *arg_datadir = 0;
static char *arg_image = 0;
static char *arg_initdata = 0;
static char *arg_data = 0;
static char *arg_sdcard = 0;
static char *arg_os_type = 0;
static char *arg_cache = 0;
static char *arg_kernel = 0;
static char *arg_guest0_load_addr = 0;
static char *arg_guest0_filename = 0;
static char *arg_guest1_load_addr = 0;
static char *arg_guest1_filename = 0;
static char *arg_ramdisk = 0;
static char *arg_tracefile = 0;
static char *arg_netspeed = 0;
static char *arg_netdelay = 0;
static char *arg_logcat = 0;
static char *arg_rotation = 0;
static char *arg_radio = 0;
static char *arg_onion = 0;
static char *arg_onion_alpha = 0;
static char *arg_timezone = 0;
static char *arg_mic = 0;
static char *arg_http_proxy = 0;
static char*  arg_dpi_device = 0;
static char*  arg_dpi_monitor = 0;
static char*  arg_window_scale = 0;
static char*  arg_port = 0;
static char*  arg_adb_port = 0;
static char*  arg_cpu_delay = 0;
       char*  arg_dns_server = 0;
static char*  arg_report_console = 0;
static char*  arg_gps = 0;
static char*  arg_keyset = 0;
static char*  arg_audio = 0;
static char*  arg_audio_in = 0;
static char*  arg_audio_out = 0;
static char*  arg_shell_serial = 0;
static int arg_nocache = 0;
static int arg_netfast = 0;
static int arg_nopoll = 0;
static int arg_noskin = 0;
static int arg_show_kernel = 0;
static int arg_console = 0;
static int arg_shell = 0;
static int arg_nojni = 0;
static int arg_noaudio = 0;
static int arg_useaudio = 0;
static int arg_no_boot_anim = 0;
static int arg_old_system = 0;

#if 0
static int arg_flashkeys = 0;
#endif
static int arg_wipedata = 0;
static int arg_raw_keys = 0;
static int arg_version = 0;
int        arg_no_window = 0;

#define  VERBOSE_OPT(str,var)   { str, &var }

#define  _VERBOSE_TAG(x,y)   { #x, VERBOSE_##x, y },
static const struct { const char*  name; int  flag; const char*  text; }
verbose_options[] = {
    VERBOSE_TAG_LIST
    { 0, 0, 0 }
};


static void
help_disk_images( void )
{
    char  datadir[256];

    bufprint_config_path( datadir, datadir + sizeof(datadir) );

    printf(
    "  the emulator needs several key read-only image files to run appropriately.\n"
    "  they are normally searched in 'lib/images' under the emulator's program\n"
    "  location, but you can also use '-system <dir>' to specify a different\n"
    "  directory\n\n"

    "  the files that are looked up in the system directory are, by default:\n\n"

    "    kernel-qemu      the emulator-specific Linux kernel image\n"
    "    ramdisk.img      the ramdisk image used to boot the system\n"
    "    system.img       the *initial* system image\n"
    "    userdata.img     the *initial* data partition image\n\n"

    "  use '-kernel <file>', '-ramdisk <file>', '-image <file>' and\n"
    "  '-initdata <file>' respectively if you want to override these.\n\n"

    "  several *writable* files are also used at runtime. they are searched\n"
    "  in a specific data directory, which is, on this system:\n\n"

    "    %s\n\n"

    "  you can use the '-datadir <dir>' option to use another directory.\n"
    "  the writable image files there are:\n\n"

    "    userdata-qemu.img    the persistent /data partition image\n"
    "    sdcard.img           an *optional* SD Card partition image file\n\n"

    "  use '-data <file>' to specify an alternative /data partition image. if\n"
    "  <file> does not exist, it will be created with a copy of the initial\n"
    "  userdata.img file.\n\n"

    "  use '-wipe-data' to copy the initial data partition image into your\n"
    "  data image. this has the effect of resetting everything in /data to the\n"
    "  'factory defaults', wiping all installed applications and settings.\n\n"

    "  use '-sdcard <file>' to specify a different SD Card partition image. these\n"
    "  are simple FAT32 image disks that can be used with the 'mksdcard' tool that\n"
    "  comes with the Android SDK. If <file> does not exist, the option is ignored\n"
    "  and the emulator will start without an attached SD Card.\n\n"

    "  finally, some *writable* *temporary* files are used at runtime:\n\n"

    "    the *writable* system image\n"
    "    the /cache partition image\n\n"

    "  the writable system image is initialized on startup with the read-only\n"
    "  system.img file. it is always deleted on exit, and there is currently no\n"
    "   way to make changes in there persistent\n\n"

    "  the /cache partition image is initially empty, and is used by the browser\n"
    "  to cache downloaded web pages and images. you can use '-cache <file>' to\n"
    "  make it persistent. if <file> does not exist, it will be created empty.\n"
    "  another option is to disable the cache partition with '-nocache'\n\n",

    datadir );
}

static void
help_keys(void)
{
    int  pass, maxw = 0;

    printf( "  When running the emulator, use the following keypresses:\n\n");

    if (!android_keyset)
        android_keyset = skin_keyset_new_from_text( skin_keyset_get_default() );

    for (pass = 0; pass < 2; pass++) {
        SkinKeyCommand  cmd;

        for (cmd = SKIN_KEY_COMMAND_NONE+1; cmd < SKIN_KEY_COMMAND_MAX; cmd++)
        {
            SkinKeyBinding  bindings[ SKIN_KEY_COMMAND_MAX_BINDINGS ];
            int             n, count, len;
            char            temp[32], *p = temp, *end = p + sizeof(temp);

            count = skin_keyset_get_bindings( android_keyset, cmd, bindings );
            if (count <= 0)
                continue;

            for (n = 0; n < count; n++) {
                p = bufprint(p, end, "%s%s", (n == 0) ? "" : ", ",
                            skin_key_symmod_to_str( bindings[n].sym, bindings[n].mod ) );
            }

            if (pass == 0) {
                len = strlen(temp);
                if (len > maxw)
                    maxw = len;
            } else {
                printf( "    %-*s  %s\n", maxw, temp, skin_key_command_description(cmd) );
            }
        }
    }
    printf( "\n" );
    printf( "  note that NumLock must be deactivated for keypad keys to work\n\n" );
}


static void
help_environment(void)
{
    printf(
    "  the Android emulator looks at various environment variables when it starts:\n\n"

    "  if ANDROID_LOG_TAGS is defined, it will be used as in '-logcat <tags>'\n\n"

    "  if 'http_proxy' is defined, it will be used as in '-http-proxy <proxy>'\n\n"

    "  If ANDROID_VERBOSE is defined, it can contain a comma-separated list of\n"
    "  verbose items. for example:\n\n"

    "      ANDROID_VERBOSE=socket,radio\n\n"

    "  is equivalent to using the '-verbose -verbose-socket -verbose-radio'\n"
    "  options together. unsupported items will be ignored\n\n"

    );
}


static void
help_keyset_file(void)
{
    int           n, count;
    const char**  strings;
    char          temp[MAX_PATH];

    printf(
    "  on startup, the emulator looks for 'keyset' file that contains the\n"
    "  configuration of key-bindings to use. the default location on this\n"
    "  system is:\n\n"
    );

    bufprint_config_file( temp, temp+sizeof(temp), KEYSET_FILE );
    printf( "    %s\n\n", temp );

    printf(
    "  if the file doesn't exist, the emulator writes one containing factory\n"
    "  defaults. you are then free to modify it to suit specific needs.\n\n"
    "  this file shall contain a list of text lines in the following format:\n\n"

    "    <command> [<modifiers>]<key>\n\n"

    "  where <command> is an emulator-specific command name, i.e. one of:\n\n"
    );

    count   = SKIN_KEY_COMMAND_MAX-1;
    strings = calloc( count, sizeof(char*) );
    for (n = 0; n < count; n++)
        strings[n] = skin_key_command_to_str(n+1);

    print_tabular( strings, count, "    ", 80-8 );
    free(strings);

    printf(
    "\n"
    "  <modifers> is an optional list of <modifier> elements (without separators)\n"
    "  which can be one of:\n\n"

    "    Ctrl-     Left Control Key\n"
    "    Shift-    Left Shift Key\n"
    "    Alt-      Left Alt key\n"
    "    RCtrl-    Right Control Key\n"
    "    RShift-   Right Shift Key\n"
    "    RAlt-     Right Alt key (a.k.a AltGr)\n"
    "\n"
    "  finally <key> is a QWERTY-specific keyboard symbol which can be one of:\n\n"
    );
    count   = skin_keysym_str_count();
    strings = calloc( count, sizeof(char*) );
    for (n = 0; n < count; n++)
        strings[n] = skin_keysym_str(n);

    print_tabular( strings, count, "    ", 80-8 );
    free(strings);

    printf(
    "\n"
    "  case is not significant, and a single command can be associated to up\n"
    "  to %d different keys. to bind a command to multiple keys, use commas to\n"
    "  separate them. here are some examples:\n\n",
    SKIN_KEY_COMMAND_MAX_BINDINGS );

    printf(
    "    TOGGLE_NETWORK      F8                # toggle the network on/off\n"
    "    CHANGE_LAYOUT_PREV  Keypad_7,Ctrl-J   # switch to a previous skin layout\n"
    "\n"
    );
}


static void
help_debug_tags(void)
{
    int  n;

    printf(
    "  the '-debug <tags>' option can be used to enable or disable debug\n"
    "  messages from specific parts of the emulator. <tags> must be a list\n"
    "  (separated by space/comma/column) of <component> names, which can be one of:\n\n"
    );

    for (n = 0; n < VERBOSE_MAX; n++)
        printf( "    %-12s    %s\n", verbose_options[n].name, verbose_options[n].text );
    printf( "    %-12s    %s\n", "all", "all components together\n" );

    printf(
    "\n"
    "  each <component> can be prefixed with a single '-' to indicate the disabling\n"
    "  of its debug messages. for example:\n\n"

    "    -debug all,-socket,-keys\n\n"

    "  enables all debug messages, except the ones related to network sockets\n"
    "  and key bindings/presses\n\n"
    );
}

static void
help_char_devices(void)
{
    printf(
    "  various emulation options take a <device> specification that can be used to\n"
    "  specify something to hook to an emulated device or communication channel.\n"
    "  here is the list of supported <device> specifications:\n\n"

    "      stdio\n"
    "          standard input/output. this may be subject to character\n"
    "          translation (e.g. LN <=> CR/LF)\n\n"

    "      COM<n>   [Windows only]\n"
    "          where <n> is a digit. host serial communication port.\n\n"

    "      pipe:<filename>\n"
    "          named pipe <filename>\n\n"

    "      file:<filename>\n"
    "          write output to <filename>, no input can be read\n\n"

    "      pty  [Linux only]\n"
    "          pseudo TTY (a new PTY is automatically allocated)\n\n"

    "      /dev/<file>  [Unix only]\n"
    "          host char device file, e.g. /dev/ttyS0. may require root access\n\n"

    "      /dev/parport<N>  [Linux only]\n"
    "          use host parallel port. may require root access\n\n"

    "      unix:<path>[,server][,nowait]]     [Unix only]\n"
    "          use a Unix domain socket. if you use the 'server' option, then\n"
    "          the emulator will create the socket and wait for a client to\n"
    "          connect before continuing, unless you also use 'nowait'\n\n"

    "      tcp:[<host>]:<port>[,server][,nowait][,nodelay]\n"
    "          use a TCP socket. 'host' is set to localhost by default. if you\n"
    "          use the 'server' option will bind the port and wait for a client\n"
    "          to connect before continuing, unless you also use 'nowait'. the\n"
    "          'nodelay' option disables the TCP Nagle algorithm\n\n"

    "      telnet:[<host>]:<port>[,server][,nowait][,nodelay]\n"
    "          similar to 'tcp:' but uses the telnet protocol instead of raw TCP\n\n"

    "      udp:[<remote_host>]:<remote_port>[@[<src_ip>]:<src_port>]\n"
    "          send output to a remote UDP server. if 'remote_host' is no\n"
    "          specified it will default to '0.0.0.0'. you can also receive input\n"
    "          through UDP by specifying a source address after the optional '@'.\n\n"

    "      fdpair:<fd1>,<fd2>  [Unix only]\n"
    "          redirection input and output to a pair of pre-opened file\n"
    "          descriptors. this is mostly useful for scripts and other\n"
    "          programmatic launches of the emulator.\n\n"

    "      none\n"
    "          no device connected\n\n"

    "      null\n"
    "          the null device (a.k.a /dev/null on Unix, or NUL on Win32)\n\n"

    "  NOTE: these correspond to the <device> parameter of the QEMU -serial option\n"
    "        as described on http://bellard.org/qemu/qemu-doc.html#SEC10\n\n"
    );
}

static const  struct {  const char*  name; const char*  descr; void (*help_func)(void); }  help_topics[] =
{
    { "disk-images", "about disk images",      help_disk_images },
    { "keys",        "supported key bindings", help_keys },
    { "debug-tags",  "debug tags for -debug <tags>", help_debug_tags },
    { "char-devices", "character <device> specification", help_char_devices },
    { "environment", "environment variables",  help_environment },
    { "keyset-file",  "key bindings configuration file", help_keyset_file },
    { NULL, NULL }
};


static void
help_system(void)
{
    char   systemdir[MAX_PATH];
    char   *p = systemdir, *end = p + sizeof(systemdir);

    p = bufprint_app_dir( p, end );
    p = bufprint( p, end, PATH_SEP "lib" PATH_SEP "images" );

    printf(
    "  use '-system <dir>' to specify a directory where system read-only\n"
    "  image files will be searched. on this system, the default directory is:\n\n"
    "      %s\n\n", systemdir );

    printf(
    "  see '-help-disk-images' for more information about disk image files\n\n" );
}

static void
help_datadir(void)
{
    char  datadir[MAX_PATH];

    bufprint_config_path(datadir, datadir + sizeof(datadir));

    printf(
    "  use '-datadir <dir>' to specify a directory where writable image files\n"
    "  will be searched. on this system, the default directory is:\n\n"
    "      %s\n\n", datadir );

    printf(
    "  see '-help-disk-images' for more information about disk image files\n\n" );
}

static void
help_kernel(void)
{
    printf(
    "  use '-kernel <file>' to specify a Linux kernel image to be run.\n"
    "  the default image is 'kernel-qemu' from the system directory.\n\n" );
    printf(
    "  you can use '-debug-kernel' to send debug messages from the kernel\n"
    "  to the terminal\n\n" );

    printf(
    "  see '-help-disk-images' for more information about disk image files\n\n" );
}

static void
help_ramdisk(void)
{
    printf(
    "  use '-ramdisk <file>' to specify a Linux ramdisk boot image to be run in\n"
    "  the emulator. the default image is 'ramdisk.img' from the system directory.\n\n" );
    printf(
    "  see '-help-disk-images' for more information about disk image files\n\n" );
}

static void
help_image(void)
{
    printf(
    "  use '-image <file>' to specify the intial system image that will be loaded.\n"
    "  the default image is 'system.img' from the system directory.\n\n");
    printf(
    "  see '-help-disk-images' for more information about disk image files\n\n" );
}

static void
help_initdata(void)
{
    printf(
    "  use '-initdata <file>' to specify an *init* /data partition file.\n"
    "  it is only used when creating a new writable /data image file, or\n"
    "  when you use '-wipe-data' to reset it. the default is 'userdata.img'\n"
    "  from the system directory.\n\n" );
    printf(
    "  see '-help-disk-images' for more information about disk image files\n\n" );
}

static void
help_data(void)
{
    char    file[MAX_PATH];

    bufprint_config_file( file, file+sizeof(file), "userdata-qemu.img" );

    printf(
    "  use '-data <file>' to specify a different /data partition image file.\n"
    "  the default, on this system is the following:\n\n"
    "      %s\n\n", file );

    printf(
    "  see '-help-disk-images' for more information about disk image files\n\n" );
}

static void
help_wipe_data(void)
{
    printf(
    "  use '-wipe-data' to reset your /data partition image to its factory\n"
    "  defaults. this removes all installed applications and settings.\n\n" );

    printf(
    "  see '-help-disk-images' for more information about disk image files\n\n" );
}

static void
help_cache(void)
{
    printf(
    "  use '-cache <file>' to specify a /cache partition image. if <file> does\n"
    "  not exist, it will be created empty. by default, the cache partition is\n"
    "  backed by a temporary file that is deleted when the emulator exits.\n"
    "  using the -cache option allows it to be persistent.\n\n" );

    printf(
    "  the '-nocache' option can be used to disable the cache partition.\n\n" );
    printf(
    "  see '-help-disk-images' for more information about disk image files\n\n" );
}

static void
help_nocache(void)
{
    printf(
    "  use '-nocache' to disable the cache partition in the emulated system.\n"
    "  the cache partition is optional, but when available, is used by the browser\n"
    "  to cache web pages and images\n\n" );
    printf(
    "  see '-help-disk-images' for more information about disk image files\n\n" );
}

static void
help_sdcard(void)
{
    char  file[MAX_PATH];

    bufprint_config_file( file, file+sizeof(file), "sdcard.img" );

    printf(
    "  use '-sdcard <file>' to specify a SD Card image file that will be attached\n"
    "  to the emulator. By default, the following file is searched:\n\n"
    "      %s\n\n", file );
    printf(
    "  if the file does not exist, the emulator will still start, but without an\n"
    "  attached SD Card.\n\n");
    printf(
    "  see '-help-disk-images' for more information about disk image files\n\n" );
}

static void
help_skindir(void)
{
    printf(
    "  use '-skindir <dir>' to specify a directory that will be used to search\n"
    "  for emulator skins. each skin must be a subdirectory of <dir>. by default\n"
    "  the emulator will look in the 'skins' sub-directory of the system directory\n\n" );
}

static void
help_skin(void)
{
    printf(
    "  use '-skin <skin>' to specify an emulator skin, each skin corresponds to\n"
    "  the visual appearance of a given device, including buttons and keyboards,\n"
    "  and is stored as subdirectory <skin> of the skin root directory\n"
    "  (see '-help-skindir')\n\n" );

    printf(
    "  note that <skin> can also be '<width>x<height>' (e.g. '320x480') to\n"
    "  specify an exact framebuffer size, without any visual ornaments.\n\n" );
}

/* default network settings for emulator */
#define  DEFAULT_NETSPEED  "full"
#define  DEFAULT_NETDELAY  "none"

static void
help_shaper(void)
{
    int  n;

    printf(
    "  the Android emulator supports network throttling, i.e. slower network\n"
    "  bandwidth as well as higher connection latencies. this is done either through\n"
    "  skin configuration, or with '-netspeed <speed>' and '-netdelay <delay>'.\n\n"

    "  the format of -netspeed is one of the following (numbers are kbits/s):\n\n" );

    for (n = 0; android_netspeeds[n].name != NULL; n++) {
        printf( "    -netspeed %-12s %-15s  (up: %.1f, down: %.1f)\n",
                        android_netspeeds[n].name,
                        android_netspeeds[n].display,
                        android_netspeeds[n].upload/1000.,
                        android_netspeeds[n].download/1000. );
    }
    printf( "\n" );
    printf( "    -netspeed %-12s %s", "<num>", "select both upload and download speed\n");
    printf( "    -netspeed %-12s %s", "<up>:<down>", "select individual up and down speed\n");

    printf( "\n  The format of -netdelay is one of the following (numbers are msec):\n\n" );
    for (n = 0; android_netdelays[n].name != NULL; n++) {
        printf( "    -netdelay %-10s   %-15s  (min %d, max %d)\n",
                        android_netdelays[n].name, android_netdelays[n].display,
                        android_netdelays[n].min_ms, android_netdelays[n].max_ms );
    }
    printf( "    -netdelay %-10s   %s", "<num>", "select exact latency\n");
    printf( "    -netdelay %-10s   %s", "<min>:<max>", "select min and max latencies\n\n");

    printf( "  the emulator uses the following defaults:\n\n" );
    printf( "    Default network speed   is '%s'\n",   DEFAULT_NETSPEED);
    printf( "    Default network latency is '%s'\n\n", DEFAULT_NETDELAY);
}

static void
help_http_proxy(void)
{
    printf(
    "  the Android emulator allows you to redirect all TCP connections through\n"
    "  a HTTP/HTTPS proxy. this can be enabled by using the '-http-proxy <proxy>'\n"
    "  option, or by defining the 'http_proxy' environment variable.\n\n"

    "  <proxy> can be one of the following:\n\n"
    "    http://<server>:<port>\n"
    "    http://<username>:<password>@<server>:<port>\n\n"

    "  the 'http://' prefix can be omitted. If '-http-proxy <proxy>' is not used,\n"
    "  the 'http_proxy' environment variable is looked up and any value matching\n"
    "  the <proxy> format will be used automatically\n\n" );
}

static void
help_report_console(void)
{
    printf(
    "  the '-report-console <socket>' option can be used to report the\n"
    "  automatically-assigned console port number to a remote third-party\n"
    "  before starting the emulation. <socket> must be in one of these\n"
    "  formats:\n\n"

    "      tcp:<port>[,server][,max=<seconds>]\n"
    "      unix:<path>[,server][,max=<seconds>]\n"
    "\n"
    "  if the 'server' option is used, the emulator opens a server socket\n"
    "  and waits for an incoming connection to it. by default, it will instead\n"
    "  try to make a normal client connection to the socket, and, in case of\n"
    "  failure, will repeat this operation every second for 10 seconds.\n"
    "  the 'max=<seconds>' option can be used to modify the timeout\n\n"

    "  when the connection is established, the emulator sends its console port\n"
    "  number as text to the remote third-party, then closes the connection and\n"
    "  starts the emulation as usual. *any* failure in the process described here\n"
    "  will result in the emulator aborting immediately\n\n"

    "  as an example, here's a small Unix shell script that starts the emulator in\n"
    "  the background and waits for its port number with the help of the 'netcat'\n"
    "  utility:\n\n"

    "      MYPORT=5000\n"
    "      emulator -no-window -report-console tcp:$MYPORT &\n"
    "      CONSOLEPORT=`nc -l localhost $MYPORT`\n"
    "\n"
    );
}

static void
help_dpi_device(void)
{
    printf(
    "  use '-dpi-device <dpi>' to specify the screen resolution of the emulated\n"
    "  device. <dpi> must be an integer between 72 and 1000. the default is taken\n"
    "  from the skin, if available, or uses the contant value %d (an average of\n"
    "  several prototypes used during Android development).\n\n", DEFAULT_DEVICE_DPI );

    printf(
    "  the device resolution can also used to rescale the emulator window with\n"
    "  the '-scale' option (see -help-scale)\n\n"
    );
}

static void
help_window_scale(void)
{
    printf(
    "  the '-window-scale <scale>' option is deprecated.\n\n"
    "  use '-scale <scale>' instead.\n\n" );
}

static void
help_audio(void)
{
    printf(
    "  the '-audio <backend>' option allows you to select a specific backend\n"
    "  to be used to both play and record audio in the Android emulator.\n\n"

    "  this is equivalent to calling both '-audio-in <backend>' and\n"
    "  '-audio-out <backend>' at the same time.\n\n"

    "  use '-audio list' to see a list of valid <backend> values.\n"
    "  use '-audio none' to disable audio completely.\n"

    "  see -help-audio-in and -help-audio-out for more details\n\n"
    );
}

static void
help_audio_out(void)
{
    int  nn;

    printf(
        "  the  '-audio-out <backend>' option allows you to select a specific\n"
        "  backend to play audio in the Android emulator. this is mostly useful\n"
        "  on Linux\n\n"

        "  on this system, output <backend> can be one of the following:\n\n"
    );
    for ( nn = 0; ; nn++ ) {
        const char*  descr;
        const char*  name = audio_get_backend_name( 0, nn, &descr );
        if (name == NULL)
            break;
        printf( "    %-10s %s\n", name, descr );
    }
    printf( "\n" );
}

static void
help_audio_in(void)
{
    int  nn;

    printf(
        "  the  '-audio-in <backend>' option allows you to select a specific\n"
        "  backend to play audio in the Android emulator. this is mostly useful\n"
        "  on Linux\n\n"

        "  IMPORTANT NOTE:\n"
        "     on some Linux systems, broken Esd/ALSA/driver implementations will\n"
        "     make your emulator freeze and become totally unresponsive when\n"
        "     using audio recording. the only way to avoid this is to use\n"
        "     '-audio-in none' to disable it\n\n"

        "  on this system, input <backend> can be one of:\n\n"
    );
    for ( nn = 0; ; nn++ ) {
        const char*  descr;
        const char*  name = audio_get_backend_name( 1, nn, &descr );
        if (name == NULL)
            break;
        printf( "    %-10s %s\n", name, descr );
    }
    printf( "\n" );
}


static void
help_scale(void)
{
    printf(
    "  the '-scale <scale>' option is used to scale the emulator window to\n"
    "  something that better fits the physical dimensions of a real device. this\n"
    "  can be *very* useful to check that your UI isn't too small to be usable\n"
    "  on a real device.\n\n"

    "  there are three supported formats for <scale>:\n\n"

    "  * if <scale> is a real number (between 0.1 and 3.0) it is used as a\n"
    "    scaling factor for the emulator's window.\n\n"

    "  * if <scale> is an integer followed by the suffix 'dpi' (e.g. '110dpi'),\n"
    "    then it is interpreted as the resolution of your monitor screen. this\n"
    "    will be divided by the emulated device's resolution to get an absolute\n"
    "    scale. (see -help-dpi-device for details).\n\n"

    "  * finally, if <scale> is the keyword 'auto', the emulator tries to guess\n"
    "    your monitor's resolution and automatically adjusts its window\n"
    "    accordingly\n\n"

    "    NOTE: this process is *very* unreliable, depending on your OS, video\n"
    "          driver issues and other random system parameters\n\n"

    "  the emulator's scale can be changed anytime at runtime through the control\n"
    "  console. see the help for the 'window scale' command for details\n\n" );
}

static void
help_trace(void)
{
    printf(
    "  use '-trace <name>' to start the emulator with runtime code profiling support\n"
    "  profiling itself will not be enabled unless you press F9 to activate it, or\n"
    "  the executed code turns it on programmatically.\n\n"

    "  trace information is stored in directory <name>, several files are created\n"
    "  there, that can later be used with the 'traceview' program that comes with\n"
    "  the Android SDK for analysis.\n\n"

    "  note that execution will be slightly slower when enabling code profiling,\n"
    "  this is a necessary requirement of the operations being performed to record\n"
    "  the execution trace. this slowdown should not affect your system until you\n"
    "  enable the profiling though...\n\n"
    );
}

static void
help_debug_kernel(void)
{
    printf(
    "  use '-debug-kernel' to redirect debug messages from the kernel to the current\n"
    "  terminal. this is useful to check that the boot process works correctly.\n\n"
    );
}

static void
help_console(void)
{
    printf(
    "  the '-console' option is obsolete and has been replaced by '-shell'.\n"
    "  please do not use it.\n\n" );
}

static void
help_shell(void)
{
    printf(
    "  use '-shell' to create a root shell console on the current terminal.\n"
    "  this is unlike the 'adb shell' command for the following reasons:\n\n"

    "  * this is a *root* shell that allows you to modify many parts of the system\n"
    "  * this works even if the ADB daemon in the emulated system is broken\n"
    "  * pressing Ctrl-C will stop the emulator, instead of the shell.\n\n"
    "  See also '-shell-serial'.\n\n" );
}

static void
help_shell_serial(void)
{
    printf(
    "  use '-shell-serial <device>' instead of '-shell' to open a root shell\n"
    "  to the emulated system, while specifying an external communication\n"
    "  channel / host device.\n\n"

    "  '-shell-serial stdio' is identical to '-shell', while you can use\n"
    "  '-shell-serial tcp::4444,server,nowait' to talk to the shell over local\n"
    "  TCP port 4444.  '-shell-serial fdpair:3:6' would let a parent process\n"
    "  talk to the shell using fds 3 and 6.\n\n"

    "  see -help-char-devices for a list of available <device> specifications.\n\n"
    "  NOTE: you can have only one shell per emulator instance at the moment\n\n"
    );
}

static void
help_logcat(void)
{
    printf(
    "  use '-logcat <tags>' to redirect log messages from the emulated system to\n"
    "  the current terminal. <tags> is a list of space/comma-separated log filters\n"
    "  where each filter has the following format:\n\n"

    "     <componentName>:<logLevel>\n\n"

    "  where <componentName> is either '*' or the name of a given component,\n"
    "  and <logLevel> is one of the following letters:\n\n"

    "      v          verbose level\n"
    "      d          debug level\n"
    "      i          informative log level\n"
    "      w          warning log level\n"
    "      e          error log level\n"
    "      s          silent log level\n\n"

    "  for example, the following only displays messages from the 'GSM' component\n"
    "  that are at least at the informative level:\n\n"

    "    -logcat '*:s GSM:i'\n\n"

    "  if '-logcat <tags>' is not used, the emulator looks for ANDROID_LOG_TAGS\n"
    "  in the environment. if it is defined, its value must match the <tags>\n"
    "  format and will be used to redirect log messages to the terminal.\n\n"

    "  note that this doesn't prevent you from redirecting the same, or other,\n"
    "  log messages through the ADB or DDMS tools too.\n\n");
}

static void
help_noaudio(void)
{
    printf(
    "  use '-noaudio' to disable all audio support in the emulator. this may be\n"
    "  unfortunately be necessary in some cases:\n\n"

    "  * at least two users have reported that their Windows machine rebooted\n"
    "    instantly unless they used this option when starting the emulator.\n"
    "    it is very likely that the problem comes from buggy audio drivers.\n\n"

    "  * on some Linux machines, the emulator might get stuck at startup with\n"
    "    audio support enabled. this problem is hard to reproduce, but seems to\n"
    "    be related too to flaky ALSA / audio driver support.\n\n"

    "  on Linux, another option is to try to change the default audio backend\n"
    "  used by the emulator. you can do that by setting the QEMU_AUDIO_DRV\n"
    "  environment variables to one of the following values:\n\n"

    "    alsa        (use the ALSA backend)\n"
    "    esd         (use the EsounD backend)\n"
    "    sdl         (use the SDL audio backend, no audio input supported)\n"
    "    oss         (use the OSS backend)\n"
    "    none        (do not support audio)\n"
    "\n"
    "  the very brave can also try to use distinct backends for audio input\n"
    "  and audio outputs, this is possible by selecting one of the above values\n"
    "  into the QEMU_AUDIO_OUT_DRV and QEMU_AUDIO_IN_DRV environment variables.\n\n"
    );
}

static void
help_raw_keys(void)
{
    printf(
    "  this option is deprecated because one can do the same using Ctrl-K\n"
    "  at runtime (this keypress toggles between unicode/raw keyboard modes)\n\n"

    "  by default, the emulator tries to reverse-map the characters you type on\n"
    "  your keyboard to device-specific key presses whenever possible. this is\n"
    "  done to make the emulator usable with a non-QWERTY keyboard.\n\n"

    "  however, this also means that single keypresses like Shift or Alt are not\n"
    "  passed to the emulated device. the '-raw-keys' option disables the reverse\n"
    "  mapping. it should only be used when using a QWERTY keyboard on your machine\n"

    "  (should only be useful to Android system hackers, e.g. when implementing a\n"
    "  new input method).\n\n"
    );
}

static void
help_radio(void)
{
    printf(
    "  use '-radio <device>' to redirect the GSM modem emulation to an external\n"
    "  character device or program. this bypasses the emulator's internal modem\n"
    "  and should only be used for testing.\n\n"

    "  see '-help-char-devices' for the format of <device>\n\n"

    "  the data exchanged with the external device/program are GSM AT commands\n\n"

    "  note that, when running in the emulator, the Android GSM stack only supports\n"
    "  a *very* basic subset of the GSM protocol. trying to link the emulator to\n"
    "  a real GSM modem is very likely to not work properly.\n\n"
    );
}


static void
help_port(void)
{
    printf(
    "  at startup, the emulator tries to bind its control console at a free port\n"
    "  starting from 5554, in increments of two (i.e. 5554, then 5556, 5558, etc..)\n"
    "  this allows several emulator instances to run concurrently on the same\n"
    "  machine, each one using a different console port number.\n\n"

    "  use '-port <port>' to force an emulator instance to use a given console port\n\n"

    "  note that <port> must be an *even* integer between 5554 and 5584 included.\n"
    "  <port>+1 must also be free and will be reserved for ADB. if any of these\n"
    "  ports is already used, the emulator will fail to start.\n\n" );
}

static void
help_adb_port(void)
{
    printf(
    "  this option is deprecated, it is equivalent to -port but is used to specify\n"
    "  the ADB control port number of a new emulator instance instead. please use\n"
    "  -port instead.\n\n"

    "  <port> must be an *odd* integer between 5555 and 5585 inluded. <port>-1 must\n"
    "  also be free and will be reserved for the console. if any of these ports\n"
    "  is already used, the emulator will fail to start.\n\n" );
}


static void
help_onion(void)
{
    printf(
    "  use '-onion <file>' to specify a PNG image file that will be displayed on\n"
    "  top of the emulated framebuffer with translucency. this can be useful to\n"
    "  check that UI elements are correctly positioned with regards to a reference\n"
    "  graphics specification.\n\n"

    "  the default translucency is 50%%, but you can use '-onion-alpha <%%age>' to\n"
    "  select a different one, or even use keypresses at runtime to alter it\n"
    "  (see -help-keys for details)\n\n"

    "  finally, the onion image can be rotated (see -help-onion-rotate)\n\n"
    );
}

static void
help_onion_alpha(void)
{
    printf(
    "  use '-onion-alpha <percent>' to change the translucency level of the onion\n"
    "  image that is going to be displayed on top of the framebuffer (see also\n"
    "  -help-onion). the default is 50%%.\n\n"

    "  <percent> must be an integer between 0 and 100.\n\n"

    "  you can also change the translucency dynamically (see -help-keys)\n\n"
    );
}

static void
help_onion_rotate(void)
{
    printf(
    "  use '-onion-rotate <rotation>' to change the rotation of the onion\n"
    "  image loaded through '-onion <file>'. valid values for <rotation> are:\n\n"

    "   0        no rotation\n"
    "   1        90  degrees clockwise\n"
    "   2        180 degrees\n"
    "   3        270 degrees clockwise\n\n"
    );
}


static void
help_timezone(void)
{
    printf(
    "  by default, the emulator tries to detect your current timezone to report\n"
    "  it to the emulated system. use the '-timezone <timezone>' option to choose\n"
    "  a different timezone, or if the automatic detection doesn't work correctly.\n\n"

    "  VERY IMPORTANT NOTE:\n\n"
    "  the <timezone> value must be in zoneinfo format, i.e. it should look like\n"
    "  Area/Location or even Area/SubArea/Location. valid examples are:\n\n"

    "    America/Los_Angeles\n"
    "    Europe/Paris\n\n"

    "  using a human-friendly abbreviation like 'PST' or 'CET' will not work, as\n"
    "  well as using values that are not defined by the zoneinfo database.\n\n"

    "  NOTE: unfortunately, this will not work on M5 and older SDK releases\n\n"
    );
}


static void
help_dns_server(void)
{
    printf(
    "  by default, the emulator tries to detect the DNS servers you're using and\n"
    "  will setup special aliases in the emulated firewall network to allow the\n"
    "  Android system to connect directly to them. use '-dns-server <servers>' to\n"
    "  select a different list of DNS servers to be used.\n\n"

    "  <servers> must be a comma-separated list of up to 4 DNS server names or\n"
    "  IP addresses.\n\n"

    "  NOTE: on M5 and older SDK releases, only the first server in the list will\n"
    "        be used.\n\n"
    );
}


static void
help_cpu_delay(void)
{
    printf(
    "  this option is purely experimental, probably doesn't work as you would\n"
    "  expect, and may even disappear in a later emulator release.\n\n"

    "  use '-cpu-delay <delay>' to throttle CPU emulation. this may be useful\n"
    "  to detect weird race conditions that only happen on 'lower' CPUs. note\n"
    "  that <delay> is a unit-less integer that doesn't even scale linearly\n"
    "  to observable slowdowns. use trial and error to find something that\n"
    "  suits you, the 'correct' machine is very probably dependent on your\n"
    "  host CPU and memory anyway...\n\n"
    );
}


static void
help_no_boot_anim(void)
{
    printf(
    "  use '-no-boot-anim' to disable the boot animation (red bouncing ball) when\n"
    "  starting the emulator. on slow machines, this can surprisingly speed up the\n"
    "  boot sequence in tremendous ways.\n\n"

    "  NOTE: unfortunately, this will not work on M5 and older SDK releases\n\n"
    );
}


static void
help_gps(void)
{
    printf(
    "  use '-gps <device>' to emulate an NMEA-compatible GPS unit connected to\n"
    "  an external character device or socket. the format of <device> is the same\n"
    "  than the one used for '-radio <device>' (see -help-char-devices for details)\n\n"
    );
}


static void
help_keyset(void)
{
    char  temp[256];

    printf(
    "  use '-keyset <name>' to specify a different keyset file name to use when\n"
    "  starting the emulator. a keyset file contains a list of key bindings used\n"
    "  to control the emulator with the host keyboard.\n\n"

    "  by default, the emulator looks for the following file:\n\n"
    );

    bufprint_config_file(temp, temp+sizeof(temp), KEYSET_FILE);
    printf(
    "    %s\n\n", temp );

    bufprint_config_path(temp, temp+sizeof(temp));
    printf(
    "  however, if -keyset is used, then the emulator does the following:\n\n"
    "  - first, if <name> doesn't have an extension, then the '.keyset' suffix\n"
    "    is appended to it (e.g. \"foo\" => \"foo.keyset\"),\n\n"

    "  - then, the emulator searches for a file named <name> in the following\n"
    "    directories:\n\n"

    "     * the emulator configuration directory: %s\n"
    "     * the 'keysets' subdirectory of <systemdir>, if any\n"
    "     * the 'keysets' subdirectory of the program location, if any\n\n",
    temp );

    printf(
    "  if no corresponding file is found, a default set of key bindings is used.\n\n"
    "  use '-help-keys' to list the default key bindings.\n"
    "  use '-help-keyset-file' to learn more about the format of keyset files.\n"
    "\n"
    );
}

static void
help_old_system(void)
{
    printf(
    "  use '-old-system' if you want to use a recent emulator binary to run\n"
    "  an old version of the Android SDK system images. Here, 'old' means anything\n"
    "  older than version 1.4 of the emulator.\n\n"

    "  NOTE: using '-old-system' with recent system images is likely to not work\n"
    "        properly, though you may not notice it immediately (e.g. failure to\n"
    "        start the emulated GPS hardware)\n\n"
    );
}


#define  OPTION_PARAM(_opt,_var,_param,_descr,_longdescr) \
    { _opt, &_var, NULL, _param, _descr, _longdescr }

#define  OPTION_FLAG(_opt,_var,_descr,_longdescr) \
    { _opt, NULL, &_var, "", _descr, _longdescr }

static const struct
{
    const char *name;   /* argument name */
    char **value;       /* if nonzero, this is a string argument */
    int *flag;          /* if nonzero, this is a flag argument */
    const char* argname;  /* description of argument */
    const char *help;     /* description text for this option */
    void (*help_func)(void);  /* long description function for this option */
} argmap[] = {
    OPTION_PARAM("system",  arg_sysdir,  "<dir>",  "read system images from <dir>", help_system),
    OPTION_PARAM("os-type",  arg_os_type,  "<ostype>",  "<ostype>", NULL),
    OPTION_PARAM("datadir", arg_datadir, "<dir>",  "write user data into <dir>", help_datadir),
    OPTION_PARAM("kernel",  arg_kernel,  "<file>", "use specific emulated kernel", help_kernel),
	OPTION_PARAM("guest0", arg_guest0_load_addr, "<address> <elf>", "address of guest0 and filename", NULL),
	OPTION_PARAM("guest1", arg_guest1_load_addr, "<address> <elf>", "address of guest1 and filename", NULL),
    OPTION_PARAM("ramdisk", arg_ramdisk, "<file>", "ramdisk image (default <system>/ramdisk.img", help_ramdisk),
    OPTION_PARAM("image",   arg_image,   "<file>", "system image (default <system>/system.img", help_image),
    OPTION_PARAM("initdata", arg_initdata,"<file>", "initial data image (default <system>/userdata.img)", help_initdata),
    OPTION_PARAM("data",    arg_data,    "<file>",  "data image (default <datadir>/userdata-qemu.img)", help_data),
    OPTION_FLAG ("wipe-data",arg_wipedata,          "reset the user data image (copy it from initdata)", help_wipe_data),
    OPTION_PARAM("cache",   arg_cache,   "<file>",  "cache partition image (default is temporary file)", help_cache),
    OPTION_FLAG ("nocache", arg_nocache,            "disable the cache partition", help_nocache),
    OPTION_PARAM("sdcard",  arg_sdcard,  "<file>",  "SD card image (default <system>/sdcard.img)", help_sdcard),

    OPTION_PARAM("skindir", arg_skindir, "<dir>",   "search skins in <dir> (default <system>/skins)", help_skindir),
    OPTION_PARAM("skin",    arg_skin,    "<file>",  "select a given skin", help_skin),
    OPTION_FLAG ("noskin",  arg_noskin,             "don't use any emulator skin", NULL),

    OPTION_PARAM("netspeed", arg_netspeed, "<speed>", "maximum network download/upload speeds", help_shaper),
    OPTION_PARAM("netdelay", arg_netdelay, "<delay>", "network latency emulation", help_shaper),
    OPTION_FLAG ("netfast",  arg_netfast,             "disable network shaping (full speed, no latency)", help_shaper),

    OPTION_PARAM("trace",    arg_tracefile, "<name>", "enable code profiling (F9 to start)", help_trace),
    OPTION_FLAG ("show-kernel", arg_show_kernel,    "display kernel messages", help_debug_kernel),
    OPTION_FLAG ("console",  arg_console,             "obsolete, use -shell instead", help_console),
    OPTION_FLAG ("shell",    arg_shell,               "enable root shell on current terminal", help_shell),
    OPTION_FLAG ("nojni",    arg_nojni,               "disable JNI checks in the Dalvik runtime", NULL),
    OPTION_PARAM("logcat",   arg_logcat, "<tags>",  "enable logcat output with given tags", help_logcat),
    OPTION_FLAG ("noaudio",  arg_noaudio,           "disable audio support", help_noaudio),
    OPTION_FLAG ("useaudio", arg_useaudio,          "enable audio support (default)", NULL),
    OPTION_PARAM("audio",    arg_audio, "<backend>", "use specific audio backend", help_audio),
    OPTION_PARAM("audio-in", arg_audio_in, "<backend>", "use specific audio input backend", help_audio_in),
    OPTION_PARAM("audio-out", arg_audio_out, "<backend>", "use specific audio output backend", help_audio_out),
    OPTION_FLAG("raw-keys", arg_raw_keys,          "disable Unicode keyboard reverse-mapping)", help_raw_keys),
    OPTION_PARAM("radio",    arg_radio,  "<device>","redirect radio modem interface to character device", help_radio),

    OPTION_PARAM("port",    arg_port,    "<port>",  "TCP port that will be used for the console", help_port),
    OPTION_PARAM("adb-port",arg_adb_port,"<port>",  "obsolete, use -port instead", help_adb_port),
    OPTION_PARAM("onion",          arg_onion,         "<image>", "use overlay PNG image over screen", help_onion),
    OPTION_PARAM("onion-alpha",    arg_onion_alpha,   "<%age>",  "specify onion-skin translucency", help_onion_alpha),
    OPTION_PARAM("onion-rotation", arg_onion_rotate,  "0|1|2|3", "specify onion-skin rotation", help_onion_rotate),

    OPTION_PARAM("window-scale",  arg_window_scale, "<scale>", "obsolete, use -scale instead", help_window_scale),
    OPTION_PARAM("scale",         arg_window_scale, "<scale>", "scale emulator window", help_scale ),
    OPTION_PARAM("dpi-device",    arg_dpi_device, "<dpi>", "specify device's resolution in dpi (default "
                 STRINGIFY(DEFAULT_DEVICE_DPI) ")", help_dpi_device ),

    OPTION_PARAM("http-proxy", arg_http_proxy, "<proxy>",    "make TCP connections through a HTTP/HTTPS proxy", \
                 help_http_proxy),
    OPTION_PARAM("timezone",   arg_timezone,   "<timezone>", "use this timezone instead of the host's default", \
                 help_timezone),
    OPTION_PARAM("dns-server", arg_dns_server, "<servers>",  "use this DNS server(s) in the emulated system", \
                  help_dns_server),
    OPTION_PARAM("cpu-delay",  arg_cpu_delay,  "<cpudelay>", "throttle CPU emulation", help_cpu_delay),
    OPTION_FLAG ("no-boot-anim", arg_no_boot_anim, "disable boot animation for faster startup", help_no_boot_anim),

    OPTION_FLAG("no-window",        arg_no_window,         "disable graphical window display", NULL),
    OPTION_FLAG("version",          arg_version,           "display emulator version number", NULL),

    OPTION_PARAM("report-console", arg_report_console, "<socket>", "report console port to remote socket", help_report_console),
    OPTION_PARAM("gps",            arg_gps,            "<device>", "redirect NMEA GPS to character device", help_gps),
    OPTION_PARAM("keyset",         arg_keyset,         "<name>",   "specify keyset file name", help_keyset),
    OPTION_PARAM("shell-serial",   arg_shell_serial,   "<device>", "specify device for root shell", help_shell_serial),
    OPTION_FLAG ("old-system", arg_old_system, "support (pre 1.4) old system images", help_old_system ),
    { 0, 0, 0, 0, 0 },
};

#define  OPTION_SUB_PARAM(_opt,_var) \
    { _opt, &_var }

static const struct
{
	const char *name;
	const char **sub_value;
} argsubmap[] = {
	OPTION_SUB_PARAM("guest0",	arg_guest0_filename),
	OPTION_SUB_PARAM("guest1",	arg_guest1_filename),
	{ 0, 0 },
};

int
android_parse_network_speed(const char*  speed)
{
    int          n;
    char*  end;
    double       sp;

    if (speed == NULL || speed[0] == 0) {
        speed = DEFAULT_NETSPEED;
    }

    for (n = 0; android_netspeeds[n].name != NULL; n++) {
        if (!strcmp(android_netspeeds[n].name, speed)) {
            qemu_net_download_speed = android_netspeeds[n].download;
            qemu_net_upload_speed   = android_netspeeds[n].upload;
            return 0;
        }
    }

    /* is this a number ? */
    sp = strtod(speed, &end);
    if (end == speed) {
        return -1;
    }

    qemu_net_download_speed = qemu_net_upload_speed = sp*1000.;
    if (*end == ':') {
        speed = end+1;
        sp = strtod(speed, &end);
        if (end > speed) {
            qemu_net_download_speed = sp*1000.;
        }
    }
    return 0;
}


int
android_parse_network_latency(const char*  delay)
{
    int  n;
    char*  end;
    double  sp;

    if (delay == NULL || delay[0] == 0)
        delay = DEFAULT_NETDELAY;

    for (n = 0; android_netdelays[n].name != NULL; n++) {
        if ( !strcmp( android_netdelays[n].name, delay ) ) {
            qemu_net_min_latency = android_netdelays[n].min_ms;
            qemu_net_max_latency = android_netdelays[n].max_ms;
            return 0;
        }
    }

    /* is this a number ? */
    sp = strtod(delay, &end);
    if (end == delay) {
        return -1;
    }

    qemu_net_min_latency = qemu_net_max_latency = (int)sp;
    if (*end == ':') {
        delay = (const char*)end+1;
        sp = strtod(delay, &end);
        if (end > delay) {
            qemu_net_max_latency = (int)sp;
        }
    }
    return 0;
}


static int
load_keyset(const char*  path)
{
    if (access(path, R_OK) == 0) {
        AConfig*  root = aconfig_node("","");
        if (!aconfig_load_file(root, path)) {
            android_keyset = skin_keyset_new(root);
            if (android_keyset != NULL) {
                D( "keyset loaded from: %s", path);
                return 0;
            }
        }
    }
    return -1;
}

static void
parse_keyset(const char*  keyset)
{
    char   kname[MAX_PATH];
    char   temp[MAX_PATH];
    char*  p;
    char*  end;

    /* append .keyset suffix if needed */
    if (strchr(keyset, '.') == NULL) {
        p   =  kname;
        end = p + sizeof(kname);
        p   = bufprint(p, end, "%s.keyset", keyset);
        if (p >= end) {
            derror( "keyset name too long: '%s'\n", keyset);
            exit(1);
        }
        keyset = kname;
    }

    /* look for a the keyset file */
    p   = temp;
    end = p + sizeof(temp);
    p = bufprint_config_file(p, end, keyset);
    if (p < end && load_keyset(temp) == 0)
        return;

    p = temp;
    p = bufprint(p, end, "%s" PATH_SEP "keysets" PATH_SEP "%s", arg_sysdir, keyset);
    if (p < end && load_keyset(temp) == 0)
        return;

    p = temp;
    p = bufprint_app_dir(p, end);
    p = bufprint(p, end, PATH_SEP "keysets" PATH_SEP "%s", keyset);
    if (p < end && load_keyset(temp) == 0)
        return;

    return;
}

static void
write_default_keyset( void )
{
    char   path[MAX_PATH];

    bufprint_config_file( path, path+sizeof(path), KEYSET_FILE );

    /* only write if there is no file here */
    if ( access( path, R_OK ) < 0 && errno == ENOENT ) {
        int          fd = open( path, O_WRONLY | O_CREAT, 0666 );
        int          ret;
        const char*  ks = skin_keyset_get_default();


        D( "writing default keyset file to %s", path );

        if (fd < 0) {
            D( "%s: could not create file", __FUNCTION__ );
            return;
        }

        do {
            ret = write( fd, ks, strlen(ks) );
        } while (ret < 0 && errno == EINTR);

        close(fd);
    }
}

static void
parse_rotation(const char*  rotation)
{
    double  sp;
    char*   end;

    if (rotation == NULL)
        rotation = "0";

    sp = strtod(rotation, &end);
    if (end == rotation || *end != 0) {
        goto Fail;
    }

    if (sp == 0.0)
        android_framebuffer_rotation = ANDROID_ROTATION_0;
    else if (sp == 90.0)
        android_framebuffer_rotation = ANDROID_ROTATION_90;
    else if (sp == 180.0)
        android_framebuffer_rotation = ANDROID_ROTATION_180;
    else if (sp == 270.0)
        android_framebuffer_rotation = ANDROID_ROTATION_270;
    else
        goto Fail;

    return;

Fail:
    fprintf(stderr, "invalid -rotation parameter '%s'\n", rotation );
    emulator_help();
}


/* If *arg isn't already set, and <dir>/<file> exists, use it. */
static void default_file(char **arg, const char *dir, const char *file) {
    if (*arg == NULL || (*arg)[0] == '\0') {
        int n;
        *arg = malloc(strlen(dir) + strlen(PATH_SEP) + strlen(file) + 1);
        sprintf(*arg, "%s%s%s", dir, PATH_SEP, file);
        for (n = 0; argmap[n].name != NULL; ++n) {
            if (argmap[n].value == arg) {
                D("autoconfig: -%s %s", argmap[n].name, *arg);
                break;
            }
        }
    }
}


/* Call default_file() and also exit if the file doesn't exist. */
static void require_file(char **arg, const char *dir, const char *file) {
    default_file(arg, dir, file);
    if (access(*arg, F_OK) != 0) {
        int n;
        fprintf(stderr, "Cannot find file: %s\n", *arg);
        for (n = 0; argmap[n].name != NULL; ++n) {
            if (argmap[n].value == arg) {
                const char* name = argmap[n].name;
                fprintf(stderr, "Please specify a valid -%s file\n", name);
                break;
            }
        }
        exit(1);
    }
}


/* If *arg is set, require it to exist, else use the default if it exists. */
static void optional_file(char **arg, const char *dir, const char *file) {
    if (*arg == NULL) {
        default_file(arg, dir, file);
        if (access(*arg, F_OK) != 0)
            *arg = NULL;
    } else {
        /* If it's supplied explciitly, it better be there. */
        require_file(arg, dir, file);
    }
}

void emulator_help( void )
{
    int n;
    int maxwidth = 0;

    printf( "Android Emulator usage: emulator [options] [-qemu args]\n");
    printf( "  options:\n" );

    for (n = 0; argmap[n].name != NULL; ++n) {
        int width = strlen(argmap[n].name) + strlen(argmap[n].argname);
        if (width > maxwidth)
            maxwidth = width;
    }

    for (n = 0; argmap[n].name != NULL; ++n) {
        printf( "    -%s %-*s %s\n",
            argmap[n].name,
            (int)(maxwidth - strlen(argmap[n].name)),
            argmap[n].argname,
            argmap[n].help);
    }

    printf( "\n" );
    printf( "     %-*s  %s\n", maxwidth, "-qemu args...",    "pass arguments to qemu");
    printf( "     %-*s  %s\n", maxwidth, "-qemu -h", "display qemu help");
    printf( "\n" );
    printf( "     %-*s  %s\n", maxwidth, "-verbose",       "same as '-debug-init'");
    printf( "     %-*s  %s\n", maxwidth, "-debug <tags>",  "enable/disable debug messages");
    printf( "     %-*s  %s\n", maxwidth, "-debug-<tag>",   "enable specific debug messages");
    printf( "     %-*s  %s\n", maxwidth, "-debug-no-<tag>","disable specific debug messages");
    printf( "\n" );
    printf( "     %-*s  %s\n", maxwidth, "-help",    "print this help");
    printf( "     %-*s  %s\n", maxwidth, "-help-<option>", "print option-specific help");
    printf( "\n" );
    for (n = 0; help_topics[n].name; n++) {
        char    help[32];
        snprintf(help, sizeof(help), "-help-%s", help_topics[n].name);
        printf( "     %-*s  %s\n", maxwidth, help, help_topics[n].descr );
    }
    printf( "     %-*s  %s\n", maxwidth, "-help-all", "prints all help content");

    printf( "\n");
    exit(1);
}

#define  ENV_VERBOSE   "ANDROID_VERBOSE"

static void
parse_verbose_tags( const char*  tags )
{
    char*        x;
    char*        y;
    char*        x0;

    if (tags == NULL)
        return;

    x = x0 = strdup(tags);
    while (*x) {
        y = strchr(x, ',');
        if (y == NULL)
            y = x + strlen(x);
        else
            *y++ = 0;

        if (y > x+1) {
            int  nn, remove = 0;
            unsigned mask = 0;

            if (x[0] == '-') {
                remove = 1;
                x += 1;
            }

            if (!strcmp( "all", x ))
                mask = ~0;
            else {
                for (nn = 0; verbose_options[nn].name != NULL; nn++) {
                    if ( !strcmp( verbose_options[nn].name, x ) ) {
                        mask |= (1 << verbose_options[nn].flag);
                        break;
                    }
                }
            }

            if (mask == 0)
                dprint( "ignoring unknown " ENV_VERBOSE " item '%s'", x );
            else {
                if (remove)
                    android_verbose &= ~mask;
                else
                    android_verbose |= mask;
            }
        }
        x = y;
    }

    free(x0);
}


static void
parse_env_verbose( void )
{
    const char*  env = getenv( ENV_VERBOSE );
    parse_verbose_tags( env );
}


static int
add_dns_server( const char*  server_name )
{
    struct in_addr   dns1;
    struct hostent*  host = gethostbyname(server_name);

    if (host == NULL) {
        fprintf(stderr,
                "### WARNING: can't resolve DNS server name '%s'\n",
                server_name );
        return -1;
    }

    dns1 = *(struct in_addr*)host->h_addr;
    D( "DNS server name '%s' resolved to %s", server_name, inet_ntoa(dns1) );

    if ( slirp_add_dns_server( dns1 ) < 0 ) {
        fprintf(stderr,
                "### WARNING: could not add DNS server '%s' to the network stack\n", server_name);
        return -1;
    }
    return 0;
}


enum {
    REPORT_CONSOLE_SERVER = (1 << 0),
    REPORT_CONSOLE_MAX    = (1 << 1)
};

static int
get_report_console_options( char*  end, int  *maxtries )
{
    int    flags = 0;

    if (end == NULL || *end == 0)
        return 0;

    if (end[0] != ',') {
        derror( "socket port/path can be followed by [,<option>]+ only\n");
        exit(3);
    }
    end += 1;
    while (*end) {
        char*  p = strchr(end, ',');
        if (p == NULL)
            p = end + strlen(end);

        if (memcmp( end, "server", p-end ) == 0)
            flags |= REPORT_CONSOLE_SERVER;
        else if (memcmp( end, "max=", 4) == 0) {
            end  += 4;
            *maxtries = strtol( end, NULL, 10 );
            flags |= REPORT_CONSOLE_MAX;
        } else {
            derror( "socket port/path can be followed by [,server][,max=<count>] only\n");
            exit(3);
        }

        end = p;
        if (*end)
            end += 1;
    }
    return flags;
}

static void
report_console( const char*  proto_port, int  console_port )
{
    int   s = -1, s2, ret;
    int   maxtries = 10;
    int   flags = 0;
    signal_state_t  sigstate;

    disable_sigalrm( &sigstate );

    if ( !strncmp( proto_port, "tcp:", 4) ) {
        char*  end;
        long   port = strtol(proto_port + 4, &end, 10);

        flags = get_report_console_options( end, &maxtries );

        if (flags & REPORT_CONSOLE_SERVER) {
            s = socket_loopback_server( port, SOCK_STREAM );
            if (s < 0) {
                fprintf(stderr, "could not create server socket on TCP:%ld: %s\n",
                        port, socket_errstr());
                exit(3);
            }
        } else {
            for ( ; maxtries > 0; maxtries-- ) {
                D("trying to find console-report client on tcp:%d", port);
                s = socket_loopback_client( port, SOCK_STREAM );
                if (s >= 0)
                    break;

                sleep_ms(1000);
            }
            if (s < 0) {
                fprintf(stderr, "could not connect to server on TCP:%ld: %s\n",
                        port, socket_errstr());
                exit(3);
            }
        }
    } else if ( !strncmp( proto_port, "unix:", 5) ) {
#ifdef _WIN32
        fprintf(stderr, "sorry, the unix: protocol is not supported on Win32\n");
        exit(3);
#else
        char*  path = strdup(proto_port+5);
        char*  end  = strchr(path, ',');
        if (end != NULL) {
            flags = get_report_console_options( end, &maxtries );
            *end  = 0;
        }
        if (flags & REPORT_CONSOLE_SERVER) {
            s = socket_unix_server( path, SOCK_STREAM );
            if (s < 0) {
                fprintf(stderr, "could not bind unix socket on '%s': %s\n",
                        proto_port+5, socket_errstr());
                exit(3);
            }
        } else {
            for ( ; maxtries > 0; maxtries-- ) {
                s = socket_unix_client( path, SOCK_STREAM );
                if (s >= 0)
                    break;

                sleep_ms(1000);
            }
            if (s < 0) {
                fprintf(stderr, "could not connect to unix socket on '%s': %s\n",
                        path, socket_errstr());
                exit(3);
            }
        }
        free(path);
#endif
    } else {
        fprintf(stderr, "-report-console must be followed by a 'tcp:<port>' or 'unix:<path>'\n");
        exit(3);
    }

    if (flags & REPORT_CONSOLE_SERVER) {
        int  tries = 3;
        D( "waiting for console-reporting client" );
        do {
            s2 = accept( s, NULL, NULL );
        } while (s2 < 0 && socket_errno == EINTR && --tries > 0);

        if (s2 < 0) {
            fprintf(stderr, "could not accept console-reporting client connection: %s\n",
                   socket_errstr());
            exit(3);
        }

        socket_close(s);
        s = s2;
    }

    /* simply send the console port in text */
    {
        char  temp[12];
        snprintf( temp, sizeof(temp), "%d", console_port );
        do {
            ret = send( s, temp, strlen(temp), 0 );
        } while (ret < 0 && socket_errno == EINTR);

        if (ret < 0) {
            fprintf(stderr, "could not send console number report: %d: %s\n",
                    socket_errno, socket_errstr() );
            exit(3);
        }
        socket_close(s);
    }
    D( "console port number sent to remote. resuming boot" );

    restore_sigalrm (&sigstate);
}


#ifdef _WIN32
#undef main  /* we don't want SDL to define main */
#endif

int main(int argc, char **argv)
{
    char   tmp[MAX_PATH];
    char*  tmpend = tmp + sizeof(tmp);
    char*  args[128];
    int    n;
    char*  opt;
    int    use_sdcard_img = 0;
    int    serial = 0;
    int    gps_serial = 0;
    int    radio_serial = 0;
    int    qemud_serial = 0;
    int    shell_serial = 0;
    int    dns_count = 0;

    const char *appdir = get_app_dir();
    char*       android_build_root = NULL;
    char*       android_build_out  = NULL;

    args[0] = argv[0];
    argc--;
    argv++;

    while(argc-- > 0){
        opt = *argv++;

        if(!strcmp(opt, "-qemu")) {
            break;
        }

        if (!strcmp(opt, "-help")) {
            emulator_help();
        }

        /* for backwards compatibility with previous versions */
        if (!strcmp(opt, "-verbose")) {
            opt = "-debug-init";
        }

        if (!strcmp(opt, "-debug")) {
            if (argc == 0) {
                derror( "-debug must be followed by tags (see -help-verbose)\n");
                exit(1);
            }
            argc--;
            parse_verbose_tags(*argv++);
            continue;
        }

        if (!strncmp(opt, "-debug-", 7)) {
            int            remove = 0;
            unsigned long  mask   = 0;
            opt += 7;
            if (!strncmp(opt, "no-", 3)) {
                opt   += 3;
                remove = 1;
            }
            if (!strcmp(opt, "all")) {
                mask = ~0;
            }
            for (n = 0; verbose_options[n].name; n++) {
                if (!strcmp(opt, verbose_options[n].name)) {
                    mask = (1 << verbose_options[n].flag);
                    break;
                }
            }
            if (remove)
                android_verbose &= ~mask;
            else
                android_verbose |= mask;
            continue;
        }

        if (!strncmp(opt, "-help-",6)) {
            opt += 6;
            for (n = 0; argmap[n].name; n++) {
                if (!strcmp(opt, argmap[n].name)) {
                    printf("\n");
                    if (argmap[n].help_func != NULL)
                        argmap[n].help_func();
                    else {
                        printf( "    -%s %s %s\n\n", argmap[n].name, argmap[n].argname, argmap[n].help );
                    }
                    exit(0);
                }
            }

            for (n = 0; help_topics[n].name; n++) {
                if (!strcmp(opt, help_topics[n].name)) {
                    printf("\n");
                    help_topics[n].help_func();
                    exit(0);
                }
            }

            if (!strcmp(opt, "all")) {
                for (n = 0; argmap[n].name; n++) {
                    printf( "========= help for option -%s:\n\n", argmap[n].name );
                    if (argmap[n].help_func != NULL)
                        argmap[n].help_func();
                    else {
                        printf( "    -%s %s %s\n\n", argmap[n].name, argmap[n].argname, argmap[n].help );
                    }
                }
                for (n = 0; help_topics[n].name; n++) {
                    printf( "========= help for -help-%s\n\n", help_topics[n].name );
                    help_topics[n].help_func();
                }
                printf( "========= top-level help\n\n" );
                emulator_help();
            }

            fprintf(stderr, "unknown option: -%s\n", opt);
            emulator_help();
        }

        if (opt[0] != '-') {
            fprintf(stderr, "unknown option: %s\n", opt);
            emulator_help();
        }

        for(n = 0; argmap[n].name; n++) {
            if(!strcmp(opt+1, argmap[n].name)) {
                if(argmap[n].flag) {
                    *argmap[n].flag = 1;
                } else {
					int s;
                    
					if(argc == 0) goto no_arg;
                    *argmap[n].value = *argv++;
                    argc--;
				
					for(s = 0; argsubmap[s].name; s++)	{
						if(!strcmp(opt+1, argsubmap[s].name))	{
							if(argc)	{
								printf("%s - %s\n", argsubmap[s].name, *argv);
								*argsubmap[s].sub_value = *argv++;
								argc--;
							}
							break;
						}
					}
                }
                break;
            }
        }
        if(argmap[n].name == 0) {
            fprintf(stderr,"unknown option '%s'\n", opt);
            emulator_help();
        }
    }

    android_charmap = android_charmaps[0];

    if (arg_version) {
        printf("Android emulator version %s\n(C) Copyright 2006-2008 Google, Inc and many others.\n"
               "this program is a derivative of the QEMU CPU emulator (www.qemu.org)\n\n",
#if defined ANDROID_BUILD_ID
               VERSION_STRING " (build_id " STRINGIFY(ANDROID_BUILD_ID) ")" );
#else
               VERSION_STRING);
#endif
        printf("  This software is licensed under the terms of the GNU General Public\n"
               "  License version 2, as published by the Free Software Foundation, and\n"
               "  may be copied, distributed, and modified under those terms.\n\n"
               "  This program is distributed in the hope that it will be useful,\n"
               "  but WITHOUT ANY WARRANTY; without even the implied warranty of\n"
               "  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n"
               "  GNU General Public License for more details.\n\n");

        exit(0);
    }

    parse_env_verbose();

    if (arg_timezone) {
        if ( timezone_set(arg_timezone) < 0 ) {
            fprintf(stderr, "emulator: it seems the timezone '%s' is not in zoneinfo format\n", arg_timezone);
        }
    }

    if(arg_nopoll) qemu_milli_needed = 0;

    /* try to find the top of the Android build tree */
    do {
        char*  out = getenv("ANDROID_PRODUCT_OUT");

        if (out == NULL || out[0] == 0)
            break;

        if (access(out, F_OK) != 0) {
            dprint("### Weird, can't access ANDROID_PRODUCT_OUT as '%s'", out);
            break;
        }

        android_build_root = path_parent( out, 4 );
        if (android_build_root == NULL) {
            break;
        }
        if (access(android_build_root, F_OK) != 0) {
            dprint("### Weird, can't access build root as '%s'", android_build_root);
            free(android_build_root);
            android_build_root = NULL;
            break;
        }
        android_build_out = out;
        D( "found Android build root: %s", android_build_root );
        D( "found Android build out:  %s", android_build_out );
    } while (0);

        /* if no arg_sysdir was specified, try to find one */
    if (arg_sysdir == NULL) {
        for (n = 0; sysdir_paths[n]; n++) {
            sprintf(tmp, sysdir_paths[n], appdir, "system.img");
            if (access(tmp, F_OK) == 0) {
                sprintf(tmp, sysdir_paths[n], appdir, "");
                arg_sysdir = strdup(tmp);
                break;
            }
        }

        if (arg_sysdir == NULL && android_build_out) {
            sprintf(tmp, "%s/%s", android_build_out, "system.img");
            if (access(tmp, F_OK) == 0)
                arg_sysdir = android_build_out;
        }

        if (arg_sysdir == NULL) {
            fprintf(stderr,
                "Cannot locate system directory, which "
                                        "contains 'system.img' and other\n"
                "system files.  Please specify one by "
                                      "using '-system <directory>' or by\n"
                "defining the environment variable ANDROID_PRODUCT_OUT.\n");
            exit(1);
        }

        D("autoconfig: -system %s", arg_sysdir);
    }

    if (arg_datadir == NULL) {
        if (android_build_out)
            arg_datadir = android_build_out;
        else {
            bufprint_config_path( tmp, tmpend );
            arg_datadir = strdup(tmp);
        }
        D("autoconfig: -datadir %s", arg_datadir);
    }

    sprintf(tmp, "%s%s.", arg_datadir, PATH_SEP);
    if (access(tmp, W_OK) != 0) {
        if (mkdir(arg_datadir, 0755) != 0) {
            fprintf(stderr,
                "Cannot create data directory: %s\n"
                "Please specify a writable directory with -datadir.\n", arg_datadir);
            exit(1);
        }
    }

    /* try to find the qemu kernel in the system directory,
     * otherwise, try to get it in the prebuilt directory */
    optional_file(&arg_kernel, arg_sysdir, "kernel-qemu");
    if (!arg_kernel && android_build_root) {
        sprintf(tmp, "%s/prebuilt/android-arm", android_build_root);
        optional_file(&arg_kernel, tmp, "kernel-qemu");
    }

    /* similar hack for emulator skins */
    if (!arg_noskin && arg_skindir == NULL) {
        if (android_build_root) {
            sprintf(tmp, "%s/prebuilt/common", android_build_root);
            optional_file(&arg_skindir, tmp, "emulator-skins");
        }
    }

    require_file(&arg_kernel, arg_sysdir, "kernel-qemu");
    require_file(&arg_ramdisk, arg_sysdir, "ramdisk.img");
    require_file(&arg_image, arg_sysdir, "system.img");
    require_file(&arg_initdata, arg_sysdir, "userdata.img");

    if (!arg_skindir)
        optional_file(&arg_skindir, arg_sysdir, "skins");

    optional_file(&arg_sdcard, arg_datadir, "sdcard.img");

    if (arg_data == NULL) {
        if (strcmp(arg_datadir, arg_sysdir) != 0) {
          /* in the past, the writable image was called "userdata.img" */
          optional_file(&arg_data, arg_datadir, "userdata.img");
        }
        default_file(&arg_data, arg_datadir, "userdata-qemu.img");

        if ( access(arg_data, F_OK) != 0) {
            /* if the file does not exist, imply a -wipe-data */
            arg_wipedata = 1;
        }
    } else if ( access(arg_data, F_OK) == ENOENT ) {
        /* if -data is used with a non-existent data file */
        arg_wipedata = 1;
    }

    {
        FileLock*  data_lock = filelock_create(arg_data);
        if (data_lock == NULL) {
            fprintf(stderr, "Cannot create /data image file lock\n");
            exit(2);
        }
        /* if the file is already used, use a temporary file instead */
        if (filelock_lock(data_lock) < 0) {
            TempFile*  data_tmp;
            fprintf(stderr,
                    "### WARNING: Another emulator is running with our data file\n");

            data_tmp = tempfile_create();
            if (data_tmp == NULL) {
                fprintf(stderr, "annot create a new temporary user data file.\n" );
                exit(2);
            }
            fprintf(stderr, "### WARNING: User data changes will NOT be saved!\n");
            arg_data     = (char*) tempfile_path(data_tmp);
            arg_wipedata = 1;
        }
    }


    /* wipe the data file if necessary */
    if (arg_wipedata || access(arg_data, F_OK) != 0) {
        if (copy_file(arg_data, arg_initdata) >= 0) {
            D("copied file '%s' from '%s'", arg_data, arg_initdata);
        } else {
            fprintf(stderr,
                "### WARNING: Cannot write user data file '%s': %s\n",
                arg_data, strerror(errno));
            exit(3);
        }
    }

    /* lock the SD card image file */
    if (arg_sdcard != NULL) {
        FileLock*  sdcard_lock = filelock_create( arg_sdcard );
        int        free_sdcard = (sdcard_lock == NULL);

        if (sdcard_lock && filelock_lock(sdcard_lock) < 0) {
            fprintf( stderr, "#### WARNING: SD Card emulation disabled. '%s' already in use\n", arg_sdcard );
            free_sdcard = 1;
        }

        if (free_sdcard) {
            arg_sdcard = NULL;
        }
    }

    parse_rotation(arg_rotation);

    if (arg_keyset) {
        parse_keyset(arg_keyset);
        if (!android_keyset) {
            fprintf(stderr,
                    "emulator: WARNING: could not find keyset file named '%s', using defaults instead\n",
                    arg_keyset);
        }
    }
    if (!android_keyset) {
        android_keyset = skin_keyset_new_from_text( skin_keyset_get_default() );
        if (!android_keyset) {
            fprintf(stderr, "PANIC: default keyset file is corrupted !!\n" );
            fprintf(stderr, "PANIC: please update the code in skins/skin_keyset.c\n" );
            exit(1);
        }
        if (!arg_keyset)
            write_default_keyset();
    }

    if (arg_audio) {
        if (arg_audio_in || arg_audio_out) {
            derror( "you can't use -audio with -audio-in or -audio-out\n" );
            exit(1);
        }
        if ( !audio_check_backend_name( 0, arg_audio ) ) {
            derror( "'%s' is not a valid audio output backend. see -help-audio-out\n",
                    arg_audio);
            exit(1);
        }
        arg_audio_out = arg_audio;
        arg_audio_in  = arg_audio;

        if ( !audio_check_backend_name( 1, arg_audio ) ) {
            fprintf(stderr,
                    "emulator: warning: '%s' is not a valid audio input backend. audio record disabled\n",
                    arg_audio);
            arg_audio_in = "none";
        }
    }

    if (arg_audio_in) {
        static char  env[64]; /* note: putenv needs a static unique string buffer */
        if ( !audio_check_backend_name( 1, arg_audio_in ) ) {
            derror( "'%s' is not a valid audio input backend. see -help-audio-in\n",
                    arg_audio_in);
            exit(1);
        }
        bufprint( env, env+sizeof(env), "QEMU_AUDIO_IN_DRV=%s", arg_audio_in );
        putenv( env );
    }
    if (arg_audio_out) {
        static char  env[64]; /* note: putenv needs a static unique string buffer */
        if ( !audio_check_backend_name( 0, arg_audio_out ) ) {
            derror( "'%s' is not a valid audio output backend. see -help-audio-out\n",
                    arg_audio_out);
            exit(1);
        }
        bufprint( env, env+sizeof(env), "QEMU_AUDIO_OUT_DRV=%s", arg_audio_out );
        putenv( env );
    }

    if (arg_noskin) {
        arg_skin    = "320x240";
        arg_skindir = NULL;
    } else if (arg_skin == NULL) {
        /* select default skin based on product type */
        const char* env = getenv("ANDROID_PRODUCT_OUT");
        if (env) {
            const char*  p = strrchr(env,'/');
            if (p) {
                if (p[1] == 's') {
                    arg_skin = "QVGA-L";
                } else if (p[1] == 'd') {
                    arg_skin = "HVGA";
                }
            }
        }
        if (arg_skin) {
            D("autoconfig: -skin %s", arg_skin);
        }
    }

    if (arg_cpu_delay) {
        char*   end;
        long    delay = strtol(arg_cpu_delay, &end, 0);
        if (end == NULL || *end || delay < 0 || delay > 1000 ) {
            fprintf(stderr, "option -cpu-delay must be an integer between 0 and 1000\n" );
            exit(1);
        }
        if (delay > 0)
            delay = (1000-delay);

        qemu_cpu_delay = (int) delay;
    }

    emulator_config_init();
    init_skinned_ui(arg_skindir, arg_skin);

    if (!arg_netspeed) {
        if (skin_network_speed)
            D("skin network speed: '%s'", skin_network_speed);
        arg_netspeed = (char*)skin_network_speed;
    }
    if (!arg_netdelay) {
        if (skin_network_delay)
            D("skin network delay: '%s'", skin_network_delay);
        arg_netdelay = (char*)skin_network_delay;
    }

    if ( android_parse_network_speed(arg_netspeed) < 0 ) {
        fprintf(stderr, "invalid -netspeed parameter '%s', see emulator -usage\n", arg_netspeed);
        emulator_help();
    }

    if ( android_parse_network_latency(arg_netdelay) < 0 ) {
        fprintf(stderr, "invalid -netdelay parameter '%s', see emulator -usage\n", arg_netdelay);
        emulator_help();
    }

    if (arg_netfast) {
        qemu_net_download_speed = 0;
        qemu_net_upload_speed = 0;
        qemu_net_min_latency = 0;
        qemu_net_max_latency = 0;
    }

    if (arg_tracefile) {
        struct stat  st;
        int          ret;
        sprintf(tmp, "%s/traces", arg_sysdir);
        do { ret = stat(tmp, &st); } while (ret == -1 && errno == EINTR);
        if (ret == -1 && errno == ENOENT) {
            /* create directory */
            do { ret = mkdir(tmp, 0755); } while (ret == -1 && errno == EINTR);
            if (ret == -1) {
                fprintf(stderr, "could not create directory '%s'\n", tmp);
                exit(2);
            }
        }
        sprintf(tmp, "%s/traces/%s", arg_sysdir, arg_tracefile);
        arg_tracefile = strdup(tmp);
    }

    if (arg_nocache)
        arg_cache = 0;

    if (arg_cache) {
        FileLock*  cache_lock = filelock_create(arg_cache);
        if (cache_lock == NULL) {
            fprintf(stderr, "Could not create cache image file lock\n" );
            exit(2);
        }
        if ( filelock_lock( cache_lock ) < 0 ) {
            fprintf(stderr, "### WARNING: Another emulator instance is using our cache file. using temp file\n");
            arg_cache = NULL;
        }
        else if ( access(arg_cache, F_OK) < 0 && errno == ENOENT ) {
            /* -cache <file> where <file> does not exit, we simply */
            /* create an empty file then                           */
            if ( make_empty_file( arg_cache ) < 0 ) {
                fprintf(stderr, "could not create cache file '%s'\n", arg_cache);
                exit(2);
            }
            D( "created non-existent cache image file: %s\n", arg_cache );
        }
    }

    if (arg_dns_server) {
        char*  x = strchr(arg_dns_server, ',');
        dns_count = 0;
        if (x == NULL)
        {
            if ( add_dns_server( arg_dns_server ) == 0 )
                dns_count = 1;
        }
        else
        {
            x = strdup(arg_dns_server);
            while (*x) {
                char*  y = strchr(x, ',');

                if (y != NULL)
                    *y = 0;

                if (y == NULL || y > x) {
                    if ( add_dns_server( x ) == 0 )
                        dns_count += 1;
                }

                if (y == NULL)
                    break;

                x = y+1;
            }
        }
        if (dns_count == 0)
            fprintf( stderr, "### WARNING: will use system default DNS server\n" );
    }

    if (dns_count == 0)
        dns_count = slirp_get_system_dns_servers();

    n = 1;
    /* generate arguments for the underlying qemu main() */
	if(arg_os_type)	{
		args[n++] = "-os-type";
		args[n++] = arg_os_type;
	}
    if(arg_kernel) {
        args[n++] = "-kernel";
        args[n++] = arg_kernel;
    }
	if(arg_guest0_load_addr)	{
		int s;

		args[n++] = "-guest0";
		args[n++] = arg_guest0_load_addr;

		for(s = 0; argsubmap[s].name; s++)	{
			if(!strcmp("guest0", argsubmap[s].name))	{
				args[n++] = *argsubmap[s].sub_value;
				break;
			}
		}
	}
	if(arg_guest1_load_addr)	{
		int s;

		args[n++] = "-guest1";
		args[n++] = arg_guest1_load_addr;

		for(s = 0; argsubmap[s].name; s++)	{
			if(!strcmp("guest1", argsubmap[s].name))	{
				args[n++] = *argsubmap[s].sub_value;
				break;
			}
		}
	}				
    if(arg_ramdisk) {
        args[n++] = "-initrd";
        args[n++] = arg_ramdisk;
    }
    if(arg_image) {
        sprintf(tmp, "system,size=0x4200000,initfile=%s", arg_image);
        args[n++] = "-nand";
        args[n++] = strdup(tmp);
    }
    if(arg_data) {
        sprintf(tmp, "userdata,size=0x4200000,file=%s", arg_data);
        args[n++] = "-nand";
        args[n++] = strdup(tmp);
    }
    if (arg_cache) {
        sprintf(tmp, "cache,size=0x4200000,file=%s", arg_cache);
        args[n++] = "-nand";
        args[n++] = strdup(tmp);
    } else if(!arg_nocache) {
        sprintf(tmp, "cache,size=0x4200000");
        args[n++] = "-nand";
        args[n++] = strdup(tmp);
    }
    if(arg_sdcard) {
        struct stat statbuf;
        /* see if we have an sdcard image.  get its size if it exists */
        if (stat(arg_sdcard, &statbuf) == 0) {
            if (statbuf.st_size < 8*1024*1024) {
                fprintf(stderr, "### WARNING: SD Card files must be at least 8 MB, ignoring '%s'\n", arg_sdcard);
            } else {
                args[n++] = "hda";
                args[n++] = arg_sdcard;
                use_sdcard_img = 1;
            }
        } else {
            dprint("could not find or access sdcard image at '%s'", arg_sdcard);
        }
    }

    if (!arg_logcat || arg_logcat[0] == 0) {
        arg_logcat = getenv("ANDROID_LOG_TAGS");
        if (arg_logcat && arg_logcat[0] == 0)
            arg_logcat = NULL;
    }

    if (arg_console) {
        derror( "option -console is obsolete. please use -shell instead" );
        exit(1);
    }

    /* we always send the kernel messages from ttyS0 to android_kmsg */
    {
        AndroidKmsgFlags  flags = 0;

        if (arg_show_kernel)
            flags |= ANDROID_KMSG_PRINT_MESSAGES;

        android_kmsg_init( flags );
        args[n++] = "-serial";
        args[n++] = "android-kmsg";
        serial++;
    }
    /* XXXX: TODO: implement -shell and -logcat through qemud instead */
    if (!arg_shell_serial)
        arg_shell_serial = "stdio";
    else
        arg_shell = 1;

    if (arg_shell || arg_logcat) {
        args[n++] = "-serial";
        args[n++] = arg_shell_serial;
        shell_serial = serial++;
    }

    if (arg_old_system)
    {
        if (arg_radio) {
            args[n++] = "-serial";
            args[n++] = arg_radio;
            radio_serial = serial++;
        }
        else {
            args[n++] = "-serial";
            args[n++] = "android-modem";
            radio_serial = serial++;
        }
        if (arg_gps) {
            args[n++] = "-serial";
            args[n++] = arg_gps;
            gps_serial = serial++;
        }
    }
    else /* !arg_old_system */
    {
        args[n++] = "-serial";
        args[n++] = "android-qemud";
        qemud_serial = serial++;

        if (arg_radio) {
            CharDriverState*  cs = qemu_chr_open(arg_radio);
            if (cs == NULL) {
                derror( "unsupported character device specification: %s\n"
                        "used -help-char-devices for list of available formats\n", arg_radio );
                exit(1);
            }
            android_qemud_set_channel( ANDROID_QEMUD_GSM, cs);
        }
        else {
            if ( android_qemud_get_channel( ANDROID_QEMUD_GSM, &android_modem_cs ) < 0 ) {
                derror( "could not initialize qemud 'gsm' channel" );
                exit(1);
            }
        }

        if (arg_gps) {
            CharDriverState*  cs = qemu_chr_open(arg_gps);
            if (cs == NULL) {
                derror( "unsupported character device specification: %s\n"
                        "used -help-char-devices for list of available formats\n", arg_gps );
                exit(1);
            }
            android_qemud_set_channel( ANDROID_QEMUD_GPS, cs);
        }
        else {
            if ( android_qemud_get_channel( "gps", &android_gps_cs ) < 0 ) {
                derror( "could not initialize qemud 'gps' channel" );
                exit(1);
            }
        }
    }

    if (arg_noaudio) {
        args[n++] = "-noaudio";
    }

    if (arg_mic) {
        struct stat statbuf;
        /* see if we have an sdcard image.  get its size if it exists */
        if (stat(arg_mic, &statbuf) == 0) {
            args[n++] = "-mic";
            args[n++] = arg_mic;
        } else {
            dprint("could not find or access audio input at '%s'", arg_mic);
        }
    }

    if (arg_tracefile) {
        args[n++] = "-trace";
        args[n++] = arg_tracefile;
        args[n++] = "-tracing";
        args[n++] = "off";
    }

    args[n++] = "-append";

    {
        static char  params[1024];
        char        *p = params, *end = p + sizeof(params);

        p = bufprint(p, end, "qemu=1 console=ttyS0" );

        if (arg_shell || arg_logcat) {
            p = bufprint(p, end, " androidboot.console=ttyS%d", shell_serial );
        }

        if (arg_tracefile) {
            p = bufprint(p, end, " android.tracing=1");
        }

        if (!arg_nojni) {
            p = bufprint(p, end, " android.checkjni=1");
        }

        if (arg_no_boot_anim) {
            p = bufprint( p, end, " android.bootanim=0" );
        }

        if (arg_logcat) {
            char*  q = bufprint(p, end, " androidboot.logcat=%s", arg_logcat);

            if (q < end) {
                /* replace any space by a comma ! */
                {
                    int  nn;
                    for (nn = 1; p[nn] != 0; nn++)
                        if (p[nn] == ' ' || p[nn] == '\t')
                            p[nn] = ',';
                    p += nn;
                }
            }
            p = q;
        }

        if (arg_old_system)
        {
            p = bufprint(p, end, " android.ril=ttyS%d", radio_serial);

            if (arg_gps) {
                p = bufprint(p, end, " android.gps=ttyS%d", gps_serial);
            }
        }
        else
        {
            p = bufprint(p, end, " android.qemud=ttyS%d", qemud_serial);
        }

        if (dns_count > 0) {
            p = bufprint(p, end, " android.ndns=%d", dns_count);
        }

        if (p >= end) {
            fprintf(stderr, "### ERROR: kernel parameters too long\n");
            exit(1);
        }

        args[n++] = strdup(params);
    }

    while(argc-- > 0) {
        args[n++] = *argv++;
    }
    args[n] = 0;

    if(VERBOSE_CHECK(init)) {
        int i;
        for(i = 0; i < n; i++) {
            fprintf(stdout, "emulator: argv[%02d] = \"%s\"\n", i, args[i]);
        }
    }
    return qemu_main(n, args);

no_arg:
    fprintf(stderr,"option %s requires an argument\n", opt);
    return -1;
}

/* this function is called from qemu_main() once all arguments have been parsed
 * it should be used to setup any Android-specific items in the emulation before the
 * main loop runs
 */
void  android_emulation_setup( void )
{
    int   tries     = 16;
    int   base_port = 5554;
    int   success   = 0;
    int   s;
    struct in_addr guest_addr;

    inet_aton("10.0.2.15", &guest_addr);

    if (arg_adb_port) {
        fprintf( stderr, "option -adb-port is obsolete, use -port instead\n" );
        exit(1);
    }

    if (arg_port) {
        char*  end;
        int    port = strtol( arg_port, &end, 0 );
        if ( end == NULL || *end ||
             (unsigned)((port - base_port) >> 1) >= (unsigned)tries )
        {
            derror( "option -port must be followed by an even integer number between %d and %d\n",
                     base_port, base_port + (tries-1)*2 );
            exit(1);
        }
        if ( (port & 1) != 0 ) {
            port &= ~1;
            dwarning( "option -port must be followed by an even integer, using  port number %d\n",
                       port );
        }
        base_port = port;
        tries     = 1;
    }

    for ( ; tries > 0; tries--, base_port += 2 ) {

        /* setup first redirection for ADB, the Android Debug Bridge */
        if ( slirp_redir( 0, base_port+1, guest_addr, 5555 ) < 0 )
            continue;

        /* setup second redirection for the emulator console */
        if ( control_console_start( base_port ) < 0 ) {
            slirp_unredir( 0, base_port+1 );
            continue;
        }

        D( "control console listening on port %d, ADB on port %d", base_port, base_port+1 );
        success = 1;
        break;
    }

    if (!success) {
        fprintf(stderr, "it seems too many emulator instances are running on this machine. Aborting\n" );
        exit(1);
    }

    if (arg_report_console) {
        report_console(arg_report_console, base_port);
    }

    android_modem_init( base_port );

    android_base_port = base_port;

   /* send a simple message to the ADB host server to tell it we just started.
    * it should be listening on port 5037. if we can't reach it, don't bother
    */
    do
    {
        struct sockaddr_in  addr;
        char                tmp[32];
        int                 ret;

        s = socket(PF_INET, SOCK_STREAM, 0);
        if (s < 0) {
            D("can't create socket to talk to the ADB server");
            break;
        }

        memset(&addr, 0, sizeof(addr));
        addr.sin_family      = AF_INET;
        addr.sin_port        = htons(5037);
        addr.sin_addr.s_addr = htonl(INADDR_LOOPBACK);

        do {
            ret = connect(s, (struct sockaddr*)&addr, sizeof(addr));
        } while (ret < 0 && socket_errno == EINTR);

        if (ret < 0) {
            D("can't connect to ADB server: errno %d: %s", socket_errno, socket_errstr() );
            break;
        }

        sprintf(tmp,"0012host:emulator:%d",base_port+1);

        while ( send(s, tmp, 18+4, 0) < 0 && errno == EINTR ) {
        };
        D("sent '%s' to ADB server", tmp);
    }
    while (0);
    if (s >= 0)
        close(s);

    /* setup the http proxy, if any */
    if (VERBOSE_CHECK(proxy))
        proxy_set_verbose(1);

    if (!arg_http_proxy) {
        arg_http_proxy = getenv("http_proxy");
    }

    do
    {
        const char*  env = arg_http_proxy;
        int          envlen;
        ProxyOption  option_tab[4];
        ProxyOption* option = option_tab;
        char*        p;
        char*        q;
        const char*  proxy_name;
        int          proxy_name_len;
        int          proxy_port;

        if (!env)
            break;

        envlen = strlen(env);

        /* skip the 'http://' header, if present */
        if (envlen >= 7 && !memcmp(env, "http://", 7)) {
            env    += 7;
            envlen -= 7;
        }

        /* do we have a username:password pair ? */
        p = strchr(env, '@');
        if (p != 0) {
            q = strchr(env, ':');
            if (q == NULL) {
            BadHttpProxyFormat:
                dprint("http_proxy format unsupported, try 'proxy:port' or 'username:password@proxy:port'");
                break;
            }

            option->type       = PROXY_OPTION_AUTH_USERNAME;
            option->string     = env;
            option->string_len = q - env;
            option++;

            option->type       = PROXY_OPTION_AUTH_PASSWORD;
            option->string     = q+1;
            option->string_len = p - (q+1);
            option++;

            env = p+1;
        }

        p = strchr(env,':');
        if (p == NULL)
            goto BadHttpProxyFormat;

        proxy_name     = env;
        proxy_name_len = p - env;
        proxy_port     = atoi(p+1);

        D( "setting up http proxy:  server=%.*s port=%d",
                proxy_name_len, proxy_name, proxy_port );

        if ( proxy_http_setup( proxy_name, proxy_name_len, proxy_port,
                               option - option_tab, option_tab ) < 0 )
        {
            dprint( "http proxy setup failed, check your $http_proxy variable");
        }
    }
    while (0);

   /* cool, now try to run the "ddms ping" command, which will take care of pinging usage
    * if the user agreed for it. the emulator itself never sends anything to any outside
    * machine
    */
    {
#ifdef _WIN32
#  define  _ANDROID_PING_PROGRAM   "ddms.bat"
#else
#  define  _ANDROID_PING_PROGRAM   "ddms"
#endif

        char         tmp[PATH_MAX];
        const char*  appdir = get_app_dir();

        if (snprintf( tmp, PATH_MAX, "%s%s%s", appdir, PATH_SEP,
                      _ANDROID_PING_PROGRAM ) >= PATH_MAX) {
            dprint( "Application directory too long: %s", appdir);
            return;
        }

        /* if the program isn't there, don't bother */
        D( "ping program: %s", tmp);
        if (access( tmp, F_OK ) == 0) {
#ifdef _WIN32
            STARTUPINFO           startup;
            PROCESS_INFORMATION   pinfo;

            ZeroMemory( &startup, sizeof(startup) );
            startup.cb = sizeof(startup);
            startup.dwFlags = STARTF_USESHOWWINDOW;
            startup.wShowWindow = SW_SHOWMINIMIZED;

            ZeroMemory( &pinfo, sizeof(pinfo) );

            char* comspec = getenv("COMSPEC");
            if (!comspec) comspec = "cmd.exe";

            // Run
            char args[PATH_MAX + 30];
            if (snprintf( args, PATH_MAX, "/C \"%s\" ping emulator " VERSION_STRING,
                          tmp) >= PATH_MAX ) {
                D( "DDMS path too long: %s", tmp);
                return;
            }

            CreateProcess(
                comspec,                                      /* program path */
                args,                                    /* command line args */
                NULL,                    /* process handle is not inheritable */
                NULL,                     /* thread handle is not inheritable */
                FALSE,                       /* no, don't inherit any handles */
                DETACHED_PROCESS,   /* the new process doesn't have a console */
                NULL,                       /* use parent's environment block */
                NULL,                      /* use parent's starting directory */
                &startup,                   /* startup info, i.e. std handles */
                &pinfo );

            D( "ping command: %s %s", comspec, args );
#else
            int  pid = fork();
            if (pid == 0) {
                int  fd = open("/dev/null", O_WRONLY);
                dup2(fd, 1);
                dup2(fd, 2);
                execl( tmp, _ANDROID_PING_PROGRAM, "ping", "emulator", VERSION_STRING, NULL );
            }
            /* don't do anything in the parent or in case of error */
            strncat( tmp, " ping emulator " VERSION_STRING, PATH_MAX - strlen(tmp) );
            D( "ping command: %s", tmp );
#endif
        }
    }
}


void  android_emulation_teardown( void )
{
}
