/* Copyright (C) 2007-2008 Google, Inc.
**
** This software is licensed under the terms of the GNU General Public
** License version 2, as published by the Free Software Foundation, and
** may be copied, distributed, and modified under those terms.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
*/
#include "vl.h"
#include "arm_pic.h"
#include "platform_device.h"

#define PDEV_BUS_OP_DONE        (0x00)
#define PDEV_BUS_OP_REMOVE_DEV  (0x04)
#define PDEV_BUS_OP_ADD_DEV     (0x08)

#define PDEV_BUS_OP_INIT        (0x00)

#define PDEV_BUS_OP             (0x00)
#define PDEV_BUS_GET_NAME       (0x04)
#define PDEV_BUS_NAME_LEN       (0x08)
#define PDEV_BUS_ID             (0x0c)
#define PDEV_BUS_IO_BASE        (0x10)
#define PDEV_BUS_IO_SIZE        (0x14)
#define PDEV_BUS_IRQ            (0x18)
#define PDEV_BUS_IRQ_COUNT      (0x1c)

struct bus_state {
    struct platform_device dev;
    struct platform_device *current;
};

void *platform_pic;
static struct platform_device *first_device;
static struct platform_device *last_device;
uint32_t platform_free_base;
uint32_t platform_free_irq;

void platform_device_set_irq(struct platform_device *dev, int irq, int level)
{
    if(irq >= dev->irq_count)
        cpu_abort (cpu_single_env, "platform_device_set_irq: Bad irq %d >= %d\n", irq, dev->irq_count);
    else
        pic_set_irq_new(platform_pic, dev->irq + irq, level);
}

int platform_add_device_no_io(struct platform_device *dev)
{
    if(dev->base == 0) {
        dev->base = platform_free_base;
        platform_free_base += dev->size;
    }
    if(dev->irq == 0 && dev->irq_count > 0) {
        dev->irq = platform_free_irq;
        platform_free_irq += dev->irq_count;
    }
    //printf("platform_add_device: %s, base %x %x, irq %d %d\n",
    //       dev->name, dev->base, dev->size, dev->irq, dev->irq_count);
    dev->next = NULL;
    if(last_device) {
        last_device->next = dev;
    }
    else {
        first_device = dev;
    }
    last_device = dev;
    return 0;
}

int platform_device_add(struct platform_device *dev,
                       CPUReadMemoryFunc **mem_read,
                       CPUWriteMemoryFunc **mem_write,
                       void *opaque)
{
    int iomemtype;
    platform_add_device_no_io(dev);
    iomemtype = cpu_register_io_memory(0, mem_read,
                                       mem_write, opaque);
    cpu_register_physical_memory(dev->base, dev->size, iomemtype);
    return 0;
}

static uint32_t platform_bus_read(void *opaque, target_phys_addr_t offset)
{
    struct bus_state *s = (struct bus_state *)opaque;
    offset -= s->dev.base;

    switch (offset) {
        case PDEV_BUS_OP:
            if(s->current) {
                s->current->reported_state = 1;
                s->current = s->current->next;
            }
            else {
                s->current = first_device;
            }
            while(s->current && s->current->reported_state == 1)
                s->current = s->current->next;
            if(s->current)
                return PDEV_BUS_OP_ADD_DEV;
            else {
                platform_device_set_irq(&s->dev, 0, 0);
                return PDEV_BUS_OP_DONE;
            }

        case PDEV_BUS_NAME_LEN:
            return s->current ? strlen(s->current->name) : 0;
        case PDEV_BUS_ID:
            return s->current ? s->current->id : 0;
        case PDEV_BUS_IO_BASE:
            return s->current ? s->current->base : 0;
        case PDEV_BUS_IO_SIZE:
            return s->current ? s->current->size : 0;
        case PDEV_BUS_IRQ:
            return s->current ? s->current->irq : 0;
        case PDEV_BUS_IRQ_COUNT:
            return s->current ? s->current->irq_count : 0;
    default:
        cpu_abort (cpu_single_env, "platform_bus_read: Bad offset %x\n", offset);
        return 0;
    }
}

static void platform_bus_op_init(struct bus_state *s)
{
    struct platform_device *dev = first_device;
    while(dev) {
        dev->reported_state = 0;
        dev = dev->next;
    }
    s->current = NULL;
    platform_device_set_irq(&s->dev, 0, first_device != NULL);
}

static void platform_bus_write(void *opaque, target_phys_addr_t offset, uint32_t value)
{
    struct bus_state *s = (struct bus_state *)opaque;
    offset -= s->dev.base;

    switch(offset) {
        case PDEV_BUS_OP:
            switch(value) {
                case PDEV_BUS_OP_INIT:
                    platform_bus_op_init(s);
                    break;
                default:
                    cpu_abort (cpu_single_env, "platform_bus_write: Bad PDEV_BUS_OP value %x\n", value);
            };
            break;
        case PDEV_BUS_GET_NAME:
            if(s->current)
                pmemcpy(value, s->current->name, strlen(s->current->name));
            break;
        default:
            cpu_abort (cpu_single_env, "platform_bus_write: Bad offset %x\n", offset);
    }
}

static CPUReadMemoryFunc *platform_bus_readfn[] = {
    platform_bus_read,
    platform_bus_read,
    platform_bus_read
};

static CPUWriteMemoryFunc *platform_bus_writefn[] = {
    platform_bus_write,
    platform_bus_write,
    platform_bus_write
};


static struct bus_state bus_state = {
    .dev = {
        .name = "goldfish_device_bus",
        .id = -1,
        .base = 0x10001000,
        .size = 0x1000,
        .irq = 1,
        .irq_count = 1,
    }
};

void platform_device_init(void *pic, uint32_t base, uint32_t size, uint32_t irq, uint32_t irq_count)
{
    platform_pic = pic;
    platform_free_base = base;
    platform_free_irq = irq;
}

int platform_device_bus_init(uint32_t base, uint32_t irq)
{
    bus_state.dev.base = base;
    bus_state.dev.irq = irq;

    return platform_device_add(&bus_state.dev, platform_bus_readfn, platform_bus_writefn, &bus_state);
}

