/* Copyright (C) 2007-2008 Google, Inc.
**
** This software is licensed under the terms of the GNU General Public
** License version 2, as published by the Free Software Foundation, and
** may be copied, distributed, and modified under those terms.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
*/
#include "vl.h"
#include "arm_pic.h"
#include "platform_device.h"

enum {
    INTERRUPT_STATUS        = 0x00, // number of pending interrupts
    INTERRUPT_NUMBER        = 0x04,
    INTERRUPT_DISABLE_ALL   = 0x08,
    INTERRUPT_DISABLE       = 0x0c,
    INTERRUPT_ENABLE        = 0x10
};

struct platform_int_state {
    arm_pic_handler handler;
    struct platform_device dev;
    uint32_t level;
    uint32_t pending_count;
    uint32_t irq_enabled;
    uint32_t fiq_enabled;
    void *parent;
    int parent_irq;
    int parent_fiq;
};

#define  PLATFORM_INT_SAVE_VERSION  1

#define  QFIELD_STRUCT  struct platform_int_state
QFIELD_BEGIN(platform_int_fields)
    QFIELD_INT32(level),
    QFIELD_INT32(pending_count),
    QFIELD_INT32(irq_enabled),
    QFIELD_INT32(fiq_enabled),
QFIELD_END

static void platform_int_save(QEMUFile*  f, void*  opaque)
{
    struct platform_int_state*  s = opaque;

    qemu_put_struct(f, platform_int_fields, s);
}

static int  platform_int_load(QEMUFile*  f, void*  opaque, int  version_id)
{
    struct platform_int_state*  s = opaque;

    if (version_id != PLATFORM_INT_SAVE_VERSION)
        return -1;

    return qemu_get_struct(f, platform_int_fields, s);
}

static void platform_int_update(struct platform_int_state *s)
{
    uint32_t flags;

    if (s->parent_irq != -1) {
        flags = (s->level & s->irq_enabled);
        pic_set_irq_new(s->parent, s->parent_irq, flags != 0);
    }
    if (s->parent_fiq != -1) {
        flags = (s->level & s->fiq_enabled);
        pic_set_irq_new(s->parent, s->parent_fiq, flags != 0);
    }
}

static void platform_int_set_irq(void *opaque, int irq, int level)
{
    struct platform_int_state *s = (struct platform_int_state *)opaque;
    uint32_t mask = (1U << irq);

    if(level) {
        if(!(s->level & mask)) {
            if(s->irq_enabled & mask)
                s->pending_count++;
            s->level |= mask;
        }
    }
    else {
        if(s->level & mask) {
            if(s->irq_enabled & mask)
                s->pending_count--;
            s->level &= ~mask;
        }
    }
    platform_int_update(s);
}

static uint32_t platform_int_read(void *opaque, target_phys_addr_t offset)
{
    struct platform_int_state *s = (struct platform_int_state *)opaque;
    offset -= s->dev.base;

    switch (offset) {
    case INTERRUPT_STATUS: /* IRQ_STATUS */
        return s->pending_count;
    case INTERRUPT_NUMBER: {
        int i;
        uint32_t pending = s->level & s->irq_enabled;
        for(i = 0; i < 32; i++) {
            if(pending & (1U << i))
                return i;
        }
        return 0;
    }
    default:
        cpu_abort (cpu_single_env, "platform_int_read: Bad offset %x\n", offset);
        return 0;
    }
}

static void platform_int_write(void *opaque, target_phys_addr_t offset, uint32_t value)
{
    struct platform_int_state *s = (struct platform_int_state *)opaque;
    uint32_t mask = (1U << value);
    offset -= s->dev.base;

    switch (offset) {
        case INTERRUPT_DISABLE_ALL:
            s->pending_count = 0;
            s->level = 0;
            break;

        case INTERRUPT_DISABLE:
            if(s->irq_enabled & mask) {
                if(s->level & mask)
                    s->pending_count--;
                s->irq_enabled &= ~mask;
            }
            break;
        case INTERRUPT_ENABLE:
            if(!(s->irq_enabled & mask)) {
                s->irq_enabled |= mask;
                if(s->level & mask)
                    s->pending_count++;
            }
            break;

    default:
        cpu_abort (cpu_single_env, "platform_int_write: Bad offset %x\n", offset);
        return;
    }
    platform_int_update(s);
}

static CPUReadMemoryFunc *platform_int_readfn[] = {
    platform_int_read,
    platform_int_read,
    platform_int_read
};

static CPUWriteMemoryFunc *platform_int_writefn[] = {
    platform_int_write,
    platform_int_write,
    platform_int_write
};

void *platform_interrupt_init(uint32_t base, void *parent, int parent_irq, int parent_fiq)
{
    int ret;
    struct platform_int_state *s;

    s = qemu_mallocz(sizeof(*s));
    s->handler = platform_int_set_irq;
    s->dev.name = "goldfish_interrupt_controller";
    s->dev.id = -1;
    s->dev.base = base;
    s->dev.size = 0x1000;
    s->parent = parent;
    s->parent_irq = parent_irq;
    s->parent_fiq = parent_fiq;

    ret = platform_device_add(&s->dev, platform_int_readfn, platform_int_writefn, s);
    if(ret) {
        qemu_free(s);
        return NULL;
    }

    register_savevm( "platform_int", 0, PLATFORM_INT_SAVE_VERSION,
                     platform_int_save, platform_int_load, s);

    return s;
}

