/*
*********************************************************************************************************
*                                                uC/OS-II
*                                          The Real-Time Kernel
*
*                                 (c) Copyright 2004, Micrium, Weston, FL
*                                          All Rights Reserved
*
*                                            Freescale i.MX21ADS
*                                              Sample code
* File : APP.C
* By   : Eric Shufro
*
* Xen-Arm version by OS LAB, Korea University (osvirtual@os.korea.ac.kr)
*********************************************************************************************************
*/


#include <includes.h>

#ifdef CONFIG_XEN
#include <os/os.h>
#include <os/irq.h>
#include <os/traps.h>
#include <os/evtchn.h>
#include <os/time.h>
#include <os/sched.h>
#include <os/lib.h>
#include <os/mm.h> 
#include <os/xmalloc.h>
#include <os/hypervisor.h>
#include <os/gnttab.h>


#define SHARED_PTE_MASK	((0xE << 0) | ( 0xFF << 4))
#define __pte(x) (x)

/* Assembler interface fns in entry.S */
void hypervisor_callback(void);
void failsafe_callback(void);

extern void create_io_mapping(unsigned long virt_addr, unsigned long phys_addr, unsigned long size);

shared_info_t *HYPERVISOR_shared_info;
start_info_t start_info;
unsigned long min_mfn;

extern char shared_info[4096 * 8];

#endif // CONFIG_XEN

/*
*********************************************************************************************************
*                                               CONSTANTS
*********************************************************************************************************
*/

/*
*********************************************************************************************************
*                                          FUNCTION PROTOTYPES
*********************************************************************************************************
*/

static  void  AppTaskStart(void *p_arg);
static  void  AppTaskCreate(void);

#if (uC_PROBE_OS_PLUGIN > 0)
static  void  AppProbeCallback(void);
#endif

#if uC_TCPIP_MODULE > 0
void AppInit_TCPIP();
#endif

/*
*********************************************************************************************************
*                                               VARIABLES
*********************************************************************************************************
*/

static  OS_STK  AppTaskStartStk[APP_TASK_START_STK_SIZE];


/*
*********************************************************************************************************
*                                             C ENTRY POINT
*********************************************************************************************************
*/


shared_info_t *map_shared_info(unsigned long pa)
{
	unsigned int offset;


	if ( (offset = HYPERVISOR_update_va_mapping(
                    (unsigned long)shared_info, __pte(pa | SHARED_PTE_MASK),
				UVMF_SHARED_INFO | UVMF_INVLPG)) < 0) {
		printk("Failed to map shared_info!!\n");
	}

	return (shared_info_t *)((int)shared_info + (offset << PAGE_SHIFT));
}


void show_start_info(start_info_t *si)
{
       printk("sizeof(shared_info_t) : %x\n", sizeof(shared_info_t));
	printk("vcpu_time_info_t : %x vcpu_info_t : %x arch_vcpu_info_t : %x\n"
               ,sizeof(vcpu_time_info_t),sizeof(vcpu_info_t), sizeof(arch_vcpu_info_t));     
       printk("uC/S_II\n");
       printk("Magic : %s\n", si->magic);
	printk("Total Pages allocated to this domain : %ld\n", si->nr_pages);
	printk("MACHINE address of shared info struct : 0x%x\n", si->shared_info);
	printk("VIRTUAL address of page directory : 0x%x\n", si->pt_base);
	printk("Number of bootstrap p.t. frames : %ld\n", si->nr_pt_frames);
	printk("VIRTUAL address of page-frame list : 0x%x\n", si->mfn_list);
	printk("VIRTUAL address of pre-loaded module : 0x%x\n", si->mod_start);
	printk("Size (bytes) of pre-loaded modules : %ld\n", si->mod_len);
	printk("min mfn(min_page in xen) : %ld\n", si->min_mfn);
	printk("Command-Linux Address : 0x%x\n", si->cmd_line);
	printk("Command-Line String : %s\n", si->cmd_line);
	printk("  flags       : 0x%x\n",  (unsigned int)si->flags);

}

/* execute guest domain from dom0 */
void create_guest_domain()
{
        int ret;
        unsigned long domain_id;
        dom0_op_t dom0_op = { 0 };

        dom0_op.cmd = DOM0_CREATEDOMAIN;
        ret = HYPERVISOR_dom0_op(&dom0_op);
        if (ret < 0)
                printk(" guest domain execution failed!! \n");

        domain_id = dom0_op.u.createdomain.domain;

        dom0_op.cmd = DOM0_GUEST_IMAGE_CTL;
        dom0_op.u.guest_image_info.domain = domain_id;
        dom0_op.u.guest_image_info.sub_cmd = CMD_GUEST_CREATE;
        ret = HYPERVISOR_dom0_op(&dom0_op);
        if (ret < 0)
                printk(" guest domain execution failed!! \n");

        dom0_op.cmd = DOM0_UNPAUSEDOMAIN;
        dom0_op.u.unpausedomain.domain = domain_id;
        ret = HYPERVISOR_dom0_op(&dom0_op);
        if (ret < 0)
                printk(" guest domain unpause failed!! \n");
}


int is_current_dom0(void)
{
	if( (unsigned long *)HYPERVISOR_shared_info == (unsigned long *)0xc0009000 )
		return 1;
	else
		return 0;
}

void start_kernel(start_info_t *si)
{
        CPU_INT08U  err;
		
#ifdef CONFIG_XEN

	show_start_info(si);
        
	memcpy(&start_info, (start_info_t *)si, sizeof(start_info_t));

        printk("\n\nsize of shared_info_t : %x\nvcpu_time_info_t : %x\n vcpu_info_t : %x \nunsigned int : %x domid_t : %x unsigned long long : %x\n",  
               sizeof(shared_info_t),sizeof(vcpu_time_info_t), sizeof(vcpu_info_t), sizeof(unsigned int), sizeof(domid_t), sizeof(unsigned long long));

	/*
	 * Mapping shared page
	 */
	HYPERVISOR_shared_info = map_shared_info(start_info.shared_info);
        
	
	/* 
	 * Registering callback handler, hypervisor_callback and failsafe_callback
	 */

	HYPERVISOR_set_callbacks((unsigned long)hypervisor_callback, (unsigned long)failsafe_callback);
        printk("xeno: set hypervisor callbacks at %x\n", hypervisor_callback);

	/* init memory management */
	mm_init();
	
	/* init event channl */
	evtchn_init();

	/* init timer */
	time_init();
	
 #endif // CONFIG_XEN      
        


    	BSP_IntDisAll();                                                    /* Disable ALL interrupts to the interrupt controller       */
        
    	OSInit();                                                           /* Initialize uC/OS-II                                      */
                                                                     /* Create start task    */  
    	OSTaskCreateExt(AppTaskStart,
                    NULL,
                    (OS_STK *)&AppTaskStartStk[APP_TASK_START_STK_SIZE - 1],
                    APP_TASK_START_PRIO,
                    APP_TASK_START_PRIO,
                    (OS_STK *)&AppTaskStartStk[0],
                    APP_TASK_START_STK_SIZE,
                    NULL,
                    OS_TASK_OPT_STK_CHK | OS_TASK_OPT_STK_CLR);

        
        /* Assign names to created tasks                            */
#if OS_TASK_NAME_SIZE > 11
    	OSTaskNameSet(APP_TASK_START_PRIO, "Start Task", (INT8U *)&err);
#endif


	
	if( is_current_dom0() )
       {
		create_guest_domain();
       }

	local_irq_enable();

	   
	OSStart();                                                          /* Start uC/OS-II                                           */

	
}





/*$PAGE*/
/*
*********************************************************************************************************
*                                              STARTUP TASK
*
* Description : This is an example of a startup task.  As mentioned in the book's text, you MUST
*               initialize the ticker only once multitasking has started.
* Arguments   : p_arg is the argument passed to 'AppStartTask()' by 'OSTaskCreate()'.
* Notes       : 1) The first line of code is used to prevent a compiler warning because 'p_arg' is not
*                  used.  The compiler should not generate any code for this statement.
*               2) Interrupts are enabled once the task start because the I-bit of the CCR register was
*                  set to 0 by 'OSTaskCreate()'.
*********************************************************************************************************
*/

static  void  AppTaskStart (void *p_arg)
{
    (void)p_arg;                                                        /* Prevent compiler warning                                 */

    printk("AppTaskStart\n");
    BSP_Init();

#if OS_TASK_STAT_EN > 0
    printk("OSStatInit \n");
    OSStatInit();                                                       /* Start stats task                                         */
#endif

#if (uC_PROBE_OS_PLUGIN > 0)
    OSProbe_Init();
    OSProbe_SetCallback(AppProbeCallback);
    OSProbe_SetDelay(50);
#endif

#if (uC_PROBE_COM_MODULE > 0)
    ProbeCom_Init();                                                    /* Initialize the uC/Probe communications module            */
    ProbeRS232_Init(115200);
    ProbeRS232_RxIntEn();
#endif

#if uC_TCPIP_MODULE > 0
    printk("AppInit_TCPIP\n");
    AppInit_TCPIP();                                                    /* Initialize uC/TCP-IP and associated appliations          */
#endif

    AppTaskCreate();                                                    /* Create additional applicaiton tasks                      */
    
    LED_Off(0);                                                         /* Turn on ALL the LEDs                                     */

    while (DEF_TRUE) {                                                  /* Task body, always written as an infinite loop.           */
       LED_Toggle(2);                                                  /* Toggle LED2, TTCP will toggle LED1 when in test          */
      OSTimeDlyHMSM(0, 0, 0, 500);
    
    }
}

/*
*********************************************************************************************************
*                              CREATE ADDITIONAL APPLICATION TASKS
*********************************************************************************************************
*/

static  void  AppTaskCreate (void)
{
}



/*
*********************************************************************************************************
*                                         uC/Probe Callback
*
* Description : This task is called by the uC/Probe uC/OS-II plug-in after updating task information.
*
* Arguments   : none.
*
* Returns     : none
*********************************************************************************************************
*/

#if (uC_PROBE_OS_PLUGIN > 0)
static  void  AppProbeCallback (void)
{
}
#endif

/*
*********************************************************************************************************
*                                          App_TaskCreateHook()
*
* Description: This function is called when a task is created.
*
* Arguments  : ptcb   is a pointer to the task control block of the task being created.
*
* Note(s)    : 1) Interrupts are disabled during this call.
*********************************************************************************************************
*/

#if (OS_APP_HOOKS_EN > 0)
void  App_TaskCreateHook (OS_TCB *ptcb)
{
#if (uC_PROBE_OS_PLUGIN > 0) && (OS_PROBE_HOOKS_EN > 0)
    OSProbe_TaskCreateHook(ptcb);
#endif
}

/*
*********************************************************************************************************
*                                           App_TaskDelHook()
*
* Description: This function is called when a task is deleted.
*
* Arguments  : ptcb   is a pointer to the task control block of the task being deleted.
*
* Note(s)    : 1) Interrupts are disabled during this call.
*********************************************************************************************************
*/

void  App_TaskDelHook (OS_TCB *ptcb)
{
    (void)ptcb;
}

/*
*********************************************************************************************************
*                                             App_TaskIdleHook()
*
* Description: This function is called by the idle task.  This hook has been added to allow you to do  
*              such things as STOP the CPU to conserve power.
*
* Arguments  : none
*
* Note(s)    : 1) Interrupts are enabled during this call.
*********************************************************************************************************
*/

#if OS_VERSION >= 251
void  App_TaskIdleHook (void)
{
}
#endif

/*
*********************************************************************************************************
*                                   App_TaskStatHook()
*
* Description: This function is called every second by uC/OS-II's statistics task.  This allows your 
*              application to add functionality to the statistics task.
*
* Returns    : none
*********************************************************************************************************
*/

void  App_TaskStatHook (void)
{
}

/*
*********************************************************************************************************
*                                           App_TaskSwHook()
*
* Description: This function is called when a task switch is performed.  This allows you to perform other
*              operations during a context switch.
*
* Arguments  : none
*
* Note(s)    : 1) Interrupts are disabled during this call.
*              2) It is assumed that the global pointer 'OSTCBHighRdy' points to the TCB of the task that
*                 will be 'switched in' (i.e. the highest priority task) and, 'OSTCBCur' points to the 
*                 task being switched out (i.e. the preempted task).
*********************************************************************************************************
*/

#if OS_TASK_SW_HOOK_EN > 0
void  App_TaskSwHook (void)
{
#if (uC_PROBE_OS_PLUGIN > 0) && (OS_PROBE_HOOKS_EN > 0)
    OSProbe_TaskSwHook();
#endif
}
#endif

/*
*********************************************************************************************************
*                                           App_TCBInitHook()
*
* Description: This function is called by OS_TCBInit() after setting up most of the TCB.
*
* Arguments  : ptcb    is a pointer to the TCB of the task being created.
*
* Note(s)    : 1) Interrupts may or may not be ENABLED during this call.
*********************************************************************************************************
*/

#if OS_VERSION >= 204
void  App_TCBInitHook (OS_TCB *ptcb)
{
    (void)ptcb;
}
#endif

/*
*********************************************************************************************************
*                                               App_TimeTickHook()
*
* Description: This function is called every tick.
*
* Arguments  : none
*
* Note(s)    : 1) Interrupts may or may not be ENABLED during this call.
*********************************************************************************************************
*/

#if OS_TIME_TICK_HOOK_EN > 0
void  App_TimeTickHook (void)
{
#if (uC_PROBE_OS_PLUGIN > 0) && (OS_PROBE_HOOKS_EN > 0)
    OSProbe_TickHook();
#endif
}
#endif
#endif

