/*
 *  linux/arch/arm/mach-imx/time.c
 *
 *  Copyright (C) 2000-2001 Deep Blue Solutions
 *  Copyright (C) 2002 Shane Nay (shane@minirl.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/init.h>
#include <linux/interrupt.h>
#include <linux/time.h>
#include <linux/clocksource.h>

#include <asm/hardware.h>
#include <asm/io.h>
#include <asm/leds.h>
#include <asm/irq.h>
#include <asm/mach/time.h>

/* Use timer 1 as system timer */
#define TIMER_BASE IMX_TIM1_BASE

extern irqreturn_t timer_interrupt(int irq, void *dev_id, struct pt_regs *regs);

/*
 * Returns number of us since last clock interrupt.  Note that interrupts
 * will have been disabled by do_gettimeoffset()
 */
static unsigned long imx_gettimeoffset(void)
{
	unsigned long ticks;

	/*
	 * Get the current number of ticks.  Note that there is a race
	 * condition between us reading the timer and checking for
	 * an interrupt.  We get around this by ensuring that the
	 * counter has not reloaded between our two reads.
	 */
	ticks = IMX_TCN(TIMER_BASE);

	/*
	 * Interrupt pending?  If so, we've reloaded once already.
	 */
	if (IMX_TSTAT(TIMER_BASE) & TSTAT_COMP){

		ticks += LATCH;
	}

	/*
	 * Convert the ticks to usecs
	 */
	return (1000000 / CLK32) * ticks;
}

static cycle_t imx_get_cycles(void)
{
        return IMX_TCN(TIMER_BASE);
}

static irqreturn_t xen_virtual_timer_interrupt(int irq, void *dev_id)
{
	write_seqlock(&xtime_lock);
	timer_tick();
	write_sequnlock(&xtime_lock);

	return IRQ_HANDLED;
}

static struct irqaction imx_timer_irq = {
	.name		= "i.MX Timer Tick",
	.flags		= SA_INTERRUPT,
	.handler	= timer_interrupt
};

static struct clocksource clocksource_imx = {
	.name           = "imx_timer",
	.rating         = 200,
	.read           = imx_get_cycles,
	.mask           = CLOCKSOURCE_MASK(32),
	.shift          = 20,
	.flags          = CLOCK_SOURCE_IS_CONTINUOUS,
};

/*
 * Set up timer interrupt, and return the current time in seconds.
 */

 void imx_timer_init(void)
{

	bind_virq_to_irqhandler(
                        VIRQ_TIMER,
                        0,
                        xen_virtual_timer_interrupt,
                        SA_INTERRUPT,
                        "timer0",
                        NULL);
        clocksource_imx.mult =
		clocksource_hz2mult(CLOCK_TICK_RATE, clocksource_imx.shift);
	clocksource_register(&clocksource_imx);
					
}

struct sys_timer imx_timer = {
	.init		= imx_timer_init,
	.offset		= imx_gettimeoffset,
};
