/******************************************************************************
 * mm/hypervisor.c
 * 
 * Update page tables via the hypervisor.
 * 
 * Copyright (c) 2002-2004, K A Fraser
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation; or, when distributed
 * separately from the Linux kernel or incorporated into other
 * software packages, subject to the following license:
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this source file (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use, copy, modify,
 * merge, publish, distribute, sublicense, and/or sell copies of the Software,
 * and to permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
 * IN THE SOFTWARE.
 */

#include <linux/module.h>
#include <asm/pgtable.h>
#include <xen/hypervisor.h>


extern unsigned char guest_shared_info[4096];
extern unsigned char guest_shared_info_base[4096*8];


void xen_drain_wb()
{
	struct mmuext_op op;
	op.cmd = MMUEXT_DRAIN_WB;
	BUG_ON(HYPERVISOR_mmuext_op(&op, 1, NULL, DOMID_SELF) < 0);
}


void xen_invalidate_i_tlb_entry(unsigned long vaddr)
{
	struct mmuext_op op;
	op.cmd = MMUEXT_INVALIDATE_I_TLB_ENTRY;
	op.arg1.linear_addr = (unsigned long) vaddr;
	BUG_ON(HYPERVISOR_mmuext_op(&op, 1, NULL, DOMID_SELF) < 0);
}

void xen_invalidate_d_tlb_entry(unsigned long vaddr)
{
	struct mmuext_op op;
	op.cmd = MMUEXT_INVALIDATE_D_TLB_ENTRY;
	op.arg1.linear_addr = (unsigned long) vaddr;
	BUG_ON(HYPERVISOR_mmuext_op(&op, 1, NULL, DOMID_SELF) < 0);
}

void xen_flush_user_tlb_range(unsigned long start, unsigned long end, struct vm_area_struct *mm)
{
	struct mmuext_op op;
	op.cmd = MMUEXT_FLUSH_KERN_TLB_RANGE;
	op.arg1.linear_addr = (unsigned long) start;
	op.arg2.end         = end;
	op.arg3.addr	    = mm;
	BUG_ON(HYPERVISOR_mmuext_op(&op, 1, NULL, DOMID_SELF) < 0);
}

void xen_flush_kern_tlb_range(unsigned long start, unsigned long end)
{
	struct mmuext_op op;
	op.cmd = MMUEXT_FLUSH_KERN_TLB_RANGE;
	op.arg1.linear_addr = (unsigned long) start;
	op.arg2.end         = end;
	BUG_ON(HYPERVISOR_mmuext_op(&op, 1, NULL, DOMID_SELF) < 0);
}


void xen_flush_kern_cache_all(void)
{
	xen_flush_cache();
}

void xen_flush_user_cache_all(void)
{
	xen_flush_cache();
}

void xen_coherent_kern_range(unsigned long start, unsigned long end)
{
	struct mmuext_op op;
	op.cmd = MMUEXT_COHERENT_KERN_RANGE;
	op.arg1.linear_addr = (unsigned long) start;
	op.arg2.end         = end;
	BUG_ON(HYPERVISOR_mmuext_op(&op, 1, NULL, DOMID_SELF) < 0);
}

void xen_coherent_user_range(unsigned long start, unsigned long end)
{
	struct mmuext_op op;
	op.cmd = MMUEXT_COHERENT_USER_RANGE;
	op.arg1.linear_addr = (unsigned long) start;
	op.arg2.end         = end;
	BUG_ON(HYPERVISOR_mmuext_op(&op, 1, NULL, DOMID_SELF) < 0);
}

void xen_flush_user_cache_range(unsigned long start, unsigned long end, unsigned int flags)
{
	struct mmuext_op op;
	op.cmd = MMUEXT_FLUSH_USER_CACHE_RANGE;
	op.arg1.linear_addr = (unsigned long) start;
	op.arg2.end         = end;
	op.arg3.flags       = flags;
	BUG_ON(HYPERVISOR_mmuext_op(&op, 1, NULL, DOMID_SELF) < 0);
}

void xen_flush_kern_dcache_page(void *page)
{
	struct mmuext_op op;
	op.cmd = MMUEXT_FLUSH_KERN_DCACHE_PAGE;
	op.arg1.linear_addr = (unsigned long) page;
	BUG_ON(HYPERVISOR_mmuext_op(&op, 1, NULL, DOMID_SELF) < 0);
}

void xen_dmac_inv_range(unsigned long start, unsigned long end)
{
	struct mmuext_op op;
	op.cmd = MMUEXT_DMAC_INV_RANGE;
	op.arg1.linear_addr = (unsigned long) start;
	op.arg2.end         = end;
	BUG_ON(HYPERVISOR_mmuext_op(&op, 1, NULL, DOMID_SELF) < 0);
}

void xen_dmac_clean_range(unsigned long start, unsigned long end)
{
	struct mmuext_op op;
	op.cmd = MMUEXT_DMAC_CLEAN_RANGE;
	op.arg1.linear_addr = (unsigned long) start;
	op.arg2.end         = end;
	BUG_ON(HYPERVISOR_mmuext_op(&op, 1, NULL, DOMID_SELF) < 0);
}

void xen_dmac_flush_range(unsigned long start, unsigned long end)
{
	struct mmuext_op op;
	op.cmd = MMUEXT_DMAC_FLUSH_RANGE;
	op.arg1.linear_addr = (unsigned long) start;
	op.arg2.end         = end;
	BUG_ON(HYPERVISOR_mmuext_op(&op, 1, NULL, DOMID_SELF) < 0);
}


void xen_flush_cache(void)
{
	struct mmuext_op op;
	op.cmd = MMUEXT_FLUSH_CACHE;
	BUG_ON(HYPERVISOR_mmuext_op(&op, 1, NULL, DOMID_SELF) < 0);
}

void xen_dcache_clean_area(void *p, int size)
{
	struct mmuext_op op;
	op.cmd = MMUEXT_DCACHE_CLEAN_AREA;
	op.arg1.linear_addr = (unsigned long) p;
	op.arg2.size        = size;
	BUG_ON(HYPERVISOR_mmuext_op(&op, 1, NULL, DOMID_SELF) < 0);
}

void xen_flush_tlb_page(unsigned int cpu_vm_set, unsigned long uaddr)
{
	struct mmuext_op op;
	op.cmd = MMUEXT_FLUSH_TLB_PAGE;
	op.arg1.linear_addr = (unsigned long) uaddr;
	BUG_ON(HYPERVISOR_mmuext_op(&op, 1, NULL, DOMID_SELF) < 0);
}

void xen_flush_tlb_all(void)
{
	struct mmuext_op op;
	op.cmd = MMUEXT_FLUSH_TLB_ALL;
	BUG_ON(HYPERVISOR_mmuext_op(&op, 1, NULL, DOMID_SELF) < 0);
}

void xen_flush_tlb_mm(unsigned int cpu_vm_set)
{
	struct mmuext_op op;
	op.cmd = MMUEXT_FLUSH_TLB_MM;
	op.arg1.flags = cpu_vm_set;
	BUG_ON(HYPERVISOR_mmuext_op(&op, 1, NULL, DOMID_SELF) < 0);
}

void xen_flush_tlb_kernel_page(unsigned long kaddr)
{
	struct mmuext_op op;
	op.cmd = MMUEXT_FLUSH_TLB_KERNEL_PAGE;
	op.arg1.linear_addr = (unsigned long) kaddr;
	BUG_ON(HYPERVISOR_mmuext_op(&op, 1, NULL, DOMID_SELF) < 0);
}

void xen_flush_pmd_entry(pmd_t *pmd)
{
	struct mmuext_op op;
	op.cmd = MMUEXT_FLUSH_PMD_ENTRY;
	op.arg1.linear_addr = (unsigned long) pmd;
	BUG_ON(HYPERVISOR_mmuext_op(&op, 1, NULL, DOMID_SELF) < 0);
}

void xen_clean_pmd_entry(pmd_t *pmd)
{
	struct mmuext_op op;
	op.cmd = MMUEXT_CLEAN_PMD_ENTRY;
	op.arg1.linear_addr = (unsigned long) pmd;
	BUG_ON(HYPERVISOR_mmuext_op(&op, 1, NULL, DOMID_SELF) < 0);
}

#define SHARED_PTE_MASK	((0xE << 0) | ( 0xFF << 4))

shared_info_t *xen_map_shared_info(unsigned long pa)
{
	unsigned int offset;

	if ( (offset = HYPERVISOR_update_va_mapping((unsigned long) guest_shared_info_base, __pte(pa | SHARED_PTE_MASK), 
						    UVMF_SHARED_INFO | UVMF_INVLPG)) < 0 ) {

		printk("Failed to map shared_info!!\n");
		*(int*)0=0;
	}
	return (shared_info_t *) ((int) guest_shared_info_base + (offset << PAGE_SHIFT));
}


void xen_switch_mm(unsigned long ptr, struct mm_struct *mm)
{
	struct mmuext_op op;
	op.cmd = MMUEXT_NEW_BASEPTR;
	op.arg1.mfn = ptr;
	BUG_ON(HYPERVISOR_mmuext_op(&op, 1, NULL, DOMID_SELF) < 0);
}

void xen_clear_user_page(void *p, unsigned long user)
{
	struct mmuext_op op;
	op.cmd = MMUEXT_CLEAR_USER_PAGE;
	op.arg1.linear_addr = (unsigned long) p;
	op.arg2.addr         = user;
	BUG_ON(HYPERVISOR_mmuext_op(&op, 1, NULL, DOMID_SELF) < 0);

}

void xen_copy_user_page(void *to, const void *from, unsigned long user)
{
	struct mmuext_op op;
	op.cmd = MMUEXT_COPY_USER_PAGE;
	op.arg1.linear_addr = (unsigned long) to;
	op.arg2.addr        = (unsigned long) from;
	op.arg3.addr        = user;
	BUG_ON(HYPERVISOR_mmuext_op(&op, 1, NULL, DOMID_SELF) < 0);
}


pgd_t *xen_get_pgd()
{
	struct mmuext_op op;
	op.cmd = MMUEXT_GET_PGD;
	BUG_ON(HYPERVISOR_mmuext_op(&op, 1, NULL, DOMID_SELF) < 0);

	return (pgd_t *) phys_to_virt(op.arg1.phys_addr);
}

void xen_debug_tlb_dump(void)
{
	struct mmuext_op op;
	op.cmd = MMUEXT_DEBUG_TLB_DUMP;
	BUG_ON(HYPERVISOR_mmuext_op(&op, 1, NULL, DOMID_SELF) < 0);
}


/* Ensure multi-page extents are contiguous in machine memory. */
int xen_create_contiguous_region(
	unsigned long vstart, unsigned int order, unsigned int address_bits)
{
	
	return 0;
}

void xen_destroy_contiguous_region(unsigned long vstart, unsigned int order)
{
	return;
}


void xen_xscale_flush_l2cache_all(void)
{
	xen_printf("xen_xscale_flush_l2cache_all is called!!!\n");
}
