/******************************************************************************
 *  linux/arch/arm/mm/pgtbl-xen.c (linux-xen only)
 *
 *  by sungkwan heo <sk.heo@samsung.com>
 *
 * This file contains the page table manuplation functions for linux-xen.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation; or, when distributed
 * separately from the Linux kernel or incorporated into other
 * software packages, subject to the following license:
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this source file (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use, copy, modify,
 * merge, publish, distribute, sublicense, and/or sell copies of the Software,
 * and to permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
 * IN THE SOFTWARE.
 */

#include <linux/module.h>
#include <asm/pgtable.h>
#include <xen/fixmap.h>
#include <linux/sched.h>
#include <xen/hypervisor.h>

extern struct mm_struct init_mm;

/*
 * PMD updates
 */
void set_pmd(pmd_t *pmdp, pmd_t pmd)
{
	mmu_update_t mmu_updates[1];

	mmu_updates[0].ptr = (unsigned long) __pa(pmdp);
	mmu_updates[0].val = (uint64_t)pmd;

	if(HYPERVISOR_mmu_update(mmu_updates, 1, NULL, DOMID_SELF) < 0)
	{
		//printk("PTE could not be updated\n");
		xen_printf("PTE could not be updated\n");
		*((unsigned long *)0) = 0;
	}

	return;
}

void copy_pmd(pmd_t *pmdpd, pmd_t *pmdps)
{
	mmu_update_t mmu_updates[2];

	mmu_updates[0].ptr = (unsigned long) __pa(&pmdpd[0]);
	mmu_updates[0].val = pmdps[0];
	
	mmu_updates[1].ptr = (unsigned long) __pa(&pmdpd[1]);
	mmu_updates[1].val = pmdps[1];

	rmb();
		
	if(HYPERVISOR_mmu_update(mmu_updates, 2, NULL, DOMID_SELF) < 0)
	{
		xen_printf("PTE could not be updated\n");
		*((unsigned long *)0) = 0;
	}

	return;
}

void pmd_clear(pmd_t *pmdp)
{
	mmu_update_t mmu_updates[2];

	mmu_updates[0].ptr = (unsigned long) __pa(&pmdp[0]);
	mmu_updates[0].val = __pmd(0);
	
	mmu_updates[1].ptr = (unsigned long) __pa(&pmdp[1]);
	mmu_updates[1].val = __pmd(0);
	
	rmb();
		
	if(HYPERVISOR_mmu_update(mmu_updates, 2, NULL, DOMID_SELF) < 0)
	{
		xen_printf("PTE could not be updated\n");
		*((unsigned long *)0) = 0;
	}

	return;
}


/*
 * PTE updates
 */
asmlinkage void  hypervisor_set_pte(pte_t *ptep, pte_t pte_l, pte_t pte_h)
{

	mmu_update_t mmu_updates[2];

	mmu_updates[0].ptr = (uint64_t) __pa(ptep);
	mmu_updates[0].val = (uint64_t)pte_l;		// linux pte
	
	mmu_updates[1].ptr = (uint64_t) __pa(ptep - 512); // -2048 bytes
	mmu_updates[1].val = (uint64_t)pte_h;		// hardware pte

	rmb();
	
	if(HYPERVISOR_mmu_update(mmu_updates, 2, NULL, DOMID_SELF) < 0)
	{
		xen_printf("PTE could not be updated\n");
		*((unsigned long *)0) = 0;
	}

	return;
}

void hypervisor_set_pte_tiny(pte_t *ptep, pte_t pte_l1, pte_t pte_h)
{
	mmu_update_t mmu_updates[2];

	mmu_updates[0].ptr = (unsigned long) __pa(ptep);
	mmu_updates[0].val = pte_l1;

	mmu_updates[1].ptr = 0;
	mmu_updates[1].val = 0;

	rmb();
	
	if(HYPERVISOR_mmu_update(mmu_updates, 1, NULL, DOMID_SELF) < 0)
	{
		xen_printf("PTE could not be updated\n");
		*((unsigned long *)0) = 0;
	}
}

/*
 * Associate a virtual page frame with a given physical page frame 
 * and protection flags for that frame.
 */ 
static void set_pte_pfn(unsigned long vaddr, unsigned long pfn, pgprot_t flags)
{
	pgd_t *pgd;
	pud_t *pud;
	pmd_t *pmd;
	pte_t *pte;

	pgd = init_mm.pgd + pgd_index(vaddr);

	if (pgd_none(*pgd)) {
		BUG();
		return;
	}

	pud = pud_offset(pgd, vaddr);
	if (pud_none(*pud)) {
		BUG();
		return;
	}
	pmd = pmd_offset(pud, vaddr);

	if (pmd_none(*pmd)) {
		BUG();
		return;
	}
	pte = pte_offset_kernel(pmd, vaddr);

	/* <pfn,flags> stored as-is, to permit clearing entries */
	set_pte_ext(pte, pfn_pte(pfn, flags),0);
}

/*
 *  FIXMAP from x86 code
 */

static int nr_fixmaps = 0;

void __set_fixmap (enum fixed_addresses idx, maddr_t phys, pgprot_t flags)
{
	unsigned long address = __fix_to_virt(idx);

	if (idx >= __end_of_fixed_addresses) {
		BUG();
		return;
	}
	switch (idx) {
	case FIX_WP_TEST:
	default:
		set_pte_pfn(address, phys >> PAGE_SHIFT, flags);
		break;
	}
	nr_fixmaps++;
}
