/**
*@defgroup Keypad Input driver _ for dom0
*/
/**@{*/


/**
*@file mx21_kpp_back.c
*@brief Keypad Input driver source file
*
* linux/drivers/input/keyboard/mx21_kpp_back.c
*/

/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Copyright (C) 2003 Motorola Inc Ltd
 *
 */
extern int digi_reinit(void);

 //<<<<<< includes
#include <linux/module.h>
#include <linux/version.h>
#include <linux/init.h>
#include <linux/fs.h>
#include <linux/delay.h>
#include <linux/string.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/types.h>
#include <linux/fcntl.h>
#include <linux/input.h>
#include <linux/interrupt.h>
#include <linux/ptrace.h>
#include <linux/ioport.h>
#include <linux/in.h>
#include <linux/slab.h>
#include <linux/vmalloc.h>
#include <asm/bitops.h>
#include <asm/io.h>
#include <linux/errno.h>
#include <linux/wait.h>
#include <asm/irq.h>
#include <asm/arch/hardware.h>
#include <asm/arch/irqs.h>
#include <asm/hardware.h> 
#include <asm/system.h>
#include <linux/device.h>
#include <linux/list.h>

#include "mx21_kpp.h"
#include <linux/platform_device.h>

#ifdef CONFIG_XENOLINUX
extern unsigned int foreground_dom;
extern unsigned int kpp_backend;
#endif

/* variables for input dev */
static struct input_dev *kpp0_dev;

/* global variable for keypad device */
static u8 kpp_pressed=1;

//functions---------------------------------------------------------

static void scancode_convert(u16 scanbit, u32* p_scancode)
{
	u16 colnum=0;
	u16 tempcol,temprow;
	u16 tmp;
	int i;
	u32 code[2];
	tmp = 1;
	tempcol = (scanbit&KPP_MX2_MASK&0xff00)>>8;
	temprow = ~(scanbit&KPP_MX2_MASK&0x00ff);
	code[0]=code[1]=0;
	TRACE("scanbit = 0x%x ,temprow = 0x%x \n",scanbit,temprow);
	if(tempcol & 0xf)//col0~3
	{
		for(i=0;i<4;i++)
		{
			if(tempcol & (1<<i))
			{
				colnum = i;
				break;
			}	
		}
		for(i=0; i<6; i++)
		{
			if(temprow&(1<<i))
				code[0] |= 1<<(colnum*8 + i);			
		}
	}
	if(tempcol & 0xf0)//col4~7
	{
		for(i=4;i<8;i++)
		{
			if(tempcol & (1<<i))
			{
				colnum = i-4;
				break;
			}	
		}
		for(i=0; i<6; i++)
		{
			if(temprow&(1<<i))
				code[1] |= 1<<(colnum*8 + i);			
		}
	}
	*p_scancode |= code[0];
	p_scancode++;
	*p_scancode |= code[1];
}

static int spkey_check(const int c, const int r, e_key_type *p_key_t)
{
	if(c== 4 && r == 1)
		*p_key_t = FD_SWITCH;
	else if(c == 4 && r == 2)
		*p_key_t = ALPHA1;
	else if(c == 4 && r == 3)
		*p_key_t = NUM;		
	else
		return 0;		//not checked
	return 1;
}


int spkey_action(const e_key_type key_t, int *p_type)
{
	//int need_switch = 0;
	int next_fg_dom = 0;
	int old_fg;
	int i;

	switch(key_t){
	case FD_SWITCH:		//foreground domain swtiching

		//keypad foreground domain info switching
		old_fg = foreground_dom;
		for (i=0;i<MAX_NUM_DOMAINS;++i){
			next_fg_dom = (foreground_dom+i+1) % MAX_NUM_DOMAINS;
			switch_foreground_dom(next_fg_dom);
			if (foreground_dom != old_fg)
				break;
		}
		break;
	case NUM:	*p_type	= 0; break;
	case ALPHA1:	*p_type	= 1; break;	
	}

	return 0;
}
static int key_type_process(const u32 *p_scancode, u8 *p_keycode, int *p_overwrite)
{
	u32 tmp = 0;
	static int c, r, old_c, old_r;
	static int old_t, t = 0;
	u8 kcol;
	int ret=0;
	e_key_type key_t;

	old_c = c;
	old_r = r;
	old_t = t;
	c = r = 0;
	if (p_scancode[1] == 0){
		tmp = p_scancode[0];		//low
	}
	else if (p_scancode[0] == 0){
		tmp = p_scancode[1];		//high
		c  = 4;
	}
	else{
		*p_keycode = 0; //multikey detected
		printk("multikey detected.\n");
	}

	while(tmp){		
		kcol = tmp & 0xff;
		if(kcol){
			switch(kcol){
			case 0x01:	r=0; break;
			case 0x02:	r=1; break;
			case 0x04:	r=2; break;
			case 0x08:	r=3; break;
			case 0x10:	r=4; break;
			case 0x20:	r=5; break;
			default:
				printk("scan error : kcol is not available.\n");
			}
			if(spkey_check(c,r,&key_t))	{	//special key is checked ? 
				spkey_action(key_t, &t);		//do it.
				ret = 1;		//if retuer value is 1, it doesn't send the key
			}
			else{	//normal key
				if(kpp_keycode[c][r][t] && t!=0){			//when the key type is alphabet
					if(c == old_c && r == old_r)	{	//same key has been pressed
						*p_overwrite = 1;
						t = (t==1) ? 2 : 1;			//t toggle 1 - 2
					}
					else				// other alphabet key follows
						t = 1;
					*p_keycode = kpp_keycode[c][r][t];
				}
				else	{
					if(kpp_keycode[c][r][t] == KEY_RIGHT && old_t!=0 ){	//in case right key follows alphabet
						ret = 1;		// ignore ( doesn't send this key )
					}
					else
						*p_keycode = kpp_keycode[c][r][0];
				}
			}
			break;
		}	
		tmp = tmp >> 8;
		c++;
	}
	return ret;			//if the key is checked : return 1
}

/*******************************************************
*key transaction function
*Parameters:key: kpp_keycode, value : key_pressed
*Return	
*	0	indicates SUCCESS
* 	-1	indicates FAILURE
*Description: transact a key
********************************************************/
static int key_transaction(unsigned char key, int value)
{
	if(foreground_dom == kpp_backend) {
		input_report_key(kpp0_dev,key, value);
	} else {
		key_send(key, value);
	}
	return 0;	
}

static int kpp_scan_matrix(void)
{
	u16 row = KPP_MX2_MASK&0x00ff;
	u16 col = KPP_MX2_MASK&0xff00;
	u16 iCol, scanMask;
	u16 scaned_rowbit;
	u32 scancode[2] = {0,0};	
	static u8  keycode;
	int is_scaned=0;
	static int scan_error=0;
	static int checked=0;
	int overwrite = 0;

	if(scan_error == 1)	{ //if button missed error was occurred
		scan_error = 0;
		return -1;		  //ignore
	}

	for(iCol=0; iCol<KPP_MX2_COL_NUM; iCol++)
	{
		KPDR |= col;		//write 1's to KPDR[15:8]
		KPCR &= ~col;	//configure columns as totem-pole outputs
		KPCR |= col;		//configure columns as open-drain
		scanMask = (1<<(iCol+8));	//write a single column to 0,others to 1
		KPDR &= ~scanMask;

		udelay(10);	// very important to detect right key data.
		scaned_rowbit = (KPDR&0x00ff)| scanMask;	//sample row inputs and save data. 

		if((KPDR&row)!=row){	//if get the scan data	
			is_scaned=1;
			break;
		}
		KPDR |= scanMask;		//restore the colmask
	}
	KPDR &= ~col;

	if(is_scaned == 1){	//if get the scan data
		if(kpp_pressed != 1){	//In case the release interrupt is occured but key is scaned
			scan_error = 1;	//this is button missed error!!
			printk("### button missed error!!\n");
			return -1;
		}

		scancode_convert(scaned_rowbit, scancode);
		checked = key_type_process(scancode, &keycode, &overwrite);	
	}
	else {	//can't find the data
		if(kpp_pressed != 0){	//In case the depress interrupt is occured but key is not scaned
			scan_error = 1;
			printk("### can not scan any key!!\n");
			return -1;
		}
	}

	if(checked == 0){
		if(overwrite && is_scaned){
			key_transaction(KEY_BACKSPACE,1);	//remove a previous character
			key_transaction(KEY_BACKSPACE,0);
		}
		key_transaction(keycode,is_scaned);
	}
	return 0;
}

irqreturn_t kpp_isr(int irq, void *dev_id, struct pt_regs *regs)
{
	//clear Key depress and key release event
	KPSR = (KPSR | KPP_KPKD_BIT_MASK | KPP_KPKR_BIT_MASK);
	
	//if first depress key
	if(kpp_pressed == 1){
		//disable key depress interrupt
		KPSR &= ~KPP_KDIE_BIT_MASK;
	
		//enable key release interrupt
		KPSR |= KPP_KRIE_BIT_MASK;
		kpp_scan_matrix();
		kpp_pressed = 0;
	return IRQ_HANDLED;
	}
	kpp_scan_matrix();
	kpp_pressed = 1;

	//if get key release interrupt
	//disable key release interrupt
	KPSR &= ~KPP_KRIE_BIT_MASK;

	//enable key depress interrupt
	KPSR |= KPP_KDIE_BIT_MASK;

	return IRQ_HANDLED;		
}


/*init procedure******************************************/
//Configure KPP setting,			
void kpp_hw_configure(void)
{
	u16 row,col;
	row = KPP_MX2_MASK&0x00ff;
	col = KPP_MX2_MASK&0xff00;
	
	//GPIO portE3467 as alternate function, since the UART2 has some conflict to Keypad
	GIUS(GPIOE) &= ~0x000000d8;
	GPR(GPIOE) |= 0x000000d8;
	
	//enable clk gate of kpp
	PCCR1 |= 0x40000000;
	KPSR |= KPP_KPPEN_BIT_MASK;
	
	//enable key
	KPCR |= row;
	//clear kpp data register
	KPDR &= ~(col|row);
	//configure col as output opendrain
	KPCR |= col;
	KDDR |= col;
	//configure row as input
	KDDR &= ~row;
	//Clear KPKD status flag and synchronizer chain
	KPSR |= KPP_KDSC_BIT_MASK;
	KPSR |= KPP_KPKD_BIT_MASK;
	//Set KDIE control bit, clear the KRIE control bit
	KPSR |= KPP_KDIE_BIT_MASK;
	KPSR &= ~KPP_KRIE_BIT_MASK;
}

static void kpp_irq_enable(void)
{
	//clear key depress event
	KPSR |= KPP_KPKD_BIT_MASK;
	//clear key release event
	KPSR |= KPP_KPKR_BIT_MASK;
	//enable irq
	//mx2ads_unmask_irq(KPP_IRQ);	
}

static int kpp0_resume( struct device * dev, u32 level )
{	
	int i;
	switch(level){
	case RESUME_POWER_ON:
		kpp_hw_configure();
		kpp_irq_enable();
		break;
	}
	
	return 0;
}

static int kpp0_suspend(struct device * dev, u32 state, u32 level)
{
	int i;
	switch(level){
	case SUSPEND_POWER_DOWN:
		printk("keypad device driver is suspend.\n");
		//disable key release interrupt
		KPSR &= ~KPP_KRIE_BIT_MASK;
		//enable key depress interrupt
		KPSR |= KPP_KDIE_BIT_MASK;
		//disable clk gate of kpp
		PCCR1 &= ~0x40000000;
		break;
	}
	
	return 0;
}


static struct device_driver kpp0_drv = {
		.bus = &platform_bus_type,
		.name = "kpp0", 
		.resume = kpp0_resume,
		.suspend = kpp0_suspend,
		};
static struct platform_device * kpp0_device;

/*******************************************************
*kppif init function
*Parameters:None
*Return	
*	0	indicates SUCCESS
* 	-1	indicates FAILURE
*Description: 
********************************************************/
static int __init kpp_init(void)
{
	int err,i,j,k;

	//ntames8
	kpp0_dev = input_allocate_device();
	if (!kpp0_dev) {
		input_free_device(kpp0_dev);
		return -ENOMEM;
	}

	kpp0_dev->evbit[0] = BIT(EV_KEY);
	kpp0_dev->keycode = kpp_keycode;
	kpp0_dev->keycodesize = sizeof(unsigned char);
	kpp0_dev->keycodemax = ARRAY_SIZE(kpp_keycode);

	for (i = 0; i < 6; i++)
		for(j=0; j < 6; j++)
			for(k=0; k < 3; k++)
			if (kpp_keycode[i][j][k])
				set_bit(kpp_keycode[i][j][k], kpp0_dev->keybit);
   
	kpp0_dev->name = "Keypad for Freescale-Suzhou";
	kpp0_dev->phys = "kpp0/input0";
	kpp0_dev->id.vendor = 0x0001;
	kpp0_dev->id.product = 0x0001;
	kpp0_dev->id.version = 0x0001;

	input_register_device(kpp0_dev);
	kpp0_device = platform_device_register_simple("kpp0",0,0,0);
	driver_register(&kpp0_drv);

	//with a keypad-------------
	err = request_irq(KPP_IRQ,kpp_isr,	SA_INTERRUPT|SA_SHIRQ,"kpp","kpp");
	if(err)
	{
		printk(KERN_CRIT "Wow!  Can't register IRQ[%d] for Keypad\n", KPP_IRQ);
		return -1;
	}

	// NOTICE: KPP should not be registered as HID irq.
	// KPP irq always should go to domain 0. (KPP irq is virtualized.)
	// HID irq means switching based irqs, not virtualized.
	// HYPERVISOR_do_set_HID_irq(KPP_IRQ);

	//enable ADC interrupt
	kpp_hw_configure();
	kpp_irq_enable();

	printk("Kpp Driver 1.0.0, for Freescale-Suzhou\n");
	return 0;
}

static void __exit kpp_cleanup(void)
{
	platform_device_unregister(kpp0_device);
	driver_unregister(&kpp0_drv);
	input_unregister_device(kpp0_dev);

	//with a keypad-------------
	/*Do some cleanup work*/
	disable_irq(KPP_IRQ);
	free_irq(KPP_IRQ, "kpp");
}

int irq_mask, irq_list;

module_init(kpp_init);
module_exit(kpp_cleanup);

#if defined(module_param)
module_param(irq_mask, int, 0644);
module_param(irq_list, int, 0444);
#else   // old style
MODULE_PARM(irq_mask, "i");
MODULE_PARM(irq_list, "1-4i");
#endif

MODULE_LICENSE("GPL");

/**@}*/



