/*************************************************************************

Title:   	
Filename:   $Header:$
Hardware:   MX21
Summay:

License:    The contents of this file are subject to the Mozilla Public
            License Version 1.1 (the "License"); you may not use this file
            except in compliance with the License. You may obtain a copy of
            the License at http://www.mozilla.org/MPL/

Author:   
Company:    Motorola Suzhou Design Center

====================Change Log========================
$Log:$

********************************************************************************/

/** @example pen_test.c */

/**
 * @defgroup DIGI DIGI driver
 */

///@file  linux/drivers/input/touchscree/mx21ads_ts.c
///@brief This is the digi driver source code.

#include <linux/module.h>
#include <linux/version.h>
#include <linux/init.h>
#include <linux/fs.h>
#include <linux/delay.h>
#include <linux/poll.h>
#include <asm/uaccess.h>        
#include <linux/miscdevice.h>
#include <linux/string.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/types.h>
#include <linux/fcntl.h>
#include <linux/interrupt.h>
#include <linux/ptrace.h>
#include <linux/ioport.h>
#include <linux/in.h>
#include <linux/slab.h>
#include <linux/vmalloc.h>
#include <linux/string.h>
#include <asm/bitops.h>
#include <asm/io.h>
#include <linux/errno.h>
#include <linux/wait.h>
#include <linux/pm.h>

#include <asm/irq.h>
#include <asm/arch/hardware.h>

#include <asm/arch/irqs.h>
#include "mx21ads_ts.h"

//<<<<<< Private Macro

#define MODULE_DIGI "digi"
#define MAJOR_NUM		253

#define DBMX_DEBUG 1
#if DBMX_DEBUG
#define TRACE(fmt, args...) \
	{ \
		printk("\n %s:%d:%s:",__FILE__, __LINE__,__FUNCTION__); \
		printk(fmt, ## args);\
	}
#else
#define TRACE(fmt, args...)
#endif

#define FAILED(fmt, args...) \
	{ \
		printk("\n %s:%d:%s:",__FILE__, __LINE__,__FUNCTION__); \
		printk(fmt, ## args);\
	}

#define INFO(fmt, args...) \
	{ \
		printk("\n"); \
		printk(fmt, ## args);\
	}


//>>>>>> Private Macro


//<<<<<Private Function Declearation
static int digi_open(struct inode * inode, struct file * filp);
static int digi_release(struct inode * inode, struct file * filp);
static int digi_fasync(int fd, struct file *filp, int mode);
static int digi_ioctl(struct inode * inode,struct file *filp,unsigned int cmd,unsigned long arg);
static ssize_t digi_read(struct file * filp, char * buf, size_t count, loff_t * l);
static unsigned int digi_poll(struct file * filp, struct poll_table_struct * wait);

static int check_device(struct inode *pInode);
static ts_event_t* get_data(void);
static int get_block(ts_event_t* * block, int size);
static void  digi_sam_callback(unsigned long data);

static void add_x_y(unsigned x, unsigned y, unsigned flag);
static int init_buf(void);
static void spi_flush_fifo(void);

static int spi_tx_fifo_empty(void);
static int spi_rx_fifo_data_ready(void);
static unsigned int spi_exchange_data(unsigned int dataTx);

//>>>>>Private Function Declearation

//<<<<<< Global Variable
static int	g_digi_major=0;
static wait_queue_head_t digi_wait;
struct fasync_struct *ts_fasync;
static u16 	rptr, wptr;
static struct timer_list pen_timer;	
u8 pen_timer_status;	
spinlock_t pen_lock;
static struct tasklet_struct	digi_tasklet; 
static char g_touch_enter_sleep = 1;
struct file_operations g_digi_fops = 
                       {
		open:           digi_open,
		release:        digi_release,
		read:           digi_read,
		poll:           digi_poll,
		ioctl:          NULL,
		fasync:         digi_fasync,
                       };                       
//>>>>>> Global Variable

static struct miscdevice g_digi_dev = {
    minor:  16,
    name:   "touchscreen",
    fops:   &g_digi_fops,
};

#define XLIMIT 160
#define YLIMIT 160

#define NODATA()	(rptr==wptr)
/**
*Misc functions for touch pannel
*/
static void touch_pan_enable(void)
{
	//set cspi1_ss0 to be low effective 
	//cspi1_ss0 --pd28, it connect with pen_cs_b
	DR(GPIOD) &= (~0x10000000);
}

static void touch_pan_disable(void)
{
#ifdef CONFIG_XEN_PRIVILEGED_GUEST
	//set the cspi1_ss0 to be high 
	DR(GPIOD) |= 0x10000000;
#endif
}

static void touch_pan_set_inter(void)
{
	int temp;
	//pe10, uart3_cts pin connect to penIrq

	GIUS(GPIOE) |= 0x00000400;
	OCR1(GPIOE) |= 0x00300000;
	IMR(GPIOE)  &= ~(0x00000400);
	DDIR(GPIOE) &= ~(0x00000400);
	temp = SSR(GPIOE);
}

static void touch_pan_enable_inter(void)
{
	IMR(GPIOE) |= 0x00000400;	
}

static void touch_pan_disable_inter(void)
{
#ifdef CONFIG_XEN_PRIVILEGED_GUEST
	IMR(GPIOE) &= ~(0x00000400);
#endif
}

static void touch_pan_clear_inter(void)
{
	volatile unsigned long *addr;
    ISR(GPIOE) = 0x00000400;
}

//here set it to be Negative level sensitive
static void touch_pan_set_neg_inter(void)
{
	ICR1(GPIOE) &= 0xffcfffff;
	ICR1(GPIOE) |= 0x00100000;
}

void touch_pan_init(void)//chip select for ADC
{
	//cspi1_ss0 --pd28, it connect with pen_cs_b
	GIUS(GPIOD) |= 0x10000000;
	OCR2(GPIOD) |= 0x03000000;
	DR(GPIOD) |= 0x10000000;	
	DDIR(GPIOD) |= 0x10000000;
}

//read data from Touch Pannel
void touch_pan_read_dev(u32* x, u32* y)
{
	u32 x_upper, x_lower, y_upper, y_lower;
	u32 temp;
	u32 imr_status;
	temp = ISR(GPIOE);
	imr_status = IMR(GPIOE) & 0x00000400;

	touch_pan_disable_inter();
	touch_pan_enable();

	x_upper = spi_exchange_data(0xD0);	// this is dummy data
	x_upper = spi_exchange_data(0x00);
	x_lower = spi_exchange_data(0x90);
	y_upper = spi_exchange_data(0x00);
	y_lower = spi_exchange_data(0x00);

      *x=( ((x_upper<<5) & 0xFFE0) | ((x_lower>>3) & 0x1F) );
      *y=( ((y_upper<<5) & 0xFFE0) | ((y_lower>>3) & 0x1F) );
      touch_pan_disable();
      TRACE("x=%x ,y=%x \n",*x,*y);

	ISR(GPIOE) = temp;

	IMR(GPIOE) |= imr_status;

	udelay(200);           	// to avoid conflicting with apm.
}

void touch_pan_read_data(u32* x,u32* y)
{
	int diff0, diff1, diff2, diff3;
	
#define NUM_OF_SAMPLE  (4)
#define MAX_POS_DIFF   (4)
    u32 xPos[NUM_OF_SAMPLE], yPos[NUM_OF_SAMPLE];

	touch_pan_read_dev(&xPos[0], &yPos[0]);
    if ((xPos[0]<100)||(xPos[0]>5000) ||
        (yPos[0]<100)||(yPos[0]>5000))
    {
        *x = TPNL_PEN_UPVALUE;
        *y = TPNL_PEN_UPVALUE;
        return;
    }
	touch_pan_read_dev(&xPos[1], &yPos[1]);
    if ((xPos[1]<100)||(xPos[1]>5000) ||
        (yPos[1]<100)||(yPos[1]>5000))
    {
        *x = TPNL_PEN_UPVALUE;
        *y = TPNL_PEN_UPVALUE;
        return;
    }
	touch_pan_read_dev(&xPos[2], &yPos[2]);
    if ((xPos[2]<100)||(xPos[2]>5000) ||
        (yPos[2]<100)||(yPos[2]>5000))
    {
        *x = TPNL_PEN_UPVALUE;
        *y = TPNL_PEN_UPVALUE;
        return;
    }
	touch_pan_read_dev(&xPos[3], &yPos[3]);
    if ((xPos[3]<100)||(xPos[3]>5000) ||
        (yPos[3]<100)||(yPos[3]>5000))
    {
        *x = TPNL_PEN_UPVALUE;
        *y = TPNL_PEN_UPVALUE;
        return;
    }

	//Add filtering process to get more precision coordinate
	diff0 = abs(xPos[0] - xPos[1]);
	diff1 = abs(xPos[1] - xPos[2]);
	diff2 = abs(xPos[2] - xPos[3]);
	diff3 = abs(xPos[3] - xPos[1]);

	 if (diff0 > XLIMIT || diff1 > XLIMIT || diff2 > XLIMIT || diff3 > XLIMIT)
	 	return;

      if (diff1 < diff2)
      {
		if (diff1 < diff0)
			*x = (xPos[1] + xPos[2]) / 2;
		else
			*x = (xPos[0] + xPos[1]) / 2;
      }
      else
      {
		if (diff2 < diff3)
			*x = (xPos[2] + xPos[3]) / 2;
		else
			*x = (xPos[3] + xPos[1]) / 2;
      }

/* Do the same for Y */
      diff0 = abs(yPos[0] - yPos[1]);
      diff1 = abs(yPos[1] - yPos[2]);
      diff2 = abs(yPos[2] - yPos[3]);
      diff3 = abs(yPos[3] - yPos[1]);

      if (diff0 > YLIMIT || diff1 > YLIMIT || diff2 > YLIMIT || diff3 > YLIMIT)
		return;

      if (diff1 < diff2)
      {
		if (diff1 < diff0)
			*y = (yPos[1] + yPos[2]) / 2;
		else
			*y = (yPos[0] + yPos[1]) / 2;
      }
      else
      {
		if (diff2 < diff3)
			*y = (yPos[2] + yPos[3]) / 2;
		else
			*y = (yPos[3] + yPos[1]) / 2;
      }
	
    return;
}

/**
 * @brief digi_isr
 *
 * Support touch pannel interrupt,since MX1 and MX21 adopted different
 * ADC chip,in MX1,AD7843 will generate interrupt according to level trigger
 * configuration, so can generate one interrupt to detect pen down, then reconfigure
 * it to generate pen up interrupt.But in MX21, the AD7873 will	have the pen irq
 * line reassert after each pen sample is taken. It will deassert when you take a
 * sample. When the pen is down, we disable the interrupt and take a sample every
 * 20ms. Before we take the sample, we check if the pen irq line is deasserted. If
 * it is deasserted, then this means the pen is up.
 */
irqreturn_t digi_isr(int irq, void *dev_id)
{
	//judge if it is the pen interrupt
	g_touch_enter_sleep = 0;
	touch_pan_clear_inter(); 
	touch_pan_disable_inter();
	pen_timer.expires = get_jiffies_64() + HZ/50 + 2*HZ/100;
	add_timer(&pen_timer);
	pen_timer_status = 1;
		
	return IRQ_HANDLED;
}

/*
*	Function Name: digi_tasklet_action
*	Input		 : data
*	Output		 : none
*	Description  : The tasklet function used in MX1.
*	
*
*/
static void digi_tasklet_action(unsigned long data)
{	
}

/**
 * @brief spi sampling timer callback function.
 */
static void digi_sam_callback(unsigned long data)
{
	u32 newX,newY;
	u32  maxDelay;
	
	maxDelay = 0;
	newX = TPNL_PEN_UPVALUE;

	if(pen_timer_status !=0)
		mod_timer(&pen_timer, get_jiffies_64() + HZ/50 + 2*HZ/100);
	
	if((SSR(GPIOE) & 0x00000400) != 0)//pen is up.
	{
		if(pen_timer_status!=0)//in use
		{
			//get Pen up
			add_x_y(0, 0, PENUP);
			// if there is a pen up, shall tell apps to handle the event, and it shall be data there
			if (!NODATA())
			{
				wake_up_interruptible(&digi_wait);
				if(ts_fasync)
					kill_fasync(&ts_fasync, SIGIO, POLL_IN);
			}
			
			//release timer
			spin_lock_irq(&pen_lock);
			pen_timer_status = 0;
			del_timer(&pen_timer);
			spin_unlock_irq(&pen_lock);

			touch_pan_set_neg_inter();
			touch_pan_clear_inter();
			touch_pan_enable_inter();
			g_touch_enter_sleep = 1;
		}
		return;
	}

	if((SSR(GPIOE) & 0x00000400) == 0)//pen is down.
	{
		//read data with MAX delay from ADC
		while((newX == TPNL_PEN_UPVALUE)&&(maxDelay <= 100))
		{
			touch_pan_read_data(&newX, &newY);
			maxDelay++;
		}
		if(newX == TPNL_PEN_UPVALUE)
			return;
		//add element into buffer
		add_x_y(newX, newY, PENDOWN);
		wake_up_interruptible(&digi_wait);
		
		if(ts_fasync)
			kill_fasync(&ts_fasync, SIGIO, POLL_IN);

		return;	
	}
	else
	{
		FAILED(" error to call me \n");
		return;
	}

}

/*
 * @brief Initialize SPI device.
 */
void spi_init(void)
{
	int datarate;
	int cs;
	int pdiv2;
#ifndef CONFIG_FRQ_350
	pdiv2 = 0x5; //6
#else
	pdiv2 = 0x8;//9
#endif
	cs = 0x0;             //ss0
	//cspi1_mosi--pd31 output
	//cspi1_miso--pd30 input
	//cspi1_sclk--pd29 output
	//cspi1_ss0 --pd28,output it connect with pen_cs_b
	GIUS(GPIOD) &= 0x0fffffff;
	GPR(GPIOD) &= 0x0fffffff;
	DDIR(GPIOD) |= 0xb0000000;
	DDIR(GPIOD) &= 0xbfffffff;
	
	PCDR1 &= ~0x3f00;
	PCDR1 |= pdiv2<<8;	//change to 0x0500 (133 MHz/6).
	udelay(100);


	//reset spi1
	CSPI_RESET(1) = 0x00000001;
	udelay(200);      //wait
	// use 32.768kHz
	CSPI_PERIOD(1) = 0x00008000;

	CSPI_CONTROL(1) &= ~0xffc000;
	CSPI_CONTROL(1) |= (cs<<19);
	datarate=16;
	CSPI_CONTROL(1) |= (datarate<<14);
	CSPI_CONTROL(1) |= 0x00000c07;//ss low active, 8 bit transfer,master mode
	
}

static int spi_tx_fifo_empty(void)
{
	return (CSPI_INT(1) & SPI_TE_INT_BIT);
}


static int spi_rx_fifo_data_ready(void)
{
	return (CSPI_INT(1) & SPI_RR_INT_BIT);
}

static unsigned int spi_exchange_data(unsigned int dataTx)
{
	while(!spi_tx_fifo_empty());

	CSPI_TXDATA(1)  = dataTx;			//transfer data
	CSPI_CONTROL(1) |= SPI_XCH_BIT;		// exchange data
   
	while(!spi_rx_fifo_data_ready()){;}
	return CSPI_RXDATA(1);
}
	

/**
 * @brief Verify if the inode is a correct inode
 * @param pInode inode to be check.
 * @return	@li	Zero on success.
 * 		@li	None-zero on failure.
 */
 static int check_device(struct inode *pInode)
{
	int minor;
	dev_t dev = pInode->i_rdev;
	
	if( MAJOR(dev) != g_digi_major)
		return -1;
	
	minor = MINOR(dev);
	
	if ( minor < MAX_ID )
		return minor;
	else
		return -1;
}

/**************************************************
Misc functions for buffer
***************************************************/
/* local buffer for store data
 * a new feature is read block, assume the buffer is a continuous buffer
 * so, if there is enough data, read data from current position to 
 * buffer end, then next time, from buffer head to data end. -- the buffer
 * is a loop buffer.
 */
ts_event_t * buffer;
#define BUFLEN		250
#define BUFSIZE 	(BUFLEN*sizeof(ts_event_t))
#define NEXTI(i)	{i=(i==BUFLEN-1)?0:i+1;}
#define GETNEXTI(i)	((i==BUFLEN-1)?0:i+1)
#define BLKEND(i)	((i)==0)

#define DSIZE()		((wptr<rptr)?(BUFLEN-rptr):(wptr-rptr))

void spi_flush_fifo(void)
{
	u32 i,j;
	for(i=0;i<8;i++)
	{
		if(CSPI_INT(1)&SPI_RR_INT_BIT)
		j = CSPI_RXDATA(1);
	}
}
/*
 * @brief Init the loop buffer for store data read out from spi FIFO and shall be copied to user.
 * @return	@li	Zero on success.
 * 		@li	None-zero on failure.
 */
static int init_buf(void)
{
	buffer = (ts_event_t*) vmalloc(BUFSIZE);	
	if(!buffer)
		return -1;

	rptr = wptr = 0;
	return 1;
}

/*
 * @brief Add pen data to buffer
 */
static void add_x_y(unsigned x, unsigned y, unsigned flag)
{
	
	if (GETNEXTI(wptr) == rptr)
		return;
	buffer[wptr].x = x;
	buffer[wptr].y = y;

	buffer[wptr].pressure = flag;

	NEXTI(wptr);	// goto next wptr
}

/**
 * @brief read block data
 *
 * Read a block of 'touch' data from buffer, read count shall less
 * than size,but shall be as more as possible. the data shall not really copy
 * to upper layer,untill copy_to_user is invoked.
 *
 * @param block Block from which we read data.
 * @param size  Size of the data to be read.
 * @return count of data size.
 */
//assume continuous buffer
static int get_block(ts_event_t * * block, int size)
{
	int cnt, rd;
	unsigned long flags;
	ts_event_t *p;
	
	if(NODATA())
		return 0;
		
	cnt = 0;
	local_save_flags(flags);
	local_irq_disable();
	
	//critical section: get the actual size of data need read
	if(DSIZE()*sizeof(ts_event_t) >= size)
		rd = size/sizeof(ts_event_t);
	else
		rd = DSIZE();
	
	* block = p = get_data();
	cnt ++;
	
	while(p && (cnt < rd))
	{
		if(rptr == 0)
			break;
		p = get_data();
		cnt ++;
	}
	
	local_irq_restore(flags);
	return (cnt)*sizeof(ts_event_t);
	
}

/**
 * @brief Read a 'touch' event from data buffer
 * @return ts_event_t: a 'touch' event data format
 */
// no really read out the data, so no copy
static ts_event_t* get_data(void)
{
	ts_event_t *data;
	
	if(NODATA())
		return NULL;

	data = & (buffer[rptr]);

	TRACE("*** Read - x: 0x%04x, y: 0x%04x, rptr: 0x%04x, wptr: 0x%04x\n", data->x, data->y, (int)rptr, (int)wptr);

	NEXTI(rptr);
	return data;
}

/**
 * @brief Release resource when close the inode
 * @param inode 
 * @param filp
 * @return	@li	Zero on success
 * 		@li	None-zero on failure
 */
int digi_release(struct inode * inode, struct file * filp)
{
	//printk("digi_release: ----\n");
	g_touch_enter_sleep = 1;
	digi_fasync(-1, filp, 0);

	return 0;
}

/**
 * @brief Allocate resource when open the inode
 * @param inode
 * @param filp
 * @return	@li	Zero on success
 * 		@li	None-zero on failure
 */

int digi_open(struct inode * inode, struct file * filp)
{
      g_touch_enter_sleep = 1;
#ifdef CONFIG_XEN_PRIVILEGED_GUEST
	//Enable ipg clock input to the CSPI1 module
	PCCR0 |= 0x10;
#endif
	
	spi_flush_fifo();
	//enable ADC interrupt
	touch_pan_disable_inter();
	touch_pan_set_neg_inter();      //set default polarity
	touch_pan_clear_inter();
	touch_pan_enable_inter();
	
	pen_timer_status = 0;
	
	return 0;
}

/**
 * @brief Provide fasync functionality for select system call
 * @param fd
 * @param filp
 * @param mode
 * @return	@li	Zero on success
 * 		@li	None-zero on failure
 */

static int digi_fasync(int fd, struct file *filp, int mode)
{
	/* TODO TODO put this data into file private data */
	int minor = check_device( filp->f_dentry->d_inode);
	if ( minor == - 1)
		return -ENODEV;
	return( fasync_helper(fd, filp, mode, &ts_fasync) );
}

/**
 * @brief ioctl for this device driver
 * @param inode
 * @param filp
 * @param cmd Command for ioctl
 * @param arg Parameter for command
 * @return	@li	Zero on success
 * 		@li	None-zero on failure
 */

int digi_ioctl(struct inode * inode, 
		struct file *filp,
		unsigned int cmd , 
		unsigned long arg)
{
	int ret = -EIO;
	int minor;
	
	
	minor = check_device( inode );
	if ( minor == - 1)
	{
		TRACE("bad minor\n");
		return -ENODEV;
	}
	
	TRACE("minor=%08x cmd=%d\n",minor,cmd);
	
	return ret;
}

/**
 * @brief Support poll and select
 * @param filp
 * @param wait
 * @return	@li	Zero on success
 * 		@li	None-zero on failure
 */
unsigned int digi_poll(struct file * filp, struct poll_table_struct * wait)
{
	int minor;
	
	minor = check_device( filp->f_dentry->d_inode);
	if ( minor == - 1)
		return -ENODEV;

	poll_wait(filp, &digi_wait, wait);
	
	return ( NODATA() ) ? 0 : (POLLIN | POLLRDNORM);
}

/**
 * @brief Read device driver
 * @param filp The file to be read
 * @param buf Data buffer
 * @param count Number of chars to be readed
 * @param l Offset of file
 * @return	@li	Zero on success
 * 		@li	None-zero on failure
 */

ssize_t digi_read(struct file * filp, char * buf, size_t count, loff_t * l)
{       
	int nonBlocking = filp->f_flags & O_NONBLOCK;
	int minor;
	ts_event_t *ev;
	int cnt=0;

	// printk("\n***digi_read: ----\n");
	minor = check_device( filp->f_dentry->d_inode );
	if ( minor == - 1){
		return -ENODEV;
	}
	
	if( nonBlocking )
	{
		if( !NODATA() ) 
		{
			/* returns length to be copied otherwise errno -Exxx */
			cnt = get_block(&ev, count) ;
			if(cnt > count){
				TRACE(KERN_ERR"Error read spi buffer\n");
				return -EINVAL;
			}

			copy_to_user(buf, (char*)ev, cnt );
			return cnt;
		}
		else{
			return -EINVAL;
		}
	}
    else
	{ 
		/* check , when woken, there is a complete event to read */
		while(1){
			if( !NODATA() )
			{
				cnt = get_block(&ev, count);
				if(cnt > count){
					TRACE(KERN_ERR"Error read spi buffer\n");
					return -EINVAL;
				}
				/* returns length to be copied otherwise errno -Exxx */

				copy_to_user(buf, (char*)ev, cnt);
				return cnt;
			}
			else
			{
				interruptible_sleep_on(&digi_wait);
				if(signal_pending(current)){
					return -ERESTARTSYS;
				}
			}
		}
	}
}

/**
 * @brief Initialize SPI device
 *
 * In this function should initialize the SPI device as well as the external
 * touch pannel hardware, for MX21 it is AD7873m.
 * @return	@li	Zero on success
 * 		@li	None-zero on failure
 */

static int __init digi_init(void)
{
	int tmp;
	int i;
	
	init_timer(&pen_timer);
	pen_timer.function = digi_sam_callback;
	
	tasklet_init(&digi_tasklet, digi_tasklet_action,(unsigned long)0);	

#ifdef LCD_USE_DEVFS
	tmp = register_chrdev(MAJOR_NUM, MODULE_DIGI, &g_digi_fops);
 	if ( tmp < 0 )
 	{
		TRACE("%s driver: Unable to register driver\n",MODULE_DIGI);
		return -ENODEV;
	}
	g_digi_major = MAJOR_NUM;
#else
	tmp = misc_register(&g_digi_dev);
    if (tmp) {
        TRACE("%s: cannot register misc device\n", MODULE_DIGI);
		return -ENODEV;
    }
    g_digi_major = 10;
#endif

	//printk("%s:%d\n", __FUNCTION__, __LINE__);
//INT_UART3,TPNL_IRQ
	tmp = request_irq(IRQ_GPIOE(10),
			(void *)digi_isr,
			TPNL_INTR_MODE,
			MODULE_DIGI,
			MODULE_DIGI);

	//printk("%s:%d\n", __FUNCTION__, __LINE__);
	if (tmp)
	{
		printk("digi_init:cannot init major= %d irq=%d\n", MAJOR_NUM, TPNL_IRQ);
#ifdef LCD_USE_DEVFS
        	unregister_chrdev(g_digi_major, MODULE_DIGI);
#else
			 misc_deregister(&g_digi_dev);
#endif
      		return -1;
	}

	
	//printk("%s:%d\n", __FUNCTION__, __LINE__);
	//init SPI buffer
	if(-1 == init_buf()){
		TRACE("digi_init: cannot init spi buffer, exit\n");

		free_irq(TPNL_IRQ,MODULE_DIGI);

#ifdef LCD_USE_DEVFS
        	unregister_chrdev(g_digi_major, MODULE_DIGI);
#else
			 misc_deregister(&g_digi_dev);
#endif
        	disable_irq(TPNL_IRQ);
		return -1;
	}
	init_waitqueue_head(&digi_wait);
	
#ifdef CONFIG_XEN_PRIVILEGED_GUEST
	HYPERVISOR_do_set_HID_irq(IRQ_GPIOE(10));
	//Enable ipg clock input to the CSPI1 module
	PCCR0 |= 0x10;
	
	//printk("%s:%d\n", __FUNCTION__, __LINE__);
	//first init touch pannel

	touch_pan_set_inter();//set pen IRQ 
	touch_pan_disable_inter();//mask pen interrupt
	//printk("%s:%d\n", __FUNCTION__, __LINE__);
	//init pen touch pannel in GPIO,disabled
	//init SPI module
	spi_init();
	touch_pan_init();
	
	//printk("%s:%d\n", __FUNCTION__, __LINE__);

	touch_pan_enable();//enable touch pannel
	//printk("%s:%d\n", __FUNCTION__, __LINE__);

      spi_exchange_data(0x90);
      spi_exchange_data(0x00);
      spi_exchange_data(0xD0);
      spi_exchange_data(0x00);
      spi_exchange_data(0x00);

	//printk("%s:%d\n", __FUNCTION__, __LINE__);
      touch_pan_disable();

	//printk("%s:%d\n", __FUNCTION__, __LINE__);
      //Delay for A/D Converter to be stable 
      for(i=0;i<50000;i++);  
  #endif
 
	printk("Pen Driver 0.4.0, Motorola SPS-Suzhou\n");
	return 0;
}

/**
 * @brief Cleanup the interrupt and devfs register
 */

static void __exit digi_cleanup(void)
{
	//Do some cleanup work

	free_irq(TPNL_IRQ, MODULE_DIGI);

	tasklet_kill(&digi_tasklet);
#ifdef LCD_USE_DEVFS
	unregister_chrdev(g_digi_major, MODULE_DIGI);
    #else
	misc_deregister(&g_digi_dev);
#endif 
}

module_init(digi_init);
module_exit(digi_cleanup);

MODULE_LICENSE("GPL");

//>>>>>Body
