/*
 *
 * Map driver for the Cogent CSBxxx boards.
 *
 * Author:	Bill Gatliff
 * Copyright:	(C) 2005 Bill Gatliff <bgat@billgatliff.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

#include <linux/module.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/dma-mapping.h>
#include <linux/mtd/mtd.h>
#include <linux/mtd/map.h>
#include <linux/mtd/partitions.h>
#include <asm/io.h>
#include <asm/hardware.h>

#define MTDID "flash00"
#define MSG_PREFIX "mx2ads: "

#if defined(CONFIG_MACH_MX2ADS)
#define WINDOW_ADDR 0xc8000000
#define WINDOW_SIZE 0x02000000
#else
#error TODO: the MTD map you need goes here...
#endif

#define NB_OF(a) (sizeof(a)/sizeof(*(a)))

static void csbxxx_map_inval_cache(struct map_info *map, unsigned long from, ssize_t len)
{
	consistent_sync((char *)map->cached + from, len, DMA_FROM_DEVICE);
}

//added by mikael
/* older boards: 2x AMD 29BDS128 parts: */
static struct mtd_partition mx2ads_AMD29BDS128_partitions[] =
 {
   {
     name:		"bootloader",
     size:		0x100000,	/* 1MB */
     offset:		0x00000000,
     mask_flags:	MTD_WRITEABLE,
   }, 
   {
     name:		"Master Boot Block",
     size:		(256 * 1024), /* 256KB */
     offset:		MTDPART_OFS_APPEND,
   }, 
   {
     name:		"Secure Partition #1",
     size:		(768 * 1024), /* 768KB */
     offset:		MTDPART_OFS_APPEND,
   }, 
   {
     name:		"Secure Partition #2",
     size:		(512 * 1024), /* 512KB */
     offset:		MTDPART_OFS_APPEND,
   }, 
   {
     name:		"Secure Partition #3",
     size:		(512 * 1024), /* 512KB */
     offset:		MTDPART_OFS_APPEND,
   }, 
   {
     name:		"Dom0 Root Filesystem", /* 16MB */
     size:		MTDPART_SIZ_FULL,
     offset:		MTDPART_OFS_APPEND,
   }
};


/* RoHS boards: 2x Spansion S29WS128N parts: */
static struct mtd_partition mx2ads_S29WS128N_partitions[] =
  {
     {
		 name:			"bootloader",
		 size:			0x00100000,
		 offset:		0x00000000,
		 mask_flags:	MTD_WRITEABLE,
	 },
	 {
		 name:			"Kernel Image Partition",
		 size:			0x00400000,
		 offset:		0x00100000,
	 },
	 {
		 name:			"Dom0 Root Filesystem",
		 size:			0x01B00000,
		 offset:		0x00500000,
	 }
  };		
//end of addition

static struct map_info csbxxx_map = {
    .size = WINDOW_SIZE,
    .phys = WINDOW_ADDR,
    .inval_cache = csbxxx_map_inval_cache,
    .bankwidth = 4,
    .name = MTDID,
};

/* default map definition,
   generally overridden on the command line */
static struct mtd_partition csbxxx_partitions[] = {
#if defined(CONFIG_MACH_MX2ADS)
	{
		name:		"bootloader",
		size:		(80 * 1024),		/* 64K for blob, 16K for params */
		offset:		0x00000000,
		mask_flags:	MTD_WRITEABLE,		/* this makes it read only */
	}, 
	{
		name:		"kernel",
		size:		(4016 * 1024),
		offset:		MTDPART_OFS_APPEND,
		mask_flags:	MTD_WRITEABLE,		/* this makes it read only */
	}, 
	{
		name:		"file system",
		size:		MTDPART_SIZ_FULL,
		offset:		MTDPART_OFS_APPEND,
	}

#else
  {
    .name = "uMON flash",
    .size = WINDOW_SIZE,
    .mask_flags = MTD_WRITEABLE  /* force read-only */
  }
#endif
};

static const char *probes[] = { "cmdlinepart", NULL };

static struct mtd_info *mymtd = 0;
static int mtd_parts_nb = 0;
static struct mtd_partition *mtd_parts = 0;

static int __init init_csbxxx(void)
{
	int ret = 0;
	const char *part_type = 0;

	csbxxx_map.virt = ioremap(csbxxx_map.phys, WINDOW_SIZE);
	if (!csbxxx_map.virt) {
		printk(KERN_WARNING "Failed to ioremap %s, MTD disabled\n", csbxxx_map.name);
		ret = -ENOMEM;
		goto err;
	}
	csbxxx_map.cached = ioremap_cached(csbxxx_map.phys, WINDOW_SIZE);
	if (!csbxxx_map.cached)
		printk(KERN_WARNING "Failed to ioremap cached %s\n", csbxxx_map.name);

	simple_map_init(&csbxxx_map);
  
	printk(KERN_NOTICE "Probing %s at physical address 0x%08lx (%d-bit bankwidth)\n",
		csbxxx_map.name, csbxxx_map.phys, csbxxx_map.bankwidth * 8);
  
	mymtd = do_map_probe("cfi_probe", &csbxxx_map);
  
	if (!mymtd)
		goto err;

	mymtd->owner = THIS_MODULE;

//added by mikael
	char *flash_chip;

 	/* need to change "parts", based on which flash part we have */
	if (mymtd->erasesize == 0x20000) {
		mtd_parts    = mx2ads_AMD29BDS128_partitions;
		mtd_parts_nb = NB_OF(mx2ads_AMD29BDS128_partitions);
		flash_chip = "AMD 29BDS128";
	} else {
		mtd_parts    = mx2ads_S29WS128N_partitions;
		mtd_parts_nb = NB_OF(mx2ads_S29WS128N_partitions);
		flash_chip = "Spansion S29WS128N";
	}
	printk(KERN_NOTICE "Detected %s flash chips.\n", flash_chip);
	printk(KERN_NOTICE "Erase block size is 0x%x \n", mymtd->erasesize);
//end of addition
  
	if (mtd_parts_nb > 0)
		part_type = "command line";
	else if (mtd_parts_nb == 0)
	{
		part_type = "static";
	}
	else 
		goto err;

	if (mtd_parts_nb == 0)
		printk(KERN_NOTICE MSG_PREFIX "no partition info available\n");
	else {
		printk(KERN_NOTICE MSG_PREFIX
			"using %s partition definition\n", part_type);
		add_mtd_partitions(mymtd, mtd_parts, mtd_parts_nb);
	}
  
	return 0;

err:
  if (csbxxx_map.virt)
    iounmap(csbxxx_map.virt);
  if (csbxxx_map.cached)
    iounmap(csbxxx_map.cached);
  if (!ret)
    ret = -EIO;

  return ret;
}

static void __exit cleanup_csbxxx(void)
{
  if (!mymtd)
    return;

  del_mtd_partitions(mymtd);
  
  map_destroy(mymtd);
  iounmap((void *)csbxxx_map.virt);
  if (csbxxx_map.cached)
    iounmap(csbxxx_map.cached);
}

module_init(init_csbxxx);
module_exit(cleanup_csbxxx);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Bill Gatliff <bgat@billgatliff.com>");
MODULE_DESCRIPTION("MTD map driver for Cogent CSBXXX");
