/*
 *  linux/drivers/video/imxfb.c
 *
 *  Freescale i.MX Frame Buffer device driver
 *
 *  Copyright (C) 2004 Sascha Hauer, Pengutronix
 *   Based on acornfb.c Copyright (C) Russell King.
 *
 * This file is subject to the terms and conditions of the GNU General Public
 * License.  See the file COPYING in the main directory of this archive for
 * more details.
 *
 * Please direct your questions and comments on this driver to the following
 * email address:
 *
 *	linux-arm-kernel@lists.arm.linux.org.uk
 */

//#define DEBUG 1

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/errno.h>
#include <linux/string.h>
#include <linux/interrupt.h>
#include <linux/slab.h>
#include <linux/fb.h>
#include <linux/delay.h>
#include <linux/init.h>
#include <linux/ioport.h>
#include <linux/cpufreq.h>
#include <linux/device.h>
#include <linux/dma-mapping.h>
#include <linux/platform_device.h>

#include <asm/hardware.h>
#include <asm/io.h>
#include <asm/mach-types.h>
#include <asm/uaccess.h>
#include <asm/arch/imxfb.h>

/*
 * Complain if VAR is out of range.
 */
#define DEBUG_VAR 1

#include "imxfb_gw.h"
#include "dom_status_img.h"

static struct imxfbgw_rgb def_rgb_16 = {
	.red	= { .offset = 11,  .length = 5, },
	.green	= { .offset = 5,  .length = 6, },
	.blue	= { .offset = 0,  .length = 5, },
	.transp = { .offset = 0,  .length = 0, },
};

static struct imxfbgw_rgb def_rgb_8 = {
	.red	= { .offset = 0,  .length = 8, },
	.green	= { .offset = 0,  .length = 8, },
	.blue	= { .offset = 0,  .length = 8, },
	.transp = { .offset = 0,  .length = 0, },
};

#define GWX  100
#define GWY  30

struct fb_gwinfo {
    __u32 enabled;		/*!< Non-zero if graphic window is enabled */
    
    /* The fields below are valid only when graphic window is enabled */
    __u32 alpha_value;		/*!< Graphic window alpha value from 0 to 255*/
    
    __u32 ck_enabled;		/*!< Non-zero if graphic window color keying is enabled. */
    /* The fields ck_red, ck_green and ck_blue are valid only when graphic window and the color keying are enabled. */
    __u32 ck_red;			/*!< Red component of graphic window color keying. */
    __u32 ck_green;		/*!< Green component of graphic window color keying. */
    __u32 ck_blue;		/*!< Blue component of graphic window color keying. */
    
    __u32 xpos;			/*!< Graphic window x position. */
    __u32 ypos;			/*!< Graphic window y position. */
    __u32 vs_reversed;		/*!< Non-zero if graphic window vertical scan in reverse direction. */
    __u32 priority;		/*!< To avoid LCDC under-run error with higher priority. */
};

static struct fb_gwinfo g_gwinfo;	//graphic window structure


static int imxfbgw_activate_var(struct fb_var_screeninfo *var, struct fb_info *info);

static inline u_int chan_to_field(u_int chan, struct fb_bitfield *bf)
{
	chan &= 0xffff;
	chan >>= 16 - bf->length;
	return chan << bf->offset;
}

static int
imxfbgw_setpalettereg(u_int regno, u_int red, u_int green, u_int blue,
		       u_int trans, struct fb_info *info)
{
	struct imxfbgw_info *fbi = info->par;
	u_int val, ret = 1;

#define CNVT_TOHW(val,width) ((((val)<<(width))+0x7FFF-(val))>>16)
	if (regno < fbi->palette_size) {
		val = (CNVT_TOHW(red, 4) << 8) |
		      (CNVT_TOHW(green,4) << 4) |
		      CNVT_TOHW(blue,  4);

		LCDC_PALETTE(regno) = val;
		ret = 0;
	}
	return ret;
}

static int
imxfbgw_setcolreg(u_int regno, u_int red, u_int green, u_int blue,
		   u_int trans, struct fb_info *info)
{
	struct imxfbgw_info *fbi = info->par;
	unsigned int val;
	int ret = 1;

	/*
	 * If inverse mode was selected, invert all the colours
	 * rather than the register number.  The register number
	 * is what you poke into the framebuffer to produce the
	 * colour you requested.
	 */
	if (fbi->cmap_inverse) {
		red   = 0xffff - red;
		green = 0xffff - green;
		blue  = 0xffff - blue;
	}

	/*
	 * If greyscale is true, then we convert the RGB value
	 * to greyscale no mater what visual we are using.
	 */
	if (info->var.grayscale)
		red = green = blue = (19595 * red + 38470 * green +
					7471 * blue) >> 16;

	switch (info->fix.visual) {
	case FB_VISUAL_TRUECOLOR:
		/*
		 * 12 or 16-bit True Colour.  We encode the RGB value
		 * according to the RGB bitfield information.
		 */
		if (regno < 16) {
			u32 *pal = info->pseudo_palette;

			val  = chan_to_field(red, &info->var.red);
			val |= chan_to_field(green, &info->var.green);
			val |= chan_to_field(blue, &info->var.blue);

			pal[regno] = val;
			ret = 0;
		}
		break;

	case FB_VISUAL_STATIC_PSEUDOCOLOR:
	case FB_VISUAL_PSEUDOCOLOR:
		ret = imxfbgw_setpalettereg(regno, red, green, blue, trans, info);
		break;
	}

	return ret;
}

/*
 *  imxfb_check_var():
 *    Round up in the following order: bits_per_pixel, xres,
 *    yres, xres_virtual, yres_virtual, xoffset, yoffset, grayscale,
 *    bitfields, horizontal timing, vertical timing.
 */
static int
imxfbgw_check_var(struct fb_var_screeninfo *var, struct fb_info *info)
{
	struct imxfbgw_info *fbi = info->par;
	int rgbidx;

	if (var->xres < GW_MIN_XRES)
		var->xres = GW_MIN_XRES;
	if (var->yres < GW_MIN_YRES)
		var->yres = GW_MIN_YRES;
	if (var->xres > fbi->max_xres)
		var->xres = fbi->max_xres;
	if (var->yres > fbi->max_yres)
		var->yres = fbi->max_yres;
	var->xres_virtual = max(var->xres_virtual, var->xres);
	var->yres_virtual = max(var->yres_virtual, var->yres);

	pr_debug("gw:var->bits_per_pixel=%d\n", var->bits_per_pixel);
	switch (var->bits_per_pixel) {
	case 16:
		rgbidx = GW_RGB_16;
		break;
	case 8:
		rgbidx = GW_RGB_8;
		break;
	default:
		rgbidx = GW_RGB_16;
	}

	/*
	 * Copy the RGB parameters for this display
	 * from the machine specific parameters.
	 */
	var->red    = fbi->rgb[rgbidx]->red;
	var->green  = fbi->rgb[rgbidx]->green;
	var->blue   = fbi->rgb[rgbidx]->blue;
	var->transp = fbi->rgb[rgbidx]->transp;

	pr_debug("gw:RGBT length = %d:%d:%d:%d\n",
		var->red.length, var->green.length, var->blue.length,
		var->transp.length);

	pr_debug("gw:RGBT offset = %d:%d:%d:%d\n",
		var->red.offset, var->green.offset, var->blue.offset,
		var->transp.offset);

	return 0;
}

/*
 * imxfb_set_par():
 *	Set the user defined part of the display for the specified console
 */
static int imxfbgw_set_par(struct fb_info *info)
{
	struct imxfbgw_info *fbi = info->par;
	struct fb_var_screeninfo *var = &info->var;

	pr_debug("set_par\n");

	if (var->bits_per_pixel == 16)
		info->fix.visual = FB_VISUAL_TRUECOLOR;
	else if (!fbi->cmap_static)
		info->fix.visual = FB_VISUAL_PSEUDOCOLOR;
	else {
		/*
		 * Some people have weird ideas about wanting static
		 * pseudocolor maps.  I suspect their user space
		 * applications are broken.
		 */
		info->fix.visual = FB_VISUAL_STATIC_PSEUDOCOLOR;
	}

	info->fix.line_length = var->xres_virtual *
				  var->bits_per_pixel / 8;
	fbi->palette_size = var->bits_per_pixel == 8 ? 256 : 16;

	imxfbgw_activate_var(var, info);

	return 0;
}

static void imxfbgw_enable_controller(struct imxfbgw_info *fbi)
{
	pr_debug("gw:Enabling LCD controller\n");

      g_gwinfo.enabled = 0;
      g_gwinfo.alpha_value = 90;
      g_gwinfo.ck_enabled = 0;
      g_gwinfo.ck_red = 0x000000;
      g_gwinfo.ck_green = 0x000000;
      g_gwinfo.ck_blue = 0x000000;
      g_gwinfo.xpos = 70;
      g_gwinfo.ypos = 289;
      g_gwinfo.vs_reversed = 0;
	
	unsigned long lgwcr = 0x00400000;	//Graphic window control register: enable graphic window
      	
	//FUNC_START

	g_gwinfo.enabled = 1;

#ifdef CONFIG_XEN_PRIVILEGED_GUEST	
	//Graphic window start address register
	_reg_LCDC_LGWSAR = fbi->screen_dma;
	printk("[SY]GW SAR = %0x\n", fbi->screen_dma);
	
	//Graphic window size register
	_reg_LCDC_LGWSR = ((GWX / 16) << 20) + GWY;
	//Graphic window virtual page width register
	_reg_LCDC_LGWVPWR = GWX / 2;
	//Graphic window position register
	_reg_LCDC_LGWPR = ((g_gwinfo.xpos & 0x000003FF) << 16) | (g_gwinfo.ypos & 0x000003FF);
	//Graphic window & main window DMA control register
	_reg_LCDC_LDCR = 0x000B000F;
	_reg_LCDC_LGWDCR = 0x000B000F;

	//Graphic window control register
       
	lgwcr |= (g_gwinfo.alpha_value & 0x000000FF) << 24; //alpha value

	lgwcr |= (g_gwinfo.alpha_value & 0x000000FF) << 24;	//alpha value
	lgwcr |= g_gwinfo.ck_enabled ? 0x00800000 : 0;	//color keying enabled?
	lgwcr |= g_gwinfo.vs_reversed ? 0x00200000 : 0;	//reverse vertical scan?
	//Color keying value
	//Todo: assume always use RGB565
	lgwcr |= (g_gwinfo.ck_red & 0x0000003F) << 12;	//color keying red
	lgwcr |= (g_gwinfo.ck_green & 0x0000003F) << 6;	//color keying green
	lgwcr |= g_gwinfo.ck_blue & 0x0000003F;		//color keying blue
	//Graphic window can only be enabled while the HCLK to the LCDC
	//is disabled. Once enabled this first time it can subsequently
	//be disabled and enabled without turning off the HCLK.

	/* Enable the LCDC Clock */
        PCCR0 &= ~PCCR0_HCLK_LCDC_EN;

	_reg_LCDC_LGWCR = lgwcr;

       PCCR0 |= PCCR0_HCLK_LCDC_EN;
#endif	
}

static void imxfbgw_disable_controller(struct imxfbgw_info *fbi)
{
	pr_debug("gw:Disabling LCD controller\n");

	if(fbi->backlight_power)
		fbi->backlight_power(0);
	if(fbi->lcd_power)
		fbi->lcd_power(0);
#ifdef CONFIG_XEN_PRIVILEGED_GUEST
	LCDC_RMCR = 0;
#endif
}

static int imxfbgw_blank(int blank, struct fb_info *info)
{
	struct imxfbgw_info *fbi = info->par;

	pr_debug("gw:imxfb_blank: blank=%d\n", blank);

	switch (blank) {
	case FB_BLANK_POWERDOWN:
	case FB_BLANK_VSYNC_SUSPEND:
	case FB_BLANK_HSYNC_SUSPEND:
	case FB_BLANK_NORMAL:
		imxfbgw_disable_controller(fbi);
		break;

	case FB_BLANK_UNBLANK:
		imxfbgw_enable_controller(fbi);
		break;
	}
	return 0;
}

static struct fb_ops imxfbgw_ops = {
	.owner		= THIS_MODULE,
	.fb_check_var	= imxfbgw_check_var,
	.fb_set_par	= imxfbgw_set_par,
	.fb_setcolreg	= imxfbgw_setcolreg,
	.fb_fillrect	= cfb_fillrect,
	.fb_copyarea	= cfb_copyarea,
	.fb_imageblit	= cfb_imageblit,
	.fb_blank	= imxfbgw_blank,
};

/*
 * imxfb_activate_var():
 *	Configures LCD Controller based on entries in var parameter.  Settings are
 *	only written to the controller if changes were made.
 */
static int imxfbgw_activate_var(struct fb_var_screeninfo *var, struct fb_info *info)
{
	struct imxfbgw_info *fbi = info->par;
	pr_debug("gw:var: xres=%d hslen=%d lm=%d rm=%d\n",
		var->xres, var->hsync_len,
		var->left_margin, var->right_margin);
	pr_debug("gw:var: yres=%d vslen=%d um=%d bm=%d\n",
		var->yres, var->vsync_len,
		var->upper_margin, var->lower_margin);

#if DEBUG_VAR
	if (var->xres < 16        || var->xres > 1024)
		printk(KERN_ERR "gw:%s: invalid xres %d\n",
			info->fix.id, var->xres);
	if (var->hsync_len < 1    || var->hsync_len > 64)
		printk(KERN_ERR "gw:%s: invalid hsync_len %d\n",
			info->fix.id, var->hsync_len);
	if (var->left_margin > 255)
		printk(KERN_ERR "gw:%s: invalid left_margin %d\n",
			info->fix.id, var->left_margin);
	if (var->right_margin > 255)
		printk(KERN_ERR "gw:%s: invalid right_margin %d\n",
			info->fix.id, var->right_margin);
	if (var->yres < 1 || var->yres > 511)
		printk(KERN_ERR "gw:%s: invalid yres %d\n",
			info->fix.id, var->yres);
	if (var->vsync_len > 100)
		printk(KERN_ERR "gw:%s: invalid vsync_len %d\n",
			info->fix.id, var->vsync_len);
	if (var->upper_margin > 63)
		printk(KERN_ERR "gw:%s: invalid upper_margin %d\n",
			info->fix.id, var->upper_margin);
	if (var->lower_margin > 255)
		printk(KERN_ERR "gw:%s: invalid lower_margin %d\n",
			info->fix.id, var->lower_margin);
#endif

	LCDC_HCR	= HCR_H_WIDTH(var->hsync_len) |
	                  HCR_H_WAIT_1(var->left_margin) |
			  HCR_H_WAIT_2(var->right_margin);

	LCDC_VCR	= VCR_V_WIDTH(var->vsync_len) |
	                  VCR_V_WAIT_1(var->upper_margin) |
			  VCR_V_WAIT_2(var->lower_margin);

	LCDC_SIZE	= SIZE_XMAX(var->xres) | SIZE_YMAX(var->yres);
	LCDC_PCR	= fbi->pcr;
	LCDC_PWMR	= fbi->pwmr;
	LCDC_LSCR1	= fbi->lscr1;

	return 0;
}

static void imxfbgw_setup_gpio(struct imxfbgw_info *fbi)
{
	int width;

	LCDC_RMCR	&= ~(RMCR_LCDC_EN | RMCR_SELF_REF);

	if( fbi->pcr & PCR_TFT )
		width = 16;
	else
		width = 1 << ((fbi->pcr >> 28) & 0x3);

#if defined(CONFIG_ARCH_IMX21) 
	switch(width) {
	case 16:
		imx_gpio_mode(PA21_PF_LD15);
		imx_gpio_mode(PA20_PF_LD14);
		imx_gpio_mode(PA19_PF_LD13);
		imx_gpio_mode(PA18_PF_LD12);
		imx_gpio_mode(PA17_PF_LD11);
		imx_gpio_mode(PA16_PF_LD10);
		imx_gpio_mode(PA15_PF_LD9);
		imx_gpio_mode(PA14_PF_LD8);
	case 8:
		imx_gpio_mode(PA13_PF_LD7);
		imx_gpio_mode(PA12_PF_LD6);
		imx_gpio_mode(PA11_PF_LD5);
		imx_gpio_mode(PA10_PF_LD4);
	case 4:
		imx_gpio_mode(PA9_PF_LD3);
		imx_gpio_mode(PA8_PF_LD2);
	case 2:
		imx_gpio_mode(PA7_PF_LD1);
	case 1:
		imx_gpio_mode(PA6_PF_LD0);
	}

	/* initialize GPIOs */
	imx_gpio_mode(PA5_PF_LSCLK);

	imx_gpio_mode(PA29_PF_FLM_VSYNC);
	imx_gpio_mode(PA28_PF_LP_HSYNC);
	imx_gpio_mode(PA31_PF_ACD_OE);

        /* Enable the LCDC Clock */
        PCCR0 |= PCCR0_HCLK_LCDC_EN;

#endif /* defined(CONFIG_ARCH_IMX) */

}

#ifdef CONFIG_PM
/*
 * Power management hooks.  Note that we won't be called from IRQ context,
 * unlike the blank functions above, so we may sleep.
 */
static int imxfbgw_suspend(struct device *dev, u32 state, u32 level)
{
	struct imxfbgw_info *fbi = dev_get_drvdata(dev);
	pr_debug("%s\n",__FUNCTION__);

	if (level == SUSPEND_DISABLE || level == SUSPEND_POWER_DOWN)
		imxfb_disable_controller(fbi);
	return 0;
}

static int imxfbgw_resume(struct device *dev, u32 level)
{
	struct imxfbgw_info *fbi = dev_get_drvdata(dev);
	pr_debug("gw:%s\n",__FUNCTION__);

	if (level == RESUME_ENABLE)
		imxfb_enable_controller(fbi);
	return 0;
}
#else
#define imxfbgw_suspend	NULL
#define imxfbgw_resume	NULL
#endif

static int __init imxfbgw_init_fbinfo(struct device *dev)
{
	struct imxfb_mach_info *inf = dev->platform_data;
	struct fb_info *info = dev_get_drvdata(dev);
	struct imxfbgw_info *fbi = info->par;

	pr_debug("gw:%s\n",__FUNCTION__);

	info->pseudo_palette = kmalloc( sizeof(u32) * 16, GFP_KERNEL);
	if (!info->pseudo_palette)
		return -ENOMEM;

	memset(fbi, 0, sizeof(struct imxfbgw_info));
	fbi->dev = dev;

	strlcpy(info->fix.id, GW_IMX_NAME, sizeof(info->fix.id));

	info->fix.type	= FB_TYPE_PACKED_PIXELS;
	info->fix.type_aux		= 0;
	info->fix.xpanstep		= 0;
	info->fix.ypanstep		= 0;
	info->fix.ywrapstep		= 0;
	info->fix.accel	= FB_ACCEL_NONE;

	info->var.nonstd		= 0;
	info->var.activate		= FB_ACTIVATE_NOW;
	info->var.height		= -1;
	info->var.width	= -1;
	info->var.accel_flags		= 0;
	info->var.vmode	= FB_VMODE_NONINTERLACED;

	info->fbops			= &imxfbgw_ops;
	info->flags			= FBINFO_FLAG_DEFAULT;
	info->pseudo_palette		= (fbi + 1);

	fbi->rgb[GW_RGB_16]		= &def_rgb_16;
	fbi->rgb[GW_RGB_8]			= &def_rgb_8;

	fbi->max_xres			= inf->xres;
	info->var.xres			= inf->xres;
	info->var.xres_virtual		= inf->xres;
	fbi->max_yres			= inf->yres;
	info->var.yres			= inf->yres;
	info->var.yres_virtual		= inf->yres;
	fbi->max_bpp			= inf->bpp;
	info->var.bits_per_pixel	= inf->bpp;
	info->var.pixclock		= inf->pixclock;
	info->var.hsync_len		= inf->hsync_len;
	info->var.left_margin		= inf->left_margin;
	info->var.right_margin		= inf->right_margin;
	info->var.vsync_len		= inf->vsync_len;
	info->var.upper_margin		= inf->upper_margin;
	info->var.lower_margin		= inf->lower_margin;
	info->var.sync			= inf->sync;
	info->var.grayscale		= inf->cmap_greyscale;
	fbi->cmap_inverse		= inf->cmap_inverse;
	fbi->pcr			= inf->pcr;
	fbi->lscr1			= inf->lscr1;
	fbi->pwmr			= inf->pwmr;
	fbi->lcd_power			= inf->lcd_power;
	fbi->backlight_power		= inf->backlight_power;
	info->fix.smem_len		= fbi->max_xres * fbi->max_yres *
					  fbi->max_bpp / 8;

	return 0;
}

/*
 *      Allocates the DRAM memory for the frame buffer.  This buffer is
 *	remapped into a non-cached, non-buffered, memory region to
 *      allow pixel writes to occur without flushing the cache.
 *      Once this area is remapped, all virtual memory access to the
 *      video memory should occur at the new region.
 */
static int __init imxfbgw_map_video_memory(struct fb_info *info)
{
	struct imxfbgw_info *fbi = info->par;

	fbi->map_size = PAGE_ALIGN(GWX*GWY*2);
	fbi->map_cpu = dma_alloc_writecombine(fbi->dev, fbi->map_size,
					&fbi->map_dma,GFP_KERNEL);

	if (fbi->map_cpu) {
		info->screen_base = fbi->map_cpu;
		fbi->screen_cpu = fbi->map_cpu;
		fbi->screen_dma = fbi->map_dma;
		info->fix.smem_start = fbi->screen_dma;

	}
	return fbi->map_cpu ? 0 : -ENOMEM;
}


int show_status_img_to_gw(int fg_dom)
{
	if( my_gw_ssa == NULL )
		return -1;
	if(fg_dom == 0)
		memcpy(my_gw_ssa, dom0_status_img, GWX*GWY*2);
	else if( fg_dom == 1)
		{
			memcpy(my_gw_ssa, dom1_status_img, GWX*GWY*2);
		}
	return 0;
}

#include <xen/fbsetif.h>
#ifndef CONFIG_XEN_PRIVILEGED_GUEST
extern fbsetif_ring_t *frontfb;
#else
fbsetif_ring_t *frontfb=NULL;
#endif
static int __init imxfbgw_probe(struct device *dev)
{
	struct imxfbgw_info *fbi;
	struct fb_info *info;
	struct imxfb_mach_info *inf;
	struct resource *res;
	int ret;
	my_gw_ssa = NULL;
	my_ssa_gw1 = NULL;
		

	printk("gw:i.MX Framebuffer driver\n");

	inf = dev->platform_data;
	if(!inf) {
		dev_err(dev,"gw:No platform_data available\n");
		return -ENOMEM;
	}

	info = framebuffer_alloc(sizeof(struct imxfbgw_info), dev);
	if(!info)
		return -ENOMEM;

	fbi = info->par;

	dev_set_drvdata(dev, info);

	ret = imxfbgw_init_fbinfo(dev);
	if( ret < 0 )
		goto failed_init;

	if (!inf->fixed_screen_cpu) {
		ret = imxfbgw_map_video_memory(info);
		if (ret) {
			dev_err(dev, "gw:Failed to allocate video RAM: %d\n", ret);
			ret = -ENOMEM;
			goto failed_map;
		}
	} else {
		/* Fixed framebuffer mapping enables location of the screen in eSRAM */
		fbi->map_cpu = inf->fixed_screen_cpu;
		fbi->map_dma = inf->fixed_screen_dma;
		info->screen_base = fbi->map_cpu;
		fbi->screen_cpu = fbi->map_cpu;
		fbi->screen_dma = fbi->map_dma;
		info->fix.smem_start = fbi->screen_dma;
	}

#ifdef CONFIG_XENOLINUX
	if(frontfb){
		frontfb->fb_ssa_gw1 = my_gw_ssa = fbi->screen_cpu;
		my_ssa_gw1 = fbi->screen_dma;
	}
	else{
	       my_gw_ssa = fbi->screen_cpu;
		my_ssa_gw1 = fbi->screen_dma;
	}
#endif


       // put satatus image to graphic window frame buffer
	#ifdef CONFIG_XEN_PRIVILEGED_GUEST
    		memcpy(my_gw_ssa, dom0_status_img, GWX*GWY*2);
	#else
        	memcpy(my_gw_ssa, dom1_status_img, GWX*GWY*2);
	#endif

	

	/*
	 * This makes sure that our colour bitfield
	 * descriptors are correctly initialised.
	 */
	imxfbgw_check_var(&info->var, info);

	ret = fb_alloc_cmap(&info->cmap, 1<<info->var.bits_per_pixel, 0);
	if (ret < 0)
		goto failed_cmap;

	imxfbgw_setup_gpio(fbi);

	imxfbgw_set_par(info);
	ret = register_framebuffer(info);
	if (ret < 0) {
		dev_err(dev, "gw:failed to register framebuffer\n");
		goto failed_register;
	}

	imxfbgw_enable_controller(fbi);

#if 0
	{
		unsigned int *addr = (void*) IMX_LCDC_BASE;
		int i;
		for(i=0;i<64;i++)
			pr_debug("LDCD: %p: %08x\n",addr+i,addr[i]);

	}
#endif

	return 0;

failed_register:
	fb_dealloc_cmap(&info->cmap);
failed_cmap:
	if (!inf->fixed_screen_cpu)
		dma_free_writecombine(dev,fbi->map_size,fbi->map_cpu,
		           fbi->map_dma);
failed_map:
	kfree(info->pseudo_palette);
failed_regs:
	release_mem_region(res->start, res->end - res->start);
failed_init:
	dev_set_drvdata(dev, NULL);
	framebuffer_release(info);
	return ret;
}

static int imxfbgw_remove(struct device *dev)
{
	struct platform_device *pdev = to_platform_device(dev);
	struct fb_info *info = dev_get_drvdata(dev);
	struct resource *res;

	res = platform_get_resource(pdev, IORESOURCE_MEM, 0);

	/* disable LCD controller */
	LCDC_RMCR &= ~RMCR_LCDC_EN;

	unregister_framebuffer(info);

	fb_dealloc_cmap(&info->cmap);
	kfree(info->pseudo_palette);
	framebuffer_release(info);

	release_mem_region(res->start, res->end - res->start + 1);
	dev_set_drvdata(dev, NULL);

	return 0;
}

void  imxfbgw_shutdown(struct device * dev)
{
	/* disable LCD Controller */
	LCDC_RMCR &= ~RMCR_LCDC_EN;
}

static struct device_driver imxfbgw_driver = {
	.name		= "imx-fb",
	.bus		= &platform_bus_type,
	.probe		= imxfbgw_probe,
	.suspend	= imxfbgw_suspend,
	.resume		= imxfbgw_resume,
	.remove		= imxfbgw_remove,
	.shutdown	= imxfbgw_shutdown,
};

extern struct device *imx21fb_dev;

int __init imxfbgw_init(void)
{
	imxfbgw_probe(imx21fb_dev);
}

static void __exit imxfbgw_cleanup(void)
{
	driver_unregister(&imxfbgw_driver);
}

module_init(imxfbgw_init);
module_exit(imxfbgw_cleanup);

MODULE_DESCRIPTION("Motorola i.MX framebuffer graphic window driver");
MODULE_AUTHOR("Sascha Hauer, Pengutronix");
MODULE_LICENSE("GPL");
