/*
 * /drivers/xen/fbsetback/interface.c
 *
 * Copyright (C) 2008 Samsung Electronics
 *          Dong-Hyuk lee  <dh5050.lee@samsung.com>
 *
 * Secure Xen on ARM architecture designed by Sang-bum Suh consists of
 * Xen on ARM and the associated access control.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public version 2 of License as published by
 * the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <xen/evtchn.h>
#include <linux/err.h>
#include <xen/driver_util.h>
#include <xen/interface/event_channel.h>
#include "common.h"

LIST_HEAD(frontfbsets);

void  dom_destroy_fbset(int domid)
{
	printk(" %d Domain framebuffer is disconnected!\n",domid);
	fbsetif_t* i;
	list_for_each_entry(i,&frontfbsets,list)
	{
		if(i->domid == domid)
		{
			list_del(&i->list);
			kfree(i);
			return;
		}	
	}
	return;
}

fbsetif_t *alloc_fbsetif(domid_t domid)
{
	fbsetif_t *fbsetif = kzalloc(sizeof(fbsetif_t),GFP_KERNEL);
	
	if (!fbsetif)
		return ERR_PTR(-ENOMEM);

	memset(fbsetif, 0, sizeof(fbsetif_t));
	
	fbsetif->domid = domid;
	fbsetif->status = DISCONNECTED;
	
	INIT_LIST_HEAD(&fbsetif->list);
	list_add_tail(&fbsetif->list, &frontfbsets);
	
	return fbsetif;
}

static int map_frontend_page(fbsetif_t *fbsetif, unsigned long shared_page)
{
	struct gnttab_map_grant_ref op;
	int ret;

	op.host_addr = (unsigned long)fbsetif->fbset_ring_area->addr;
	op.flags     = GNTMAP_host_map;
	op.ref       = shared_page;
	op.dom       = fbsetif->domid;
   /* the value of use should be same with the value of frontend's devicetype registerd in the xenstore */
	op.use       = 'v'^'f'^'b';

	lock_vm_area(fbsetif->fbset_ring_area);
	ret = HYPERVISOR_grant_table_op(GNTTABOP_map_grant_ref, &op, 1);
	unlock_vm_area(fbsetif->fbset_ring_area);
	BUG_ON(ret);

	if (op.status) {
		printk(" Grant table operation failure !\n");
		return op.status;
	}

	fbsetif->shmem_ref = shared_page;
	fbsetif->shmem_handle = op.handle;

	return 0;
}


void unmap_frontend_page_fbset(fbsetif_t *fbsetif)
{
	struct gnttab_unmap_grant_ref op;
	int ret;

	op.host_addr    = (unsigned long)fbsetif->fbset_ring_area->addr;
	op.handle       = fbsetif->shmem_handle;
	op.dev_bus_addr = 0;

	lock_vm_area(fbsetif->fbset_ring_area);
	ret = HYPERVISOR_grant_table_op(GNTTABOP_unmap_grant_ref, &op, 1);
	if(op.status) {
		printk("\nfbsetback: unmap_frontend_page failed\n");
	}
	unlock_vm_area(fbsetif->fbset_ring_area);
	BUG_ON(ret);
}



int fbsetif_map(fbsetif_t *fbsetif, unsigned long shared_page, unsigned int evtchn)
{
	int err;
	
	evtchn_op_t op = {
		.cmd = EVTCHNOP_bind_interdomain,
		.u.bind_interdomain.remote_dom = fbsetif->domid,
		.u.bind_interdomain.remote_port = evtchn, 
		/* the value of use should be same with the value of frontend's devicetype registerd in the xenstore */
		.u.bind_interdomain.use = 'v'^'f'^'b'};
	
	/* Already connected through? */
	if (fbsetif->irq)
		return 0;
	
	if ( (fbsetif->fbset_ring_area = alloc_vm_area(PAGE_SIZE)) == NULL )
		return -ENOMEM;

	err = map_frontend_page(fbsetif, shared_page);
	if (err) {
		free_vm_area(fbsetif->fbset_ring_area);
		return err;
	}
	
	err = HYPERVISOR_event_channel_op(&op);
	if (err) {
		unmap_frontend_page_fbset(fbsetif);
		free_vm_area(fbsetif->fbset_ring_area);
		return err;
	}
	
	fbsetif->evtchn = op.u.bind_interdomain.local_port;
		

	
	fbsetif->irq = bind_evtchn_to_irqhandler(fbsetif->evtchn, fbsetif_be_int, 0,
			"fbsetif-backend", fbsetif);
	
	return 0;
}

