/*
 * /drivers/xen/fbsetback/xenbus.c
 *
 * Copyright (C) 2008 Samsung Electronics
 *          Dong-Hyuk lee  <dh5050.lee@samsung.com>
 *
 * Secure Xen on ARM architecture designed by Sang-bum Suh consists of
 * Xen on ARM and the associated access control.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public version 2 of License as published by
 * the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <xen/xenbus.h>
#include "common.h"

#undef DPRINTK
#define DPRINTK(fmt, args...) \
    pr_debug("fbsetback/xenbus (%s:%d) " fmt ".\n", __FUNCTION__, __LINE__, ##args)

struct backend_info
{
	struct xenbus_device *dev;
	fbsetif_t *fbsetif;
	XenbusState frontend_state;
};

static int connect(struct backend_info *);

static int fbsetback_remove(struct xenbus_device *dev)
{
	DPRINTK("");

	struct backend_info *be = dev->data;
	if (be->fbsetif) {
		if(be->fbsetif->status == CONNECTED){
			unmap_frontend_page_fbset(be->fbsetif);
			free_vm_area(be->fbsetif->fbset_ring_area);
		be->fbsetif->status = DISCONNECTED; 
			
		}
		if (be->fbsetif->irq) unbind_from_irqhandler(be->fbsetif->irq, be->fbsetif);
		dom_destroy_fbset(be->fbsetif->domid);
		be->fbsetif = NULL;
	}
	
	kfree(be);
	dev->data = NULL;
	return 0;
}

static int fbsetback_probe(struct xenbus_device *dev,
		const struct xenbus_device_id *id)
{
	int err;

	printk("fbsetback_probe called\n");
	
	struct backend_info *be = kzalloc(sizeof(struct backend_info),GFP_KERNEL);
	if (!be) {
		xenbus_dev_fatal(dev, -ENOMEM, "allocating backend structure");
		return -ENOMEM;
	}

	be->dev = dev;
	dev->data = be;
	be->fbsetif = alloc_fbsetif(dev->otherend_id);

	printk("fbsetback_probe: be->fbsetif = 0x%x\n",be->fbsetif);
	
	err = xenbus_switch_state(dev, XenbusStateInitWait);

	if (err) {
		printk("fbsetback_probe: switch_state failed");
		goto fail;
	}

	return 0;

fail:
	DPRINTK("failed");
	fbsetback_remove(dev);
	return err;
}

static void frontend_changed(struct xenbus_device *dev,
			     XenbusState frontend_state)
{
	struct backend_info *be = dev->data;
    int err;

	be->frontend_state = frontend_state;

	switch (frontend_state) {
	case XenbusStateInitialising:
		 break;
	case XenbusStateInitialised:
	case XenbusStateConnected:
		if (dev->state == XenbusStateConnected){
			break;
		}
		err = connect(be);

		if(err)
			break;

		be->fbsetif->status = CONNECTED;
		break;

	case XenbusStateClosing:
		xenbus_switch_state(dev, XenbusStateClosing);
		break;

	case XenbusStateClosed:
		device_unregister(&dev->dev);
		break;

	case XenbusStateUnknown:
	case XenbusStateInitWait:
	default:
		xenbus_dev_fatal(dev, -EINVAL, "saw state %d at frontend",
				 frontend_state);
		break;
	}
}

static int connect(struct backend_info *be)
{
	struct xenbus_device *dev = be->dev;
	unsigned long ring_ref;
	unsigned int evtchn;
	int err;
	
	DPRINTK("%s", dev->otherend);
	
	err = xenbus_gather(XBT_NULL, dev->otherend, "ring-ref", "%lu", &ring_ref, "event-channel", "%u", &evtchn,  NULL);

	if (err) {
		xenbus_dev_fatal(dev, err, "reading %s/ring-ref", dev->otherend);
		return err;
	}

	/* Map the shared frame, irq etc. */
	if(be->fbsetif != NULL)
	{
		err = fbsetif_map(be->fbsetif, ring_ref, evtchn);
		if (err) {
			xenbus_dev_fatal(dev, err, "mapping ring-ref %lu",ring_ref);
			return err;
			}			
	}

	err = xenbus_switch_state(dev, XenbusStateConnected);
	if (err)
		xenbus_dev_fatal(dev, err, "switching to Connected state",
				 dev->nodename);

	return 0;
}

static struct xenbus_device_id fbsetback_ids[] = {
	{ "vfb" },
	{ "" }
};

static struct xenbus_driver fbsetback = {
	.name = "vfb",
	.owner = THIS_MODULE,
	.ids = fbsetback_ids,
	.probe = fbsetback_probe,
	.remove = fbsetback_remove,
	.otherend_changed = frontend_changed
};


void fbset_xenbus_init(void)
{
	xenbus_register_backend(&fbsetback);
}



