/*
 * /drivers/xen/fbsetfront/fbsetfront-imx21.c
 *
 * Copyright (C) 2008 Samsung Electronics
 *          Dong-Hyuk lee  <dh5050.lee@samsung.com>
 *
 * Secure Xen on ARM architecture designed by Sang-bum Suh consists of
 * Xen on ARM and the associated access control.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public version 2 of License as published by
 * the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <linux/input.h>
#include <xen/xenbus.h>
#include<xen/gnttab.h>
#include <xen/evtchn.h>
#include "common.h"

#define FBSETIF_STATE_DISCONNECTED 0
#define FBSETIF_STATE_CONNECTED    1
#define FBSETIF_STATE_SUSPENDED    2

#define GRANT_INVALID_REF	0
#define KPP_DBG	1

#ifdef KPP_DBG
#define DPRINTK(fmt, args...) printk(KERN_DEBUG "%s: " fmt,__FUNCTION__,## args)
#else
#define DPRINTK(fmt, args...)
#endif


extern u32 my_ssa;
extern u32 my_ssa_gw1;

struct fbsetfront_info
{
	struct xenbus_device *xbdev;
	unsigned int handle;
	int connected;//a flag to indicate if connect to baceknd or not
	int ring_ref;
	fbsetif_ring_t *fbset_ring;
	unsigned int evtchn, irq;
};

static struct xenbus_device_id fbsetfront_ids[] = {
	{ "vfb" },
	{ "" }
};


static void connect(struct fbsetfront_info *info)
{

	if ((info->connected == FBSETIF_STATE_CONNECTED) ||
	    (info->connected == FBSETIF_STATE_SUSPENDED) )
		return;

	DPRINTK("fbsetfront.c:connect:%s.\n", info->xbdev->otherend);
	xenbus_switch_state(info->xbdev, XenbusStateConnected);
	info->connected = FBSETIF_STATE_CONNECTED;
}

static irqreturn_t fbsetif_int(int irq, void *dev_id)
{
	return IRQ_HANDLED;
}

static void fbsetif_recover(struct fbsetfront_info* info)
{
	printk (" not implemented yet!\n");
	return ;
}


static void fbsetif_free(struct fbsetfront_info *info, int suspend)
{
	/* Prevent new requests being issued until we fix things up. */

	info->connected = suspend ? FBSETIF_STATE_SUSPENDED : FBSETIF_STATE_DISCONNECTED;

	/* Free resources associated with old device channel. */
	if (info->ring_ref != GRANT_INVALID_REF) {
		gnttab_end_foreign_access(info->ring_ref, 0,
					  (unsigned long)info->fbset_ring);
		info->ring_ref = GRANT_INVALID_REF;
		info->fbset_ring = NULL;
	}
	
	if (info->irq)
		unbind_from_irqhandler(info->irq, info);
	info->evtchn = info->irq = 0;
			
}

fbsetif_ring_t * frontfb = NULL;

static int setup_fbsetring(struct xenbus_device *dev,
			 struct fbsetfront_info *info)
{
	fbsetif_ring_t *ring;
	int err;
	int i;

	info->ring_ref = GRANT_INVALID_REF;
	ring = (fbsetif_ring_t *)__get_free_page(GFP_KERNEL);
	if (!ring) {
		xenbus_dev_fatal(dev, -ENOMEM, "allocating shared ring");
		return -ENOMEM;
	}
	
	ring->fb_ssa = my_ssa;
	ring->fb_ssa_gw1 = my_ssa_gw1;

	frontfb = ring;

	
	info->fbset_ring = ring;

	err = xenbus_grant_ring(dev, virt_to_mfn(info->fbset_ring));
	if (err < 0) {
		free_page((unsigned long)ring);
		info->fbset_ring = NULL;
		goto fail;
	}
	info->ring_ref = err;
	
	err = xenbus_alloc_evtchn(dev, &info->evtchn);
	if (err)
		goto fail;
	
	err = bind_evtchn_to_irqhandler(
			info->evtchn, fbsetif_int, SA_SAMPLE_RANDOM, "fbsetif", info);
	if (err <= 0) {
		xenbus_dev_fatal(dev, err,
				"bind_evtchn_to_irqhandler failed");
		goto fail;
	}
	info->irq = err;


	
	return 0;
fail:
	fbsetif_free(info, 0);
	return err;
}

/* Common code used when first setting up, and when resuming. */
static int talk_to_backend(struct xenbus_device *dev,
			   struct fbsetfront_info *info)
{
	const char *message = NULL;
	xenbus_transaction_t xbt;
	int err;
	/* Create shared ring, alloc event channel. */
	err = setup_fbsetring(dev, info);
	if (err)
		goto out;

again:
	err = xenbus_transaction_start(&xbt);
	if (err) {
		xenbus_dev_fatal(dev, err, "starting transaction");
		goto destroy_ring;
	}

	err = xenbus_printf(xbt, dev->nodename,
			    "ring-ref","%u", info->ring_ref);
	if (err) {
		message = "writing ring-ref";
		goto abort_transaction;
	}
	
	err = xenbus_printf(xbt, dev->nodename,
			"event-channel", "%u", info->evtchn);
	if (err) {
		message = "writing event-channel";
		goto abort_transaction;
	}
	
	err = xenbus_transaction_end(xbt, 0);
	if (err) {
		if (err == -EAGAIN)
			goto again;
		xenbus_dev_fatal(dev, err, "completing transaction");
		goto destroy_ring;
	}

	xenbus_switch_state(dev, XenbusStateInitialised);

	return 0;

 abort_transaction:
	xenbus_transaction_end(xbt, 1);
	if (message)
		xenbus_dev_fatal(dev, err, "%s", message);
 destroy_ring:
	fbsetif_free(info, 0);
 out:
	return err;
}

static int fbsetfront_probe(struct xenbus_device *dev,
			  const struct xenbus_device_id *id)
{
	int err, vdevice, i,j,k;
	struct fbsetfront_info *info;
	/* FIXME: Use dynamic device id if this is not set. */
	
	info = kzalloc(sizeof(*info), GFP_KERNEL);
	if (!info) {
		xenbus_dev_fatal(dev, -ENOMEM, "allocating info structure");
		return -ENOMEM;
	}

	info->xbdev = dev;
	info->connected = FBSETIF_STATE_DISCONNECTED;  

	/* Front end dir is a number, which is used as the id. */
	info->handle = simple_strtoul(strrchr(dev->nodename,'/')+1, NULL, 0);
	
	dev->data = info;
	info->xbdev = dev;

	return 0;
}

static int fbsetfront_remove(struct xenbus_device *dev)
{
	struct fbsetfront_info *info = dev->data;

	DPRINTK("blkfront_remove: %s removed\n", dev->nodename);

	fbsetif_free(info, 0);

	kfree(info);

	return 0;
}


static int fbsetfront_resume(struct xenbus_device *dev)
{
	struct fbsetfront_info *info = dev->data;
	int err;

	DPRINTK("fbsetfront_resume: %s\n", dev->nodename);

	fbsetif_free(info, 1);

	err = talk_to_backend(dev, info);
	if (!err)
		fbsetif_recover(info);

	return err;
}

static void fbsetfront_closing(struct xenbus_device *dev)
{
	/* to do implement some device unregister code */

	DPRINTK("fbsetfront_closing: %s removed\n", dev->nodename);

	xenbus_switch_state(dev, XenbusStateClosed);
}

static void backend_changed(struct xenbus_device *dev,
			    XenbusState backend_state)
{
	struct fbsetfront_info *info = dev->data;
	int err;

	DPRINTK("fbsetfront:backend_changed.\n");

	switch (backend_state) {
	case XenbusStateUnknown:
		break;
		
	case XenbusStateInitialising:
		break;
	case XenbusStateInitWait:
		err = talk_to_backend(dev, info);
		if(err)  printk("=== 		talk_to_backend() error!!\n");
		break;
	case XenbusStateInitialised:
		 break;

	case XenbusStateConnected:
		 connect(info);
		 break;

		 
	case XenbusStateClosed:
	case XenbusStateClosing:
		 fbsetfront_closing(dev);
		 break;
	}
}

static struct xenbus_driver fbsetfront = {
	.name = "vfb",
	.owner = THIS_MODULE,
	.ids = fbsetfront_ids,
	.probe = fbsetfront_probe,
	.remove = fbsetfront_remove,
	.resume = fbsetfront_resume,
	.otherend_changed = backend_changed,
};

static int __init vfbset_init(void)
{
	xenbus_register_frontend(&fbsetfront);
	return 0;
}

module_init(vfbset_init);


static void vfbset_exit(void)
{
	return xenbus_unregister_driver(&fbsetfront);
}
module_exit(vfbset_exit);

MODULE_LICENSE("Dual BSD/GPL");




