/*
 * /drivers/xen/kppback/interface.c
 *
 * Copyright (C) 2008 Samsung Electronics
 *          Cheol-Ryeon kim  <cr2104.kim@samsung.com>
 *          Dong-Hyuk lee  <dh5050.lee@samsung.com>
 *
 * Secure Xen on ARM architecture designed by Sang-bum Suh consists of
 * Xen on ARM and the associated access control.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public version 2 of License as published by
 * the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
 
#include <xen/evtchn.h>
#include <linux/err.h>
#include <xen/driver_util.h>
#include <xen/interface/event_channel.h>
#include "common.h"

LIST_HEAD(frontkpps);

extern unsigned int foreground_dom;

void  dom_destroy_cb(int domid)
{
	printk(" %d Domain keypad is disconnected!\n",domid);
	kppif_t* i;
	list_for_each_entry(i,&frontkpps,list)
	{
		if(i->domid == domid)
		{
			list_del(&i->list);
			kfree(i);
			return;
		}	
	}
	return;
}

int key_send(unsigned char x, int value)
{
	kppif_t* i;
	kppif_ring_t* shared_ring;
	
	int exist = 0;
	unsigned long wptr;
	list_for_each_entry(i,&frontkpps,list)
	{
		if(i->domid != foreground_dom)
			continue;
		
		if(i->status != CONNECTED)
		{
			printk("foreground domain kpp interface is not connected!\n");
			return -1;
		}
		exist = 1;
		shared_ring = (kppif_ring_t*)i->kpp_ring_area->addr;
		wptr =shared_ring->wptr;
		shared_ring->sring[wptr].key = x;
		shared_ring->sring[wptr].value = value;
		shared_ring->counts++;
		shared_ring->wptr++;
		shared_ring->wptr %= MAX_RING_SIZE;
		notify_remote_via_evtchn(i->evtchn);
	}
	
	if(exist != 1 )
	{
		printk("fail to find  foregound domain kpp interface!\n");
		return -1;
	}
	
	return 0;
}
EXPORT_SYMBOL(key_send);

kppif_t *alloc_kppif(domid_t domid)
{
	kppif_t *kppif = kzalloc(sizeof(kppif_t),GFP_KERNEL);
	
	if (!kppif)
		return ERR_PTR(-ENOMEM);

	memset(kppif, 0, sizeof(kppif_t));
	kppif->domid = domid;
	kppif->status = DISCONNECTED;
	INIT_LIST_HEAD(&kppif->list);
	list_add_tail(&kppif->list, &frontkpps);
	
	return kppif;
}


static int map_frontend_page(kppif_t *kppif, unsigned long shared_page)
{
	struct gnttab_map_grant_ref op;
	int ret;

	op.host_addr = (unsigned long)kppif->kpp_ring_area->addr;
	op.flags     = GNTMAP_host_map;
	op.ref       = shared_page;
	op.dom       = kppif->domid;
   /* the value of use should be same with the value of frontend's devicetype registerd in the xenstore */
	op.use       = 'v'^'k'^'p'^'p';

	lock_vm_area(kppif->kpp_ring_area);
	ret = HYPERVISOR_grant_table_op(GNTTABOP_map_grant_ref, &op, 1);
	unlock_vm_area(kppif->kpp_ring_area);
	BUG_ON(ret);

	if (op.status) {
		printk(" Grant table operation failure !\n");
		return op.status;
	}

	kppif->shmem_ref = shared_page;
	kppif->shmem_handle = op.handle;

	return 0;
}


void unmap_frontend_page_kpp(kppif_t *kppif)
{
	struct gnttab_unmap_grant_ref op;
	int ret;

	op.host_addr    = (unsigned long)kppif->kpp_ring_area->addr;
	op.handle       = kppif->shmem_handle;
	op.dev_bus_addr = 0;

	lock_vm_area(kppif->kpp_ring_area);
	ret = HYPERVISOR_grant_table_op(GNTTABOP_unmap_grant_ref, &op, 1);
	if(op.status) {
		printk("\nkppback: unmap_frontend_page failed\n");
	}
	unlock_vm_area(kppif->kpp_ring_area);
	BUG_ON(ret);
}

int kppif_map(kppif_t *kppif, unsigned long shared_page, unsigned int evtchn)
{
	int err;
	evtchn_op_t op = {
		.cmd = EVTCHNOP_bind_interdomain,
		.u.bind_interdomain.remote_dom = kppif->domid,
		.u.bind_interdomain.remote_port = evtchn, 
   	/* the value of use should be same with the value of frontend's devicetype registerd in the xenstore */
		.u.bind_interdomain.use = 'v'^'k'^'p'^'p'};

	/* Already connected through? */
	if (kppif->irq)
		return 0;

	if ( (kppif->kpp_ring_area = alloc_vm_area(PAGE_SIZE)) == NULL )
		return -ENOMEM;

	err = map_frontend_page(kppif, shared_page);
	if (err) {
		free_vm_area(kppif->kpp_ring_area);
		return err;
	}

	err = HYPERVISOR_event_channel_op(&op);
	if (err) {
		unmap_frontend_page_kpp(kppif);
		free_vm_area(kppif->kpp_ring_area);
		return err;
	}

	kppif->evtchn = op.u.bind_interdomain.local_port;
	kppif->irq = bind_evtchn_to_irqhandler(kppif->evtchn, kppif_be_int, 0, "kppif-backend", kppif);

	return 0;
}
