/*
 * /drivers/xen/kppback/xenbus.c
 *
 * Copyright (C) 2008 Samsung Electronics
 *          Cheol-Ryeon kim  <cr2104.kim@samsung.com>
 *          Dong-Hyuk lee  <dh5050.lee@samsung.com>
 *
 * Secure Xen on ARM architecture designed by Sang-bum Suh consists of
 * Xen on ARM and the associated access control.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public version 2 of License as published by
 * the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
 
#include <xen/xenbus.h>
#include "common.h"

extern int dom;  

#undef DPRINTK
#define DPRINTK(fmt, args...) \
    pr_debug("kppback/xenbus (%s:%d) " fmt ".\n", __FUNCTION__, __LINE__, ##args)

struct backend_info
{
	struct xenbus_device *dev;
	kppif_t *kppif;
	XenbusState frontend_state;
};

static int connect(struct backend_info *);

static int kppback_remove(struct xenbus_device *dev)
{
	DPRINTK("");
	
	struct backend_info *be = dev->data;
	if (be->kppif) {
		if(be->kppif->status == CONNECTED){
			unmap_frontend_page_kpp(be->kppif);
			free_vm_area(be->kppif->kpp_ring_area);
		be->kppif->status = DISCONNECTED; 
			
		}
		if (be->kppif->irq) unbind_from_irqhandler(be->kppif->irq, be->kppif);
		dom_destroy_cb(be->kppif->domid);
		be->kppif = NULL;
	}
	kfree(be);
	dev->data = NULL;
	return 0;
}

static int kppback_probe(struct xenbus_device *dev,
		const struct xenbus_device_id *id)
{
	int err;

	struct backend_info *be = kzalloc(sizeof(struct backend_info),GFP_KERNEL);
	if (!be) {
		xenbus_dev_fatal(dev, -ENOMEM,
				 "allocating backend structure");
		return -ENOMEM;
	}

	be->dev = dev;
	dev->data = be;
	be->kppif = alloc_kppif(dev->otherend_id);

	err = xenbus_switch_state(dev, XenbusStateInitWait);
	
	if (err) {
		goto fail;
	}

	return 0;

fail:
	DPRINTK("failed");
	kppback_remove(dev);
	return err;
}

static void frontend_changed(struct xenbus_device *dev,
			     XenbusState frontend_state)
{
	struct backend_info *be = dev->data;
    int err;

	be->frontend_state = frontend_state;

	switch (frontend_state) {
	case XenbusStateInitialising:
		 break;
	case XenbusStateInitialised:
	case XenbusStateConnected:
		if (dev->state == XenbusStateConnected){
			break;
		}
		err = connect(be);

		if(err)
			break;

		be->kppif->status = CONNECTED;
		break;

	case XenbusStateClosing:
		xenbus_switch_state(dev, XenbusStateClosing);
		break;

	case XenbusStateClosed:
		device_unregister(&dev->dev);
		break;

	case XenbusStateUnknown:
	case XenbusStateInitWait:
	default:
		xenbus_dev_fatal(dev, -EINVAL, "saw state %d at frontend",
				 frontend_state);
		break;
	}
}

static int connect(struct backend_info *be)
{
	struct xenbus_device *dev = be->dev;
	unsigned long ring_ref;
	unsigned int evtchn;
	int err;

	DPRINTK("%s", dev->otherend);

	err = xenbus_gather(XBT_NULL, dev->otherend, "ring-ref", "%lu", &ring_ref,
			    "event-channel", "%u", &evtchn, NULL);

	if (err) {
		xenbus_dev_fatal(dev, err, "reading %s/ring-ref and event-channel", dev->otherend);
		return err;
	}

	/* Map the shared frame, irq etc. */
	if(be->kppif != NULL)
	{
		err = kppif_map(be->kppif, ring_ref, evtchn);
		if (err) {
			xenbus_dev_fatal(dev, err, "mapping ring-ref %lu port %u",
					 ring_ref, evtchn);
			return err;
			}
			
	}

	err = xenbus_switch_state(dev, XenbusStateConnected);
	if (err)
		xenbus_dev_fatal(dev, err, "switching to Connected state",
				 dev->nodename);

	return 0;
}

static struct xenbus_device_id kppback_ids[] = {
	{ "vkpp" },
	{ "" }
};

static struct xenbus_driver kppback = {
	.name = "vkpp",
	.owner = THIS_MODULE,
	.ids = kppback_ids,
	.probe = kppback_probe,
	.remove = kppback_remove,
	.otherend_changed = frontend_changed
};

void kpp_xenbus_init(void)
{
	xenbus_register_backend(&kppback);
}
