/*
 * /drivers/xen/kppfront/kppfront.c
 *
 * Copyright (C) 2008 Samsung Electronics
 *          Cheol-Ryeon kim  <cr2104.kim@samsung.com>
 *          Dong-Hyuk lee  <dh5050.lee@samsung.com>
 *
 * Secure Xen on ARM architecture designed by Sang-bum Suh consists of
 * Xen on ARM and the associated access control.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public version 2 of License as published by
 * the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
 
#include <linux/input.h>
#include <xen/xenbus.h>
#include <xen/gnttab.h>
#include <xen/evtchn.h>
#include "common.h"
#include "kppfront.h"

extern int dom;

static struct input_dev *vkpp_dev;

#define KPPIF_STATE_DISCONNECTED 0
#define KPPIF_STATE_CONNECTED    1
#define KPPIF_STATE_SUSPENDED    2

#define GRANT_INVALID_REF	0
#define KPP_DBG	1

#ifdef KPP_DBG
#define DPRINTK(fmt, args...) printk(KERN_DEBUG "%s: " fmt,__FUNCTION__,## args)
#else
#define DPRINTK(fmt, args...)
#endif

struct kppfront_info
{
	struct input_dev* vkpp_dev;
	struct xenbus_device *xbdev;
	int vdevice;
	unsigned int handle;
	int connected;//a flag to indicate if connect to baceknd or not
	int ring_ref;
	kppif_ring_t *kpp_ring;
	unsigned int evtchn, irq;
};

static struct xenbus_device_id kppfront_ids[] = {
	{ "vkpp" },
	{ "" }
};

static void connect(struct kppfront_info *info)
{

	if ((info->connected == KPPIF_STATE_CONNECTED) ||
	    (info->connected == KPPIF_STATE_SUSPENDED) )
		return;

	DPRINTK("kppfront.c:connect:%s.\n", info->xbdev->otherend);
	xenbus_switch_state(info->xbdev, XenbusStateConnected);
	info->connected = KPPIF_STATE_CONNECTED;
}

static irqreturn_t kppif_int(int irq, void *dev_id)
{
	unsigned char key;
	int value;
	unsigned long rptr;
	struct kppfront_info *info = (struct kppfront_info *)dev_id;
    kppif_ring_t* ring;
 
	//printk("* keypad event handler of domain1,<%d>\n",shared_ring_front->counts);
    if( info->kpp_ring == NULL)
    {
		printk(" ***** FIXME   shared_ring_front is not initialized, yet.\n");
		return -1;
	}

    if(info->connected != KPPIF_STATE_CONNECTED )
    {
		printk("#foreground domain can't be changed.\n");
		return -1;
    }
	
     ring = info->kpp_ring;

	if(ring->counts <= 0)
		printk("* Access is tried, but there is no meseage!\n");
	
	while(ring->counts){
		rptr = ring->rptr;
		key  = ring->sring[rptr].key;
		value = ring->sring[rptr].value;

		printk("here is dom1 : *[%d] key = %d\n",value,key);
   
		input_report_key(vkpp_dev, key, value);

		ring->rptr++;
		ring->rptr %= MAX_RING_SIZE;
		ring->counts--;
	}
	
	return IRQ_HANDLED;
}


static void kppif_recover(struct kppfront_info* info)
{
	printk (" not implemented yet!\n");
	return ;
}

static void kppif_free(struct kppfront_info *info, int suspend)
{
	/* Prevent new requests being issued until we fix things up. */

	info->connected = suspend ? KPPIF_STATE_SUSPENDED : KPPIF_STATE_DISCONNECTED;

	/* Free resources associated with old device channel. */
	if (info->ring_ref != GRANT_INVALID_REF) {
		gnttab_end_foreign_access(info->ring_ref, 0,
					  (unsigned long)info->kpp_ring);
		info->ring_ref = GRANT_INVALID_REF;
		info->kpp_ring = NULL;
	}
	if (info->irq)
		unbind_from_irqhandler(info->irq, info);
	info->evtchn = info->irq = 0;

}

static int setup_kppring(struct xenbus_device *dev,
			 struct kppfront_info *info)
{
	kppif_ring_t *ring;
	int err;
	int i;

	info->ring_ref = GRANT_INVALID_REF;
	ring = (kppif_ring_t *)__get_free_page(GFP_KERNEL);
	if (!ring) {
		xenbus_dev_fatal(dev, -ENOMEM, "allocating shared ring");
		return -ENOMEM;
	}
	ring->rptr = 0;
	ring->wptr = 0;
	ring->counts = 0;
       for(i= 0; i < MAX_RING_SIZE ; i++)
	{
		ring->sring[i].key = 0;
		ring->sring[i].value=0;
	}
	
	info->kpp_ring = ring;

	err = xenbus_grant_ring(dev, virt_to_mfn(info->kpp_ring));
	if (err < 0) {
		free_page((unsigned long)ring);
		info->kpp_ring = NULL;
		goto fail;
	}
	info->ring_ref = err;

	err = xenbus_alloc_evtchn(dev, &info->evtchn);
	if (err)
		goto fail;

	err = bind_evtchn_to_irqhandler(
		info->evtchn, kppif_int, SA_SAMPLE_RANDOM, "kppif", info);
	if (err <= 0) {
		xenbus_dev_fatal(dev, err,
				 "bind_evtchn_to_irqhandler failed");
		goto fail;
	}
	info->irq = err;

	return 0;
fail:
	kppif_free(info, 0);
	return err;
}

/* Common code used when first setting up, and when resuming. */
static int talk_to_backend(struct xenbus_device *dev,
			   struct kppfront_info *info)
{
	const char *message = NULL;
	xenbus_transaction_t xbt;
	int err;
	/* Create shared ring, alloc event channel. */
	err = setup_kppring(dev, info);
	if (err)
		goto out;

again:
	err = xenbus_transaction_start(&xbt);
	if (err) {
		xenbus_dev_fatal(dev, err, "starting transaction");
		goto destroy_ring;
	}

	err = xenbus_printf(xbt, dev->nodename,
			    "ring-ref","%u", info->ring_ref);
	if (err) {
		message = "writing ring-ref";
		goto abort_transaction;
	}
	err = xenbus_printf(xbt, dev->nodename,
			    "event-channel", "%u", info->evtchn);
	if (err) {
		message = "writing event-channel";
		goto abort_transaction;
	}

	err = xenbus_transaction_end(xbt, 0);
	if (err) {
		if (err == -EAGAIN)
			goto again;
		xenbus_dev_fatal(dev, err, "completing transaction");
		goto destroy_ring;
	}

	err = xenbus_switch_state(dev, XenbusStateInitialised);

	if(err) {
		if (err == -EAGAIN)
			goto again;
		xenbus_dev_fatal(dev, err, "switching state");
		goto destroy_ring;
	}
	
	return 0;

 abort_transaction:
	xenbus_transaction_end(xbt, 1);
	if (message)
		xenbus_dev_fatal(dev, err, "%s", message);
 destroy_ring:
	kppif_free(info, 0);
 out:
	return err;
}

static struct platform_device*  vkpp_device;

static int kppfront_probe(struct xenbus_device *dev,
			  const struct xenbus_device_id *id)
{
	int err, vdevice, i,j,k;
	struct kppfront_info *info; 
	
	/* FIXME: Use dynamic device id if this is not set. */

	info = kzalloc(sizeof(*info), GFP_KERNEL);
	if (!info) {
		xenbus_dev_fatal(dev, -ENOMEM, "allocating info structure");
		return -ENOMEM;
	}

	info->xbdev = dev;
	info->vdevice = vdevice;
	info->connected = KPPIF_STATE_DISCONNECTED;

	/* Front end dir is a number, which is used as the id. */
	info->handle = simple_strtoul(strrchr(dev->nodename,'/')+1, NULL, 0);
	
	//virtual kpp register
	if ((vkpp_dev = input_allocate_device()) == NULL) {
		return -ENOMEM;
	}

	vkpp_dev->name = "kpp";
	vkpp_dev->phys = "kpp/input0";
	vkpp_dev->id.vendor = 0x0001;
	vkpp_dev->id.product = 0x0001;
	vkpp_dev->id.version = 0x0001;

	set_bit(EV_KEY, vkpp_dev->evbit);
	
	vkpp_dev->keycode = kpp_keycode;
	vkpp_dev->keycodesize = sizeof(unsigned char);
	vkpp_dev->keycodemax = ARRAY_SIZE(kpp_keycode);
	
    for (i = 0; i < 6; i++)
        for(j=0; j < 6; j++)
            for(k=0; k < 3; k++)
            if (kpp_keycode[i][j][k])
                set_bit(kpp_keycode[i][j][k], vkpp_dev->keybit);

	info->vkpp_dev = &vkpp_dev;
	input_register_device(vkpp_dev);
	
	dev->data = info;
	info->xbdev = dev;
	
	return 0;
}

static int kppfront_remove(struct xenbus_device *dev)
{
	struct kppfront_info *info = dev->data;

	input_unregister_device(vkpp_dev);
	
	DPRINTK("blkfront_remove: %s removed\n", dev->nodename);

	kppif_free(info, 0);

	kfree(info);

	return 0;
}

static int kppfront_resume(struct xenbus_device *dev)
{
	struct kppfront_info *info = dev->data;
	int err;

	DPRINTK("kppfront_resume: %s\n", dev->nodename);

	kppif_free(info, 1);

	err = talk_to_backend(dev, info);
	if (!err)
		kppif_recover(info);

	return err;
}

static void kppfront_closing(struct xenbus_device *dev)
{
	/* to do implement some device unregister code */
	DPRINTK("kppfront_closing: %s removed\n", dev->nodename);

	xenbus_switch_state(dev, XenbusStateClosed);
}

static void backend_changed(struct xenbus_device *dev,
			    XenbusState backend_state)
{
	struct kppfront_info *info = dev->data;
	int err;

	DPRINTK("kppfront:backend_changed.\n");

	switch (backend_state) {
	case XenbusStateUnknown:
		break;
		
	case XenbusStateInitialising:
		break;
	case XenbusStateInitWait:
		err = talk_to_backend(dev, info);
		if(err)  printk("== talk_to_backend() error!!\n");
		break;
	case XenbusStateInitialised:
		 break;

	case XenbusStateConnected:
		 connect(info);
		 break;
		 
       case XenbusStateClosed:
	case XenbusStateClosing:
			kppfront_closing(dev);
		break;
	}
}

static struct xenbus_driver kppfront = {
	.name = "vkpp",
	.owner = THIS_MODULE,
	.ids = kppfront_ids,
	.probe = kppfront_probe,
	.remove = kppfront_remove,
	.resume = kppfront_resume,
	.otherend_changed = backend_changed,
};

static int __init vkpp_init(void)
{
    printk("Initialising virtual keypad driver\n");
	xenbus_register_frontend(&kppfront);
	return 0;
}

module_init(vkpp_init);

static void vkpp_exit(void)
{
	return xenbus_unregister_driver(&kppfront);
}

module_exit(vkpp_exit);

MODULE_LICENSE("Dual BSD/GPL");
