/*
 *  linux/include/asm-arm/page.h
 *
 *  Copyright (C) 1995-2003 Russell King
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */
#ifndef _ASMARM_PAGE_H
#define _ASMARM_PAGE_H


#ifdef __KERNEL__

/* PAGE_SHIFT determines the page size */
#define PAGE_SHIFT		12
#define PAGE_SIZE		(1UL << PAGE_SHIFT)
#define PAGE_MASK		(~(PAGE_SIZE-1))

/* to align the pointer to the (next) page boundary */
#define PAGE_ALIGN(addr)	(((addr)+PAGE_SIZE-1)&PAGE_MASK)

#ifndef __ASSEMBLY__

#ifndef CONFIG_MMU

#include "page-nommu.h"

#else

#include <asm/glue.h>

/*
 *	User Space Model
 *	================
 *
 *	This section selects the correct set of functions for dealing with
 *	page-based copying and clearing for user space for the particular
 *	processor(s) we're building for.
 *
 *	We have the following to choose from:
 *	  v3		- ARMv3
 *	  v4wt		- ARMv4 with writethrough cache, without minicache
 *	  v4wb		- ARMv4 with writeback cache, without minicache
 *	  v4_mc		- ARMv4 with minicache
 *	  xscale	- Xscale
 *	  xsc3		- XScalev3
 */
#undef _USER
#undef MULTI_USER

#ifdef CONFIG_CPU_COPY_V3
# ifdef _USER
#  define MULTI_USER 1
# else
#  define _USER v3
# endif
#endif

#ifdef CONFIG_CPU_COPY_V4WT
# ifdef _USER
#  define MULTI_USER 1
# else
#  define _USER v4wt
# endif
#endif

#ifdef CONFIG_CPU_COPY_V4WB
# ifdef _USER
#  define MULTI_USER 1
# else
#  define _USER v4wb
# endif
#endif

#ifdef CONFIG_CPU_SA1100
# ifdef _USER
#  define MULTI_USER 1
# else
#  define _USER v4_mc
# endif
#endif

#ifdef CONFIG_CPU_XSCALE
# ifdef _USER
#  define MULTI_USER 1
# else
#  define _USER xscale_mc
#  endif
# endif

#ifdef CONFIG_CPU_XSC3
# ifdef _USER
#  define MULTI_USER 1
# else
#  define _USER xsc3_mc
# endif
#endif

#ifdef CONFIG_CPU_COPY_V6
# define MULTI_USER 1
#endif

#if !defined(_USER) && !defined(MULTI_USER)
#error Unknown user operations model
#endif

struct cpu_user_fns {
	void (*cpu_clear_user_page)(void *p, unsigned long user);
	void (*cpu_copy_user_page)(void *to, const void *from,
				   unsigned long user);
};

#ifdef MULTI_USER
extern struct cpu_user_fns cpu_user;

#define __cpu_clear_user_page	cpu_user.cpu_clear_user_page
#define __cpu_copy_user_page	cpu_user.cpu_copy_user_page

#else

#ifndef CONFIG_XENOLINUX
#define __cpu_clear_user_page	__glue(_USER,_clear_user_page)
#define __cpu_copy_user_page	__glue(_USER,_copy_user_page)
#else
#define __cpu_clear_user_page	xen_clear_user_page
#define __cpu_copy_user_page	xen_copy_user_page
#endif /* CONFIG_XENOLINUX */


extern void __cpu_clear_user_page(void *p, unsigned long user);
extern void __cpu_copy_user_page(void *to, const void *from,
				 unsigned long user);
#endif

#define clear_user_page(addr,vaddr,pg)	 __cpu_clear_user_page(addr, vaddr)
#define copy_user_page(to,from,vaddr,pg) __cpu_copy_user_page(to, from, vaddr)

#define clear_page(page)	memzero((void *)(page), PAGE_SIZE)
extern void copy_page(void *to, const void *from);

#undef STRICT_MM_TYPECHECKS

#ifdef STRICT_MM_TYPECHECKS
/*
 * These are used to make use of C type-checking..
 */
typedef struct { unsigned long pte; } pte_t;
typedef struct { unsigned long pmd; } pmd_t;
typedef struct { unsigned long pgd[2]; } pgd_t;
typedef struct { unsigned long pgprot; } pgprot_t;

#define pte_val(x)      ((x).pte)
#define pmd_val(x)      ((x).pmd)
#define pgd_val(x)	((x).pgd[0])
#define pgprot_val(x)   ((x).pgprot)

#define __pte(x)        ((pte_t) { (x) } )
#define __pmd(x)        ((pmd_t) { (x) } )
#define __pgprot(x)     ((pgprot_t) { (x) } )

#else
/*
 * .. while these make it easier on the compiler
 */
typedef unsigned long pte_t;
typedef unsigned long pmd_t;
typedef unsigned long pgd_t[2];
typedef unsigned long pgprot_t;

#define pte_val(x)      (x)
#define pmd_val(x)      (x)
#define pgd_val(x)	((x)[0])
#define pgprot_val(x)   (x)

#define __pte(x)        (x)
#define __pmd(x)        (x)
#define __pgprot(x)     (x)

#endif /* STRICT_MM_TYPECHECKS */

#ifdef CONFIG_XENOLINUX

#include <linux/types.h>
#include <xen/interface/xen.h>

typedef unsigned long maddr_t;  //added for xen

#define L1_PAGETABLE_SHIFT      12
#define L2_PAGETABLE_SHIFT	20

#define L1_PAGETABLE_ENTRIES	256
#define L2_PAGETABLE_ENTRIES	4096

#define	PADDR_BITS		32
#define	PADDR_MASK		(~0UL)

#if defined(CONFIG_X86_64)
#define MULTI_UVMFLAGS_INDEX 2
#define MULTI_UVMDOMID_INDEX 3
#else
#define MULTI_UVMFLAGS_INDEX 3
#define MULTI_UVMDOMID_INDEX 4
#endif

#define xen_machphys_update(_x,_y)	do {} while (0)

#define INVALID_P2M_ENTRY	(~0UL)
#define FOREIGN_FRAME_BIT	(1UL<<31)
#define FOREIGN_FRAME(m)	((m) | FOREIGN_FRAME_BIT)

extern unsigned long *phys_to_machine_mapping;

extern unsigned long xen_min_mfn;

extern unsigned long long __supported_pte_mask;

#define __pte_ma(x)	((pte_t) { (x) } )

#include <xen/features.h>

static inline int phys_to_machine_mapping_valid(unsigned long pfn)
{
	if (xen_feature(XENFEAT_auto_translated_physmap))
		return 1;
	return (phys_to_machine_mapping[pfn] != INVALID_P2M_ENTRY);
}

#define pfn_to_mfn(_pfn) (phys_to_machine_mapping[(_pfn)])
static inline unsigned long phys_to_machine(unsigned long phys)
{
	unsigned long machine = pfn_to_mfn(phys >> L1_PAGETABLE_SHIFT);
	machine = (machine << L1_PAGETABLE_SHIFT) | (phys & ~PAGE_MASK);
	return machine;
}

#define mfn_to_pfn(_mfn) (machine_to_phys_mapping[(_mfn)-xen_min_mfn])
static inline unsigned long machine_to_phys(unsigned long machine)
{
	unsigned long phys = mfn_to_pfn(machine >> L1_PAGETABLE_SHIFT);
	phys = (phys << L1_PAGETABLE_SHIFT) | (machine & ~PAGE_MASK);
	return phys;
}

/* for pseudo-physical code compatibility */

#define virt_to_mfn(v)				(__pa(v) >> PAGE_SHIFT)
#define FOREIGN_FRAME_BIT			(1UL<<31)

#ifndef FOREIGN_FRAME 
#define FOREIGN_FRAME(m)			((m)| FOREIGN_FRAME_BIT)
#endif
#define pfn_to_kaddr(pfn)			(__va((pfn) << PAGE_SHIFT))
#define set_phys_to_machine(_x, _y)	do {} while (0)
#define pfn_pte_ma(pfn, prot)		pfn_pte(pfn, prot)	   
#define pte_mfn(_x)					pte_pfn(_x)

#define to_phys(x)                        ((unsigned long)(x)-PAGE_OFFSET)
#define to_virt(x)                        ((void *)((unsigned long)(x)+PAGE_OFFSET))

#define virt_to_pfn(_virt)                (PFN_DOWN(to_phys(_virt)))
#define mach_to_virt(_mach)               (to_virt(machine_to_phys(_mach)))
#define mfn_to_virt(_mfn)                 (mach_to_virt(_mfn << PAGE_SHIFT))
#define pfn_to_virt(_pfn)                 (to_virt(_pfn << PAGE_SHIFT))

#define scrub_pages(_p,_n)			((void) 0)

#endif /* CONFIG_XENOLINUX */

#endif /* CONFIG_MMU */

#include <asm/memory.h>

#endif /* !__ASSEMBLY__ */

#define VM_DATA_DEFAULT_FLAGS	(VM_READ | VM_WRITE | VM_EXEC | \
				 VM_MAYREAD | VM_MAYWRITE | VM_MAYEXEC)

/*
 * With EABI on ARMv5 and above we must have 64-bit aligned slab pointers.
 */
#if defined(CONFIG_AEABI) && (__LINUX_ARM_ARCH__ >= 5)
#define ARCH_SLAB_MINALIGN 8
#endif

#include <asm-generic/page.h>

#endif /* __KERNEL__ */

#endif
